#' Fitting a NARFIMA Model
#'
#' This function fits a Neural AutoRegressive Fractionally Integrated Moving Average (NARFIMA) model to univariate time series. The model uses \code{p} Autoregressive terms, \code{q} moving average
#' terms, and a single hidden layer with \code{size} nodes. If \code{p}, \code{q}, or \code{size} are not specified, they are automatically
#' determined:
#' \itemize{
#'   \item \code{p} is set to the number of significant Autoregressive coefficients estimated from an AR model.
#'   \item \code{q} is set to the order of the moving average component from an ARFIMA model fitted to the residuals.
#'   \item \code{size} is set to the floor of half the sum of \code{p} and \code{q}: \deqn{size = \lfloor\frac{p + q}{2}\rfloor.}}
#' Exogenous variables can be included via \code{xreg}. If \code{er} is not
#' provided, it is computed from an ARFIMA model fitted to \code{y}. When \code{y} or \code{xreg} have missing values, the corresponding
#' rows and any dependent lagged rows are removed before fitting the model. Multiple neural network models are fitted, each initialized
#' with random weights, and the final model is obtained by averaging their outputs. Optionally, Box-Cox transformations can be applied
#' to stabilize variance, and inputs can be scaled to improve model convergence.
#'
#' @param y A numeric vector or time series of class \code{ts}.
#' @param er A numeric vector or time series object representing the series of residuals. If missing, it will be calculated from an ARFIMA model with exogenous variable (if provided).
#' @param xreg An optional numeric matrix of exogenous variables to be included in the model (default is \code{NULL}).
#' @param p Integer indicating the number of lags of the input series \code{y}.
#' @param q Integer indicating the number of lags of the errors \code{er}.
#' @param P Integer indicating the number of seasonal lags of the input series \code{y} (default is \code{1}).
#' @param size Integer specifying the number of nodes of the feed-forward neural networks with a single hidden layer.
#' @param skip Logical value indicating whether to use the direct connections in the neural network (default is \code{TRUE}).
#' @param repeats Integer specifying the number of times to fit the neural network model (default is \code{1000}).
#' @param lambda Numeric value for the Box-Cox transformation parameter of \code{y} (default is \code{0.5}).
#' @param lambdae Numeric value for the Box-Cox transformation parameter of \code{er} (default is \code{0.5}).
#' @param scale.inputs Logical value indicating whether to standardize the inputs before fitting the model (default is \code{TRUE}).
#' @param ... Additional arguments passed to \code{auto_narfima}.
#'
#' @return Returns an object of class "narfima", containing the following components:
#' \item{series}{The name of the input series.}
#' \item{method}{A string describing the model parameters.}
#' \item{model}{The fitted NARFIMA model.}
#' \item{fitted}{The fitted values from the model.}
#' \item{residuals}{The residuals from the model.}
#' \item{m}{The seasonal frequency of the input series.}
#' \item{p}{The number of Autoregressive terms used.}
#' \item{q}{The number of moving average terms used.}
#' \item{P}{The number of seasonal lags used.}
#' \item{size}{The number of nodes in the hidden layer used in the neural network.}
#' \item{skip}{Indicates if the direct connections were used in the neural network.}
#' \item{lambda}{The Box-Cox transformation parameter for the input series.}
#' \item{scaley}{Scaling parameters for the input series.}
#' \item{lags}{The lags used for the input series.}
#' \item{lambdae}{The Box-Cox transformation parameter for the error term.}
#' \item{scalee}{Scaling parameters for the error term.}
#' \item{lagse}{The lags used for the error term.}
#' \item{scalexreg}{Scaling parameters for the exogenous variables, if provided.}
#' \item{y}{The input time series.}
#' \item{e}{The transformed residual series}
#' \item{xreg}{The exogenous variables used in the model, if provided.}
#' \item{nnetargs}{Additional arguments passed to the neural network function.}
#'
#' @importFrom forecast arfima BoxCox InvBoxCox na.interp seasadj mstl
#' @importFrom stats ar as.ts complete.cases frequency fitted is.ts predict residuals ts tsp "tsp<-"
#'
#' @examples
#' h <- 3
#'
#  # Take the last 53 observations from EuStockMarkets (FTSE index): 50 for training and the final 3 for testing.
#' data <- EuStockMarkets[(nrow(EuStockMarkets) - 53):nrow(EuStockMarkets),4]
#'
#' train <- data[1:(length(data) - h)]
#' test <- data[(length(data) - h + 1):length(data)]
#'
#' narfima_model <- auto_narfima(train)
#'
#' @export

auto_narfima <- function(y, er, xreg = NULL, p, q, P = 1, size, skip, repeats = 1000, lambda = 0.5, lambdae = 0.5, scale.inputs = TRUE, ...) {

  yname <- deparse(substitute(y))


  if(missing(er)){
    arfima_model <- arfima(as.ts(y), xreg = if (!is.null(xreg)) as.ts(xreg) else NULL)
    arfima_er <-  residuals(arfima_model)
    arfima_er[is.na(arfima_er)] <-  0
    er <- arfima_er
    e <- er
  }

  e <- er
  x <- y

  # Check for NAs in y
  if (any(is.na(x))) {
    warning("Missing values in y, omitting rows")
  }


  # Transform data
  if (!is.null(lambda)) {
    xx <- BoxCox(x, lambda)
    lambda <- attr(xx, "lambda")
  } else {
    xx <- x
  }


  # Transform error
  if (!is.null(lambdae)) {
    ee <- BoxCox(e, lambdae)
    lambdae <- attr(ee, "lambdae")
  } else {
    ee <- e
  }


  # Scale series x and error
  scaley <- NULL
  if (scale.inputs) {

    tmpx <- scale(xx, center = TRUE, scale = TRUE)
    tmpe <- scale(ee, center = TRUE, scale = TRUE)
    scaley <- list(
      center = attr(tmpx, "scaled:center"),
      scale = attr(tmpx, "scaled:scale")
    )
    scalee <- list(
      center = attr(tmpe, "scaled:center"),
      scale = attr(tmpe, "scaled:scale")
    )

    xx <- scale(xx, center = scaley$center, scale = scaley$scale)
    xx <- xx[, 1]
    ee <- scale(ee, center = scalee$center, scale = scalee$scale)
    ee <- ee[, 1]
  }


  # Check xreg class & dim
  xxreg <- NULL
  scalexreg <- NULL

  if (!is.null(xreg)) {
    xxreg <- xreg <- as.matrix(xreg)
    if (length(x) != NROW(xreg)) {
      stop("Number of rows in xreg does not match series length")
    }

    # Check for NAs in xreg
    if (any(is.na(xreg))) {
      warning("Missing values in xreg, omitting rows")
    }

    # Scale xreg
    if (scale.inputs) {
      tmpx <- scale(xxreg, center = TRUE, scale = TRUE)
      scalexreg <- list(
        center = attr(tmpx, "scaled:center"),
        scale = attr(tmpx, "scaled:scale")
      )

      xxreg <- scale(xxreg, center = scalexreg$center, scale = scalexreg$scale)
    }
  }


  # Set up lagged matrix
  n <- length(xx)
  xx <- as.ts(xx)
  m <- max(round(frequency(xx)), 1L)


  if (m == 1) {

    if (missing(p)) {
      p <- max(length(ar(na.interp(xx))$ar), 1)
    }

    if (missing(q)) {
      if (!missing(er)){
        arfima_model <- arfima(as.ts(y), xreg = if (!is.null(xreg)) as.ts(xreg) else NULL)
        q <- max(order(arfima_model$ma),0)
      }

      else{
        q <- max(order(arfima_model$ma),0)
      }
    }

    if(missing(size)){
      size <- floor((q + p) / 2)
    }

    # For non-seasonal data also use default calculation for p if that argument is 0, but issue a warning
    if (p == 0){
      warning("Cannot set p = 0 for non-seasonal data; using default calculation for p")
      p <- max(length(ar(na.interp(xx))$ar), 1)
    }

    if (q == 0){
      warning("Cannot set q = 0; setting q to 1")
      q <- 1
    }

    if (p >= n) {
      warning("Reducing number of lagged inputs due to short series")
      p <- n - 1
    }

    lags <- seq_len(p)
    lagse <- seq_len(q)


    if (P > 1) {
      warning("Non-seasonal data, ignoring seasonal lags")
    }

    P <- 0
  }


  # Seasonal data
  else {

    if (missing(p)) {

      if (n > 2 * m) {
        x.sa <- seasadj(mstl(forecast::na.interp(xx)))
      }

      else {
        x.sa <- na.interp(xx)
      }

      p <- max(length(ar(x.sa)$ar), 1)
    }

    if (missing(q)) {
      q <- max(order(arfima_model$ma),0)
    }

    if(missing(size)){
      size <- floor((q + p) / 2)
    }


    if (p == 0 && P == 0){
      stop("'p' and 'P' cannot both be zero")
    }


    if (p >= n) {
      warning("Reducing number of lagged inputs due to short series")
      p <- n - 1
    }


    if (P > 0 && n >= m * P + 2) {
      lagse <- sort(unique(c(seq_len(q), m * (seq_len(P)))))
      lags <- seq_len(p)
    }

    else {
      lagse <- seq_len(q)
      lags <- seq_len(p)

      if (P > 0) {
        warning("Series too short for seasonal lagse")
        P <- 0
      }
    }
  }


  if(missing(skip)){
    skip <- TRUE
  }


  # Setting up lagged matrices for Y and Errors
  maxlage <- max(lagse)
  maxlage <- ifelse(maxlage < 0, 0, maxlage)
  nlage <- length(lagse)
  maxlag <- max(lags)
  nlag <- length(lags)

  er <- ee[-(1:(max(maxlag, maxlage)))]
  y <- xx[-(1:(max(maxlag, maxlage)))]


  # lagged matrix for x(y)
  lags.X <- matrix(NA_real_, ncol = nlag, nrow = n - maxlag)
  for (i in 1:nlag)
    lags.X[, i] <- xx[(maxlag - lags[i] + 1):(n - lags[i])]


  # lagged matrix for er
  lags.E <- matrix(NA_real_, ncol = nlage, nrow = n - maxlage)
  for (i in 1:nlage)
    lags.E[, i] <- ee[(maxlage - lagse[i] + 1):(n - lagse[i])]

  max_lag_all = max(maxlag, maxlage)

  if (p >= q){
    lags.E = lags.E[(p-q+1):nrow(lags.E),]
  }

  if(p < q){
    lags.X = lags.X[(q-p+1):nrow(lags.X),]
  }


  # Combining lags of y and errors
  lags.X <- cbind(lags.X, lags.E)

  # Add xreg into lagged matrix
  lags.X <- cbind(lags.X, xxreg[-(1:max_lag_all), , drop = FALSE])


  # Remove missing values if present
  j <- complete.cases(lags.X, y)


  # Stop if there's no data to fit (e.g. due to NAs or NaNs)
  if (NROW(lags.X[j,, drop=FALSE]) == 0) {
    stop("No data to fit (possibly due to NA or NaN)")
  }


  # Passing the value(y and err combine matrix) to average on nnet function
  if(skip == FALSE){
    fit <- avnnet(lags.X[j, , drop = FALSE], y[j], size = size, repeats = repeats)
  }


  if(skip == TRUE){
    fit <- avnnet_T(lags.X[j, , drop = FALSE], y[j], size = size, repeats = repeats)
  }


  # To return the output
  out <- list()
  out$y <- as.ts(x)
  out$e <- as.ts(e)
  out$m <- m
  out$p <- p
  out$q <- q
  out$P <- P
  out$scaley <- scaley
  out$scalee <- scalee
  out$scalexreg <- scalexreg
  out$size <- size
  out$xreg <- xreg
  out$skip <- skip
  out$lambda <- lambda
  out$lambdae <- lambdae
  out$model <- fit
  out$nnetargs <- list(...)

  if (NROW(lags.X[j,, drop = FALSE]) == 1){
    message("coe")
    fits <- c(rep(NA_real_, maxlag), mean(sapply(fit, predict)))
  } else{
    fits <- c(rep(NA_real_, max(maxlag, maxlage)), rowMeans(sapply(fit, predict)))
  }


  if (scale.inputs) {
    fits <- fits * scaley$scale + scaley$center
  }


  fits <- ts(fits)


  if (!is.null(lambda)) {
    fits <- InvBoxCox(fits, lambda)
  }

  out$fitted <- ts(rep(NA_real_, length(out$y)))
  out$fitted[c(rep(TRUE, max(maxlag, maxlage)), j)] <- fits
  tsp(out$fitted) <- tsp(out$y)
  out$residuals <- out$y - out$fitted
  out$lags <- lags
  out$lagse <- lagse
  out$series <- yname
  out$method <- paste("NARFIMA(", p,",",q, sep = "")

  if (P > 0) {
    out$method <- paste(out$method, ",", P, sep = "")
  }
  out$method <- paste(out$method, ",", size, "," , skip,")", sep = "")
  if (P > 0) {
    out$method <- paste(out$method, "[", m, "]", sep = "")
  }
  out$call <- match.call()
  return(structure(out, class = c("narfima")))
}


# Aggregate several neural network models
avnnet <- function(x, y, repeats, linout = TRUE, trace = FALSE, ...) {
  mods <- list()
  for (i in 1:repeats)
    mods[[i]] <- nnet::nnet(x, y, linout = linout, trace = trace, ...)
  return(structure(mods, class = "narfima"))
}



# Aggregate several neural network models
avnnet_T <- function(x, y, repeats, linout = TRUE, trace = FALSE, ...) {
  mods <- list()
  for (i in 1:repeats)
    mods[[i]] <- nnet::nnet(x, y, linout = linout, trace = trace, skip = TRUE, ...)
  return(structure(mods, class = "narfima"))
}



#' @export
print.nnetarmodels <- function(x, ...) {
  cat(paste("\nAverage of", length(x), "networks, each of which is\n"))
  print(x[[1]])
}
