#' Print an 'ordregr' or an 'ordgam' object.
#'
#' @description Print a summary of the information contained in an \code{\link{ordregr.object}} or \code{\link{ordgam.object}} generated by \code{\link{ordregr}} or \code{\link{ordgam}}.
#'
#' @usage
#' \method{print}{ordregr}(x, expEst, ...)
#'
#' @param x  An \link{ordregr.object} generated by \link{ordregr} or \link{ordgam}.
#' @param expEst Logical indicating if the exponential of the regression coefficients should be printed (Default: TRUE)
#' @param ... Possible additional printing options.
#'
#' @return Print summary statistics.
#'
#' @author Philippe Lambert \email{p.lambert@uliege.be}
#' @references
#' Lambert, P. and Gressani, 0. (2023) Penalty parameter selection and asymmetry corrections
#' to Laplace approximations in Bayesian P-splines models. Statistical Modelling. <doi:10.1177/1471082X231181173>. Preprint: <arXiv:2210.01668>.
#'
#' @seealso \code{\link{ordregr}}, \code{\link{ordgam}}
#'
#' @author Philippe Lambert \email{p.lambert@uliege.be}
#'
#' @examples
#' library(ordgam)
#' data(freehmsData)
#' mod = ordgam(freehms ~ gndr + s(eduyrs) + s(age),
#'              data=freehmsData, descending=TRUE)
#' print(mod)
#' plot(mod)
#'
#' @export
print.ordregr <- function(x,expEst=TRUE, ...){
    obj = x
    cat("Call:\n",deparse(obj$call),"\n")
    ##
    if (obj$use.prior){
        cat("\nPrior set on the regression paramaters <beta>\n")
    } else {
        cat("\nPrior: none\n")
    }
    ##
    if ("ordgam" %in% class(obj)){
        idx = 1:with(obj,nalpha+nfixed)
    } else {
        idx = 1:length(obj$theta)
    }
    ##
    cat("\nParameter estimation:\n")
    printCoefmat(obj$theta.mat[idx,,drop=FALSE],
                 digits=3,P.values=TRUE,has.Pvalue=TRUE,signif.legend=FALSE, ...)
    ##
    if (expEst & obj$nfixed > 0){ ## Print exp(est)
        cat("\n")
        idx2 = with(obj, (nalpha+1):(nalpha+nfixed))
        mat = with(obj, cbind("exp(est)"=exp(theta.mat[,"est"]),
                                  "exp(-est)"=exp(-theta.mat[,"est"]),
                                  low=exp(theta.mat[,"low"]),
                                  up=exp(theta.mat[,"up"]),
                                  Pval=theta.mat[,"Pval"]) )
        printCoefmat(mat[idx2,,drop=FALSE], ##cs.ind=1:4,tst.ind=5,
                     digits=3,P.values=TRUE,has.Pvalue=TRUE,signif.legend=FALSE, ...)
    }
    ##
    if ("ordgam" %in% class(obj)){
        if (obj$regr$J > 0){
            cat("\nApproximate significance of smooth terms:\n")
            temp = cbind(obj$ED.Tr,obj$ED.Chi2)[,-4,drop=FALSE]
            printCoefmat(temp,digits=3,cs.ind=1L,tst.ind=c(2,4),
                         P.values=TRUE,has.Pvalue=TRUE,signif.legend=FALSE, ...)
            if (obj$select.lambda){
                cat("\nSelected penalty parameters <lambda>: ",obj$lambda,"\n")
                cat("Lambda log prior:  ") ; print(body(obj$lprior.lambda))
            } else {
                cat("\n(Fixed) penalty parameters <lambda>: ",obj$lambda,"\n")
            }
        }
    }
    ##
    cat("\nLikelihood - Information criterions (n = ",obj$n,"):\n",sep="")
    print(round(with(obj,c(edf=sum(ED.full),logL=llik,logLmarg=levidence,AIC=AIC,BIC=BIC)),2))
    ##
    if (obj$descending){
        cat("\nNOTE: model the odds of a response value in the upper scale\n")
    } else {
        cat("\nNOTE: model the odds of a response value in the lower scale\n")
    }
}
