% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/palaeorotate.R
\name{palaeorotate}
\alias{palaeorotate}
\title{Palaeorotate fossil occurrences}
\usage{
palaeorotate(
  occdf,
  lng = "lng",
  lat = "lat",
  age = "age",
  model = "MERDITH2021",
  method = "point",
  uncertainty = TRUE,
  round = 3
)
}
\arguments{
\item{occdf}{\code{data.frame}. Fossil occurrences to be
palaeogeographically reconstructed. \code{occdf} should contain columns
with longitudinal and latitudinal coordinates, as well as age estimates.
The age of rotation should be supplied in millions of years before
present.}

\item{lng}{\code{character}. The name of the column you wish to be treated
as longitude (defaults to "lng").}

\item{lat}{\code{character}. The name of the column you wish to be treated
as latitude (defaults to "lat").}

\item{age}{\code{character}. The name of the column you wish to be treated
as the age for rotation (defaults to "age").}

\item{model}{\code{character}. The name(s) of the Global Plate Model(s)
to be used to reconstruct palaeocoordinates. See details for available
models.}

\item{method}{\code{character}. Method used to calculate palaeocoordinates
for fossil occurrences. Either "grid" to use reconstruction files, or
"point" (default) to use the GPlates API service. See details section for
specific details.}

\item{uncertainty}{\code{logical}. Should the uncertainty in
palaeogeographic reconstructions be returned? If set to TRUE (default),
the palaeolatitudinal range and maximum geographic distance (in km)
between output palaeocoordinates are calculated. This argument is only
relevant if more than one Global Plate Model is specified in \code{model}.}

\item{round}{\code{numeric}. Numeric value indicating the number of decimal
places \code{lng}, \code{lat} and \code{age} should be rounded to. This functionality is
only relevant for the "point" \code{method}. Rounding can speed up
palaeorotation by reducing the number of unique coordinate pairs.
Defaults to a value of 3. If no rounding is desired, set this value to
\code{NULL}.}
}
\value{
A \code{data.frame} containing the original input occurrence
\code{data.frame} and the reconstructed coordinates (i.e. "p_lng", "p_lat"). The
"grid" \code{method} also returns the age of rotation ("rot_age") and the
reference coordinates rotated ("rot_lng" and "rot_lat"). If only one
model is requested, a column containing the rotation model used
("rot_model") is also appended. Otherwise, the name of each model is
appended to the name of each column containing palaeocoordinates (e.g.
"p_lng_GOLONKA"). If \code{uncertainty} is set to \code{TRUE}, the
palaeolatitudinal range ("range_p_lat") and the maximum geographic
distance ("max_dist") in km between palaeocoordinates will also be
returned (the latter calculated via \code{\link[geosphere]{distGeo}}).
}
\description{
A function to estimate palaeocoordinates for fossil occurrence data
(i.e. reconstruct the geographic distribution of organisms'
remains at time of deposition). Each occurrence is assigned
palaeocoordinates based on its current geographic position and age
estimate.
}
\details{
This function can estimate palaeocoordinates using two different
approaches (\code{method}):
\itemize{
\item Reconstruction files: The "grid" \code{method} uses reconstruction files from
Jones & Domeier (2024) to spatiotemporally link present-day geographic
coordinates and age estimates with a discrete global grid rotated at one
million-year time steps throughout the Phanerozoic (540--0 Ma). Here,
resolution 3 (~119 km spacing) of the reconstruction files is used. All
files, and the process used to generate them, are available and documented
in Jones & Domeier (2024). If fine-scale spatial analyses are being
conducted, use of the "point" \code{method} (see GPlates API below) may be
preferred (particularly if occurrences are close to plate boundaries). When
using the "grid" \code{method}, coordinates within the same grid cell will be
assigned equivalent palaeocoordinates due to spatial aggregation. However,
this approach enables efficient estimation of the past distribution of
fossil occurrences. Note: each reconstruction file is ~45 MB in size.
\item GPlates API: The "point" \code{method} uses the \href{https://gwsdoc.gplates.org}{GPlates Web Service} to reconstruct palaeocoordinates for point
data. The use of this \code{method} is slower than the "grid" \code{method} if many
unique time intervals exist in your dataset. However, it provides
palaeocoordinates with higher precision.
}

Available models and timespan for each \code{method}:
\itemize{
\item "MERDITH2021" (Merdith et al., 2021)
\itemize{
\item 0--1000 Ma (point)
\item 0--540 Ma (grid)
}
\item "TorsvikCocks2017" (Torsvik and Cocks, 2016)
\itemize{
\item 0--540 Ma (point/grid)
}
\item "PALEOMAP" (Scotese, 2016)
\itemize{
\item 0--1100 Ma (point)
\item 0--540 Ma (grid)
}
\item "MATTHEWS2016_pmag_ref"  (Matthews et al., 2016)
\itemize{
\item 0--410 Ma (grid/point)
}
\item "GOLONKA" (Wright et al., 2013)
\itemize{
\item 0--540 Ma (grid/point)
}
}
}
\section{References}{

\itemize{
\item Jones, L.A., Domeier, M. A Phanerozoic gridded dataset for palaeogeographic
reconstructions. Sci Data 11, 710 (2024). \doi{10.1038/s41597-024-03468-w}.
\item Matthews, K.J., Maloney, K.T., Zahirovic, S., Williams, S.E., Seton, M.,
and Müller, R.D. (2016). Global plate boundary evolution and kinematics
since the late Paleozoic. Global and Planetary Change, 146, 226-250.
\doi{10.1016/j.gloplacha.2016.10.002}.
\item Merdith, A., Williams, S.E., Collins, A.S., Tetley, M.G., Mulder, J.A.,
Blades, M.L., Young, A., Armistead, S.E., Cannon, J., Zahirovic, S.,
Müller. R.D. (2021). Extending full-plate tectonic models into deep time:
Linking the Neoproterozoic and the Phanerozoic. Earth-Science Reviews,
214(103477). \doi{10.1016/j.earscirev.2020.103477}.
\item Scotese, C., & Wright, N. M. (2018). PALEOMAP Paleodigital Elevation Models
(PaleoDEMs) for the Phanerozoic. \href{https://www.earthbyte.org/paleodem-resource-scotese-and-wright-2018/}{PALEOMAP Project}.
\item Torsvik, T. H. & Cocks, L. R. M. Earth History and Palaeogeography.
Cambridge University Press, 2016.
\item Wright, N., Zahirovic, S., Müller, R. D., & Seton, M. (2013). Towards
community-driven paleogeographic reconstructions: integrating open-access
paleogeographic and paleobiology data with plate tectonics. Biogeosciences,
10(3), 1529-1541. \doi{10.5194/bg-10-1529-2013}.
}

See \href{https://gwsdoc.gplates.org/reconstruction}{GPlates documentation}
for additional information and details.
}

\section{Developer(s)}{

Lewis A. Jones
}

\section{Reviewer(s)}{

Kilian Eichenseer, Lucas Buffan & Will Gearty
}

\examples{
\dontrun{
#Generic example with a few occurrences
occdf <- data.frame(lng = c(2, -103, -66),
                lat = c(46, 35, -7),
                age = c(88, 125, 200))

#Calculate palaeocoordinates using reconstruction files
ex1 <- palaeorotate(occdf = occdf, method = "grid")

#Calculate palaeocoordinates using the GPlates API
ex2 <- palaeorotate(occdf = occdf, method = "point")

#Calculate uncertainity in palaeocoordinates from models
ex3 <- palaeorotate(occdf = occdf,
                    method = "grid",
                    model = c("MERDITH2021",
                              "GOLONKA",
                              "PALEOMAP"),
                    uncertainty = TRUE)

#Now with some real fossil occurrence data!

#Grab some data from the Paleobiology Database
data(tetrapods)

#Assign midpoint age of fossil occurrence data for reconstruction
tetrapods$age <- (tetrapods$max_ma + tetrapods$min_ma)/2

#Rotate the data
ex3 <- palaeorotate(occdf = tetrapods)

#Calculate uncertainity in palaeocoordinates from models
ex4 <- palaeorotate(occdf = tetrapods,
                    model = c("MERDITH2021",
                              "GOLONKA",
                              "PALEOMAP"),
                    uncertainty = TRUE)
}
}
