#' @title Generate Mermaid syntax for a pedigree flowchart in Markdown & R Markdown
#'
#' @details
#' The Mermaid syntax generated by the \code{\link{mermaid_md}} function can be embeded in Markdown files and Markdown renderes such as GitHub, but not in R Markdown.
#' Currently, Mermaid flowcharts in R Markdown have more limitations in terms of shape and style customizations, and the required syntax is less compact.
#' Note that the generated syntax by this function can be embeded in both R Markdown and Markdown.
#'
#' @param ped : A data frame with the mandatory columns ID, SIRE, and DAM,
#' and the optional columns BgColor, BorderColor, RoundBorder, DashBorder, and lwd.
#' The optional columns define (child) node-specific shape and style (corresponding to the ID column).
#' The order of columns does not matter, but column names do matter and case-sensitive. \cr
#'
#' `ped` columns: \cr
#'
#' ID : Numeric or alphanumeric individual identifications. \cr
#'
#' SIRE : Sire identifications. Missing sires are denoted as 0. \cr
#'
#' DAM : Dam identifications. Missing dams are denoted as 0. \cr
#'
#' BgColor : Child (corresponding to ID) node's background color (valid color names and valid hex color codes).
#' If this column is missing, the default color (`"#ECECFF"`) is used.
#' Also, `NA` enforces the default color. \cr
#'
#' BorderColor : Child (corresponding to ID) node's border color (valid color names and valid hex color codes).
#' If this column is missing, the default color (`"#9370DB"`) is used.
#' Also, `NA` enforces the default color. \cr
#'
#' RoundBorder : Child (corresponding to ID) node's border shape (90\eqn{\circ} vs rounded edges).
#' This column receives `"Y"` (rounded edges), `"N"` (90\eqn{\circ} edges), and `NA` (equivalent to the default value).
#' If this column is missing, the default border shape (`"N"`) is used. \cr
#'
#' DashBorder : Child (corresponding to ID) node's border line style (dashed vs solid).
#' This column receives `"Y"` (solid line), `"N"` (dashed line), and `NA` (equivalent to the default value).
#' If this column is missing, the default border line style (`"N"`) is used. \cr
#'
#' lwd : Child (corresponding to ID) node's border line width.
#' This column receives values > 0, \eqn{\leq} 5, and `NA` (equivalent to the default value).
#' If this column is missing, the default value (1) is used. \cr
#'
#' @param orient : Defines the orientation of the flowchart (`"TB"` for top-to-bottom vs `"LR"` for left-to-right).
#' If no input is provided, the default orientation (`"TB"`) is used.
#' @param type : Defines the type of links in the flowchart (`"arrow"` vs `"line"`).
#' If no input is provided, the default link type (`"arrow"`) is used.
#'
#' @return : A vector of character strings. Each character string is a Mermaid syntax line.
#' Assuming the returned output is saved in object `x`,
#' use `cat(x, sep = "\n")` to display the output on-screen,
#' and `cat(x, sep = "\n", file = "output.txt")` or `write(x, file = "output.txt")` to write the output into a file.
#'
#' @examples
#' # A sample pedigree data frame with only the three mandatory columns.
#' ped <- data.frame(ID = 1:7, SIRE = c(0, 0, 1, 0, 3, 0, 5), DAM = c(0, 0, 2, 2, 4, 0, 6))
#'
#' # Example 1: A pedigree Mermaid syntax without customizations.
#' x <- mermaid_rmd(ped)
#' # Read the "Value" part about displaying the output on-screen and writing it into a file.
#'
#' # Example 2: Repeat example 1. Change arrow links to lines and the orientation to horizontal.
#' x <- mermaid_rmd(ped, orient = "LR", type = "line")
#'
#' # Example 3: Repeat example 1. Pink background and round border edges for females (2, 4, 6).
#' ped$BgColor <- c(NA, "pink", NA, "pink", NA, "pink", NA)
#' ped$RoundBorder <- c(NA, "Y", NA, "Y", NA, "Y", NA)
#' x <- mermaid_rmd(ped)
#'
#' # Example 4: Repeat example 3. Ticker border line for individuals in the control group (2, 5, 7).
#' ped$lwd <- c(1, 3, 1, 1, 3, 1, 3)
#' x <- mermaid_rmd(ped)
#'
#' # Example 5: Use the default value and NA alternatively. This is not different from example 4.
#' ped$lwd <- c(NA, 3, NA, NA, 3, NA, 3)
#' x <- mermaid_rmd(ped)
#'
#' @export
mermaid_rmd <- function(ped, orient = "TB", type = "arrow") {
    # Check inputs
    ## ped
    if (!is.data.frame(ped)) stop("The pedigree object is not a data.frame.")
    if (!all(c("ID", "SIRE", "DAM") %in% colnames(ped))) stop("ID, SIRE, and DAM are the mandatory columns of the pedigree data.frame, and should be present.")
    if (any(is.na(ped[, c("ID", "SIRE", "DAM")]))) stop("Found missing value(s) in the mandatory columns (ID, SIRE, and DAM) of the pedigree data.frame.")
    if (any(ped$ID == 0)) stop("Found 0 in the ID column of the pedigree data.frame.")
    if (any(duplicated(ped$ID))) stop("Found duplicates in the ID column of the pedigree data.frame.")
    if ("BgColor" %in% colnames(ped) & any(!is.color_name(unique(na.omit(ped$BgColor))) & !sapply(unique(na.omit(ped$BgColor)), is.hex_color))) {
        stop("Found an invalid color name/code in the BgColor column of the pedigree data.frame.")
    }
    if ("BorderColor" %in% colnames(ped) & any(!is.color_name(unique(na.omit(ped$BorderColor))) & !sapply(unique(na.omit(ped$BorderColor)), is.hex_color))) {
        stop("Found an invalid color name/code in the BorderColor column of the pedigree data.frame.")
    }
    if ("RoundBorder" %in% colnames(ped) & any(!unique(ped$RoundBorder) %in% c(NA, "N", "Y"))) {
        stop('Found an invalid element in the RoundBorder column of the pedigree data.frame. The acceptable elements are "Y", "N", and NA.')
    }
    if ("DashBorder" %in% colnames(ped) & any(!unique(ped$DashBorder) %in% c(NA, "Y", "N"))) {
        stop('Found an invalid element in the DashBorder column of the pedigree data.frame. The acceptable elements are "Y", "N", and NA.')
    }
    if ("lwd" %in% colnames(ped)) {
        if (min(na.omit(ped$lwd)) <= 0) stop("Found a value <= 0 in the lwd column of the pedigree data.frame.")
        if (max(na.omit(ped$lwd)) > 5) stop("Found a value > 5 in the lwd column of the pedigree data.frame.")
    }
    ## Other arguments
    for (i in c("orient", "type"))
    {
        if (length(get(i)) != 1) stop(paste0("length(", i, ") != 1"))
    }
    if (!orient %in% c("TB", "LR")) stop('The acceptable inputs to the orient argument are "TB" and "LR".')
    if (!type %in% c("arrow", "line")) stop('The acceptable inputs to the type argument are "arrow" and "line".')

    # Modify inputs
    if (type == "arrow") {
        type <- " --> "
    } else {
        type <- " --- "
    }
    ## Set defaults to NA (nothing to paste using paste.na.omit())
    ### The mandatory columns of ped
    ped[ped$SIRE == 0, ]$SIRE <- NA
    ped[ped$DAM == 0, ]$DAM <- NA
    ### The optional columns of ped
    if ("BgColor" %in% colnames(ped)) {
        if (any(toupper(ped$BgColor) %in% "#ECECFF")) ped[toupper(ped$BgColor) %in% "#ECECFF", ]$BgColor <- NA
        if (all(is.na(ped$BgColor))) {
            ped$BgColor <- NULL # Drop the column
        } else {
            ped[!is.na(ped$BgColor), ]$BgColor <- paste0("fill:", ped[!is.na(ped$BgColor), ]$BgColor)
        }
    }
    if ("BorderColor" %in% colnames(ped)) {
        if (any(toupper(ped$BorderColor) %in% "#9370DB")) ped[toupper(ped$BorderColor) %in% "#9370DB", ]$BorderColor <- NA
        if (all(is.na(ped$BorderColor))) {
            ped$BorderColor <- NULL # Drop the column
        } else {
            ped[!is.na(ped$BorderColor), ]$BorderColor <- paste0("stroke:", ped[!is.na(ped$BorderColor), ]$BorderColor)
        }
    }
    if ("RoundBorder" %in% colnames(ped)) {
        if (any(ped$RoundBorder %in% "N")) ped[ped$RoundBorder %in% "N", ]$RoundBorder <- NA
        if (all(is.na(ped$RoundBorder))) ped$RoundBorder <- NULL # Drop the column
    }
    if ("DashBorder" %in% colnames(ped)) {
        if (any(ped$DashBorder %in% "N")) ped[ped$DashBorder %in% "N", ]$DashBorder <- NA
        if (all(is.na(ped$DashBorder))) {
            ped$DashBorder <- NULL # Drop the column
        } else {
            ped[ped$DashBorder %in% "Y", ]$DashBorder <- "stroke-dasharray:4"
        }
    }
    if ("lwd" %in% colnames(ped)) {
        if (any(ped$lwd %in% 1)) ped[ped$lwd %in% 1, ]$lwd <- NA
        if (all(is.na(ped$lwd))) {
            ped$lwd <- NULL # Drop the column
        } else {
            ped[!is.na(ped$lwd), ]$lwd <- paste0("stroke-width:", ped[!is.na(ped$lwd), ]$lwd)
        }
    }

    # Write graph.line
    graph.line <- paste("graph", orient)

    # Write style.lines, if any of the 4 of 5 optional columns exists
    style.lines <- NA
    if (any(c("BgColor", "BorderColor", "DashBorder", "lwd") %in% colnames(ped))) {
        for (i in c("BgColor", "BorderColor", "DashBorder", "lwd"))
        {
            # Create an NA column for the missing ones.
            if (!i %in% colnames(ped)) ped[, i] <- NA
        }
        ped$style <- apply(ped[, c("BgColor", "BorderColor", "DashBorder", "lwd")], 1, paste.na.omit, sep = ",")
        if (any(ped$style == "")) ped[ped$style == "", ]$style <- NA
        style.lines <- paste("    style", ped[!is.na(ped$style), ]$ID, ped[!is.na(ped$style), ]$style)
    }

    # Write node.lines
    ## Filter ped rows depending on the presence of ped$RoundBorder
    if ("RoundBorder" %in% colnames(ped)) {
        ped <- ped[!is.na(ped$SIRE) | !is.na(ped$DAM) | !is.na(ped$RoundBorder), ]
    } else {
        ped <- ped[!is.na(ped$SIRE) | !is.na(ped$DAM), ]
    }
    ## Create parent and child nodes
    node.lines <- ped[, colnames(ped) %in% c("ID", "SIRE", "DAM", "RoundBorder")]
    colnames(node.lines)[colnames(node.lines) == "ID"] <- "childnode"
    node.lines$rownum <- 1:nrow(node.lines)
    node.lines.set1 <- node.lines[!is.na(node.lines$SIRE), colnames(node.lines) != "DAM"]
    colnames(node.lines.set1)[colnames(node.lines.set1) == "SIRE"] <- "parentnode"
    node.lines.set2 <- node.lines[!is.na(node.lines$DAM), colnames(node.lines) != "SIRE"]
    colnames(node.lines.set2)[colnames(node.lines.set2) == "DAM"] <- "parentnode"
    if ("RoundBorder" %in% colnames(ped) & nrow(ped[is.na(ped$SIRE) & is.na(ped$DAM) & !is.na(ped$RoundBorder), ]) > 0) {
        node.lines.set3 <- node.lines[is.na(node.lines$SIRE) & is.na(node.lines$DAM) & !is.na(node.lines$RoundBorder), ]
        node.lines.set3$parentnode <- NA
        node.lines.set3 <- node.lines.set3[, colnames(node.lines.set1)]
        node.lines <- rbind(node.lines.set3, node.lines.set1, node.lines.set2)
    } else {
        node.lines <- rbind(node.lines.set1, node.lines.set2)
    }
    node.lines <- node.lines[order(node.lines$rownum), colnames(node.lines) != "rownum"]
    ## If RoundBorder column exists, apply it
    if ("RoundBorder" %in% colnames(node.lines) & any(node.lines$RoundBorder %in% "Y")) {
        node.lines[node.lines$RoundBorder %in% "Y", ]$childnode <- paste0(
            node.lines[node.lines$RoundBorder %in% "Y", ]$childnode, "(",
            node.lines[node.lines$RoundBorder %in% "Y", ]$childnode, ")"
        )
    }
    ## Link parentnode to childnode
    node.lines <- apply(node.lines[, c("parentnode", "childnode")], 1, paste.na.omit, sep = type)
    node.lines <- trimws(node.lines)
    ## Indent
    node.lines <- paste("   ", node.lines)

    # Cocatenate lines into the output
    output <- c("```{r, echo=FALSE}", 'mermaid("', na.omit(c(graph.line, node.lines, style.lines)), '")', "```")

    # Return the output
    return(output)
}
