% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/one_compartment_oral_nl.R
\name{one_compartment_oral_nl}
\alias{one_compartment_oral_nl}
\title{One-Compartment Oral Pharmacokinetic Model (Nonlinear, First-Order Absorption)}
\arguments{
\item{data}{A data frame containing plasma concentration-time data.}

\item{time_col}{Character string specifying the column name for time.}

\item{conc_col}{Character string specifying the column name for plasma concentration.}

\item{dose}{Numeric value specifying the administered oral dose.}

\item{group_col}{Optional character string specifying a grouping variable
(e.g., formulation, subject).}

\item{plot}{Logical; if TRUE, generates a concentration-time plot with fitted curves.}

\item{annotate}{Logical; if TRUE, annotates the plot with PK parameters
(only if <= 2 groups).}
}
\value{
A list containing:
\describe{
  \item{\code{fitted_parameters}}{Data frame with k_a, k_el, Tmax, Cmax, Vd/F, CL/F, and R^2.}
  \item{\code{data}}{Processed data used for fitting and plotting.}
}
}
\description{
Fits plasma concentration-time data to a one-compartment pharmacokinetic model
following oral administration with first-order absorption and first-order
elimination. The model assumes that drug absorption from the gastrointestinal
tract and systemic elimination are both proportional to the amount of drug
present (linear kinetics), and that the body can be represented as a single,
well-mixed compartment.

Model parameters are estimated by nonlinear least squares regression for each
group (if specified). The primary parameters include the absorption rate constant
(k_a), elimination rate constant (k_el), and apparent volume of distribution
(Vd/F). Model-consistent secondary pharmacokinetic parameters are derived,
including time to maximum concentration (Tmax), maximum concentration (Cmax),
area under the concentration-time curve (AUC), and apparent clearance (CL/F),
where AUC = Dose / (k_el * Vd/F) and CL/F = k_el * Vd/F.

The function includes safeguards against numerical instability when k_a and
k_el are similar, enforces positive parameter bounds during fitting, and performs
validity checks for the computation of Tmax and Cmax. Model performance is
summarized using root mean squared error (RMSE) and information criteria
(AIC and BIC), which are more appropriate than R^2 for nonlinear pharmacokinetic
models.

If grouping is specified (e.g., by subject or formulation), the model is fit
independently to each group. Groups with insufficient observations trigger a
warning, as parameter estimates may be unreliable.

When plotting is enabled, the function generates publication-quality
concentration-time plots showing observed data and the corresponding
model-predicted curves based on the fitted parameters. Annotations summarizing
key pharmacokinetic parameters and model diagnostics are optionally added for
one group.

An optional LOESS (locally estimated scatterplot smoothing) curve may be overlaid
on the observed data for exploratory visualization only. This smoother is purely
descriptive, does not represent any pharmacokinetic mechanism, and should not be
interpreted as part of the fitted model. For this reason, LOESS smoothing is
disabled by default.

Model:
C(t) = ((F * Dose * k_a) / (Vd * (k_a - k_el))) * [exp(-k_el * t) - exp(-k_a * t)]
}
\examples{
# Example I: Single subject oral dosing
df <- data.frame(
  time = c(0.25, 0.5, 1, 2, 4, 6, 8, 12),
  concentration = c(1.2, 2.8, 5.1, 6.4, 5.2, 4.1, 3.0, 1.8)
)
one_compartment_oral_nl(
  data = df,
  time_col = "time",
  conc_col = "concentration",
  dose = 100
)

# Example II: Condition-dependent oral pharmacokinetics (e.g., formulation or pH effect)
df_cond <- data.frame(
  time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8), 2),
  concentration = c(
    1.4, 3.1, 5.6, 6.8, 5.9, 4.7, 3.6,   # Condition A
    0.9, 2.2, 4.1, 5.3, 4.8, 3.9, 3.0    # Condition B
  ),
  condition = rep(c("Condition A", "Condition B"), each = 7)
)
one_compartment_oral_nl(
  data = df_cond,
  time_col = "time",
  conc_col = "concentration",
  dose = 100,
  group_col = "condition"
)

# Example III: Multiple subjects (population-style oral pharmacokinetics)
df_subjects <- data.frame(
  time = rep(c(0.25, 0.5, 1, 2, 4, 6, 8, 12), 6),
  concentration = c(
    1.3, 3.0, 5.4, 6.7, 5.8, 4.6, 3.5, 2.3,   # Subject 1
    1.1, 2.7, 5.0, 6.3, 5.5, 4.4, 3.3, 2.2,   # Subject 2
    1.0, 2.5, 4.7, 6.0, 5.3, 4.2, 3.2, 2.1,   # Subject 3
    0.9, 2.3, 4.4, 5.7, 5.0, 4.0, 3.0, 2.0,   # Subject 4
    0.8, 2.1, 4.1, 5.4, 4.8, 3.8, 2.9, 1.9,   # Subject 5
    0.7, 2.0, 3.9, 5.2, 4.6, 3.7, 2.8, 1.8    # Subject 6
  ),
  subject = rep(paste0("S", 1:6), each = 8)
)
one_compartment_oral_nl(
  data = df_subjects,
  time_col = "time",
  conc_col = "concentration",
  dose = 100,
  group_col = "subject"
)
}
\references{
Gibaldi, M. & Perrier, D. (1982) <isbn:9780824710422> Pharmacokinetics,
2nd Edition. Marcel Dekker, New York.

Gabrielsson, J. & Weiner, D. (2000) <isbn:9186274929> Pharmacokinetic/Pharmacodynamic
Data Analysis: Concepts and Applications, 3rd Edition, Revised and Expanded.
Swedish Pharmaceutical Press, Stockholm.
}
\author{
Paul Angelo C. Manlapaz
}
