% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/bed.R, R/ini.R
\name{scanBED}
\alias{scanBED}
\alias{loopBED}
\alias{bed}
\title{travers variants in a PLINK1 BED fileset}
\usage{
scanBED(
  pfx,
  FUN,
  ...,
  win = 1,
  iid = 1,
  vid = 1,
  vfr = NULL,
  vto = NULL,
  buf = 2^24,
  simplify = TRUE
)

loopBED(
  pfx,
  EXP,
  GVR = "g",
  win = 1,
  iid = 1,
  vid = 1,
  vfr = NULL,
  vto = NULL,
  buf = 2^24,
  simplify = TRUE
)
}
\arguments{
\item{pfx}{prefix of PLINK BED.}

\item{FUN}{a function to process each window of variants;}

\item{...}{additional argument for \emph{\code{FUN}} when \code{scanBED} is used.}

\item{win}{reading window size (def=100 variants per window)}

\item{iid}{option to read \code{N} IID as row names (def=1, see \code{\link[=readIID]{readIID()}}).}

\item{vid}{option to read \code{P} VID as col names (def=1, see \code{\link[=readVID]{readVID()}}).}

\item{vfr}{variant-wise, from where to read (number/proportion, def=1)?}

\item{vto}{varinat-wise, to where then stop (number/proportion, def=P)?}

\item{buf}{buffer size in byptes (def=2^24, or 16 MB).}

\item{simplify}{try simplifying the results  into an array, or leave them in
a list, or specify a function to simplify the said list.}

\item{EXP}{a R expression to evaluate with each window of variants;}

\item{GVR}{a R variable name to assign the window to (def="g").}
}
\value{
results of all windows processed by the user script.
}
\description{
Sequentially visits variants in a PLINK1  BED fileset with a stepping window
matrix, and process each window matrix  with user scripts either in function
or expression form, meant for data to big to fit in the memory.

To read the entire BED into a R matrix, use \verb{[readBED]()} instead.
}
\section{Functions}{
\itemize{
\item \code{scanBED()}: apply a function to variants in a PLINK1 BED fileset

Travers \code{P}  variants via a sliding  window while calling a  function on each
window of variants without side effects on the calling environment, mimicking
various R \code{apply} utilities.

\item \code{loopBED()}: evaluate an expression on variants in a PLINK1 BED

Travers \code{P} variants via a sliding  window and evaluate an R expression given
each  window of  variants,  with  side effects  on  the calling  environment,
mimicking the syntax of R \code{for} loop.

}}
\section{BED PLINK1 Binary Pedigree fileset}{

A popular format to store biallelic dosage genotype, with three files,
\itemize{
\item \emph{pfx}.fam: text table for \code{N} individuals, detailed in \link{readFAM};
\item \emph{pfx}.bim: text table for \code{P} variants, detailed in \link{readBIM};
\item \emph{pfx}.bed: transposed genotype matrix (\code{P} x \code{N} ) in binary format.
}

The triplets are commonly referred by the shared prefix (\code{pfx}), e.g., the X
chromosome represented by "chrX.bed", "chrX.fam", and "chrX.bim" are refered
by \code{"chrX"}.

The binary file "\emph{pfx}.bed" represent each dosage value with two bits - just
enough to encode all four possiblities: 0, 1, or 2 alleles, or missing.

The number of variants (\code{P}) and samples (\code{N}) equals to the number of lines
in text file "\emph{pfx}.bim" and "\emph{pfx}.fam", respectively.

For the detailed specification of PLINK1 BED genotype format, see the lagecy
PLINK v1.07 page at: \\
\url{https://zzz.bwh.harvard.edu/plink/binary.shtml}. \\
For the modern use and management of PLINK1 BED, see the PLINK v1.9 page: \\
\url{https://www.cog-genomics.org/plink/1.9/input#bed}.
}

\section{detailed arguments}{

\itemize{
\item \code{win}: visiting window size.

the number of variants per window, that  is, the number of columns in each
window matrix passed to the user script.

For example, a size one window means  the user script will be dealing with
only one variant at  a time, received from in a matrix  of a single column
-- a manner similar to  genome wide association analysis (GWAS).  However,
a larger, multi-variant window coupled with R language's vector and matrix
syntax can significantly boost efficiency.

The default size is 1000 variants / columns per window.
\item \code{buf}: buffer size in bytes

a large buffer reduces the frequency of hard disk visits when traversing a
PLINK1 BED file, which in turn reduces non-computation overhead.

The default size is \code{2^24} bytes, or 16 MB.
\item \code{simplify}:

when FALSE: resuts  of user script processing each window  of variants are
returned in a list;

when TRUE,  use \code{simplify2array} to put  the results into an  array, if it
fails, fallback and return a list.

when a function is specified, it is then used to simplify the results, if
an execption is thrown, fallback and return a list.

e.g., the window script returns a  data frame of estimate, standard error,
t-statistic, and p-value  for each variant, \code{simplify =  rbind} to combine
results of all windows into one data frame of \code{P} rows and four columns of
statistics.
}
}

\section{genotype context}{

context infomation  such the number of  variants and samples are  updated in
the window processing environment to ease user scripting, which includes:
\itemize{
\item \code{.i}: indies of variants in the current visiting window;
\item \code{.p}: number of variants in the current visiting window.
\item \code{.P}: total number of variants;
\item \code{.w}: index of the current window;
\item \code{.W}: total number of windows to go through;
\item \code{.N}: number of individuals.
\item \code{.b}: index of the current buffer.
\item \code{.B}: number of buffers to be swapped.
}

e.g. (1) print percentage progress with \code{print(.w / .W * 100)}; \\
e.g. (2) use \code{inf <- readBIM(pfx)} to  read the table of variants before the
window visits,  later use \code{inf[.i,  ]} to  access meta-data for  variants in
each window.
}

\examples{
## traverse genotype, apply R function without side effects
pfx <- file.path(system.file("extdata", package="plinkFile"), "000")
ret <- scanBED(pfx, function(g)
{
    .af <- colMeans(g, na.rm=TRUE) / 2
    maf <- pmin(.af, 1 - .af)
    mis <- colSums(is.na(g)) / .N
    pct <- round(.w / .W * 100, 2)
    cbind(buf=.b, wnd=.w, idx=.i, MAF=maf, MIS=mis, PCT=pct)
},
vfr=NULL, vto=NULL, win=13, simplify=rbind, buf=2^18)
head(ret)
tail(ret)

## traversing genotype, evaluate R expression with side effects
pfx <- file.path(system.file("extdata", package="plinkFile"), "000.bed")
ret <- list() # use side effect to keep the result of each window.
loopBED(pfx,
{
    af <- colMeans(gt, na.rm=TRUE) / 2
    sg <- af * (1 - af)
    ret[[.w]] <- cbind(wnd=.w, alf=af, var=sg)
},
win=13, GVR="gt", vid=3, buf=2^18)
head(ret)
tail(ret)

}
\seealso{
{\verb{[readBED]}}
}
