% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tool_methods.R
\name{predict.plm}
\alias{predict.plm}
\title{Model Prediction for plm Objects}
\usage{
\method{predict}{plm}(
  object,
  newdata = NULL,
  na.fill = !inherits(newdata, "pdata.frame"),
  ...
)
}
\arguments{
\item{object}{An object of class \code{"plm"},}

\item{newdata}{An optional pdata.frame in which to look for variables to be
used for prediction. If \code{NULL}, the fitted values are returned.
For fixed effects models, supplying a pdata.frame is recommended.}

\item{na.fill}{A logical, only relevant if \code{object} is a pdata.frame, indicating
whether for any supplied out-of-sample indexes (individual, time,
combination of both), the missing fixed effect estimate is filled
with the weighted mean of the model's present fixed effect estimates
or not.}

\item{\dots}{further arguments.}
}
\value{
A numeric (or a pseries if \code{newdata} is a pdata.frame) carrying the
predicted values with length equal to the number of rows as the data
supplied in \code{newdata} and with names the row names of \code{newdata} or, if
\code{newdata = NULL}, the original model's fitted values given in \code{object}.
}
\description{
Predicted values of response based on plm models.
}
\details{
\code{predict}calculates predicted values by evaluating the regression function of
a plm model for \code{newdata} or, if \code{newdata = NULL}, it returns the fitted values
the plm model.

The fixed effects (within) model is somewhat special in prediction as it has
fixed effects estimated per individual, time period (one-way) or both (two-ways
model) which should to be respected when predicting values relating to these
fixed effects in the model: To do so, it is recommended to supply a pdata.frame
(and not a plain data.frame) in \code{newdata} as it describes the relationship
between the data supplied to the individual. and/or time periods. In case
the \code{newdata}´'s pdata.frame has out-of-sample data (data contains individuals
and/or time periods not contained in the original model), it is not clear
how values are to be predicted and the result will contain \code{NA}
values for these out-of-sample data. Argument \code{na.fill} can be set to \code{TRUE}
to apply the original model's weighted mean of fixed effects for the
out-of-sample data to derive a prediction.

If a plain data.frame is given in \code{newdata} for a fixed effects model, the
weighted mean is used for all fixed effects as \code{newdata} for prediction as a
plain data.frame cannot describe any relation to individuals/time periods
(\code{na.fill} is automatically set to \code{TRUE} and the function warns).

See also \strong{Examples}.
}
\examples{
library(plm)
data("Grunfeld", package = "plm")

# fit a fixed effect model
fit.fe <- plm(inv ~ value + capital, data = Grunfeld, model = "within")

# generate 55 new observations of three firms used for prediction:
#  * firm 1 with years 1935:1964 (has out-of-sample years 1955:1964), 
#  * firm 2 with years 1935:1949 (all in sample),
#  * firm 11 with years 1935:1944 (firm 11 is out-of-sample)
set.seed(42L)

new.value2   <- runif(55, min = min(Grunfeld$value),   max = max(Grunfeld$value))
new.capital2 <- runif(55, min = min(Grunfeld$capital), max = max(Grunfeld$capital))

newdata <- data.frame(firm = c(rep(1, 30), rep(2, 15), rep(11, 10)),
                      year = c(1935:(1935+29), 1935:(1935+14), 1935:(1935+9)),
                      value = new.value2, capital = new.capital2)
# make pdata.frame
newdata.p <- pdata.frame(newdata, index = c("firm", "year"))

## predict from fixed effect model with new data as pdata.frame
predict(fit.fe, newdata = newdata.p)

## set na.fill = TRUE to have the weighted mean used to for fixed effects -> no NA values
predict(fit.fe, newdata = newdata.p, na.fill = TRUE)

## predict with plain data.frame from fixed effect model: uses mean fixed effects 
## for prediction and, thus, yields different result with a warning
predict(fit.fe, newdata = newdata)

}
\keyword{regression}
