#' @title
#' Computing downsets.
#'
#' @description
#' Computes the downset of a set of elements of the input poset.
#'
#' @param poset An object of S4 class `POSet`.
#' Argument `poset` must be created by using any function contained in the package aimed at building object of S4 class `POSet`
#' (e.g. [POSet()], [LinearPOSet()], [ProductPOSet()], ...) .
#'
#' @param elements A vector of character strings (the names of the input elements).
#'
#' @return
#' A vector of character strings (the names of the elements of the downset).
#'
#' @examples
#' elems<- c("a", "b", "c", "d")
#'
#' dom <- matrix(c(
#'   "a", "b",
#'   "c", "b",
#'   "a", "d"
#' ), ncol = 2, byrow = TRUE)
#'
#' pos <- POSet(elements = elems, dom = dom)
#'
#' dwn <- DownsetOf(pos, c("b","d"))
#'
#' @name DownsetOf
#' @export DownsetOf
DownsetOf <- function(poset, elements) {
  if (!methods::is(poset, "POSet")) {
    stop("poset must be of class POSet")
  }
  tryCatch({
    if (!is.character(elements))
      stop("element must be a string vector")
    elements <- unique(elements)
    result <- .Call("_DownsetOf", poset@ptr, elements)
    return (result)
  }, error = function(err) {
    # error handler picks up where error was generated
    print(paste("MY_ERROR:  ",err))
  }) # END tryCatch
}
