% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/poth.R
\name{poth}
\alias{poth}
\alias{print.poth}
\alias{summary.poth}
\alias{print.summary.poth}
\title{Calculate precision of treatment hierarchy (POTH) metric}
\usage{
poth(x, se = NULL, small.values, pooled, trts = NULL)

\method{print}{poth}(x, sort = TRUE, digits = 3, ...)

\method{summary}{poth}(object, ...)

\method{print}{summary.poth}(x, sort = TRUE, digits = 3, ...)
}
\arguments{
\item{x}{Mandatory argument with suitable information on the treatment
hierarchy (see Details).}

\item{se}{Matrix of estimated standard errors for relative effects.}

\item{small.values}{A character string specifying whether small
outcome values indicate a beneficial (\code{"desirable"}) or
harmful (\code{"undesirable"}) effect, can be abbreviated.}

\item{pooled}{A character string indicating whether the treatment hierarchy
is based on a common or random effects model. Either \code{"common"} or
\code{"random"}, can be abbreviated.}

\item{trts}{An optional vector with treatment names. Must match the
order of treatments provided for argument \code{x}.}

\item{sort}{A logical indicating whether printout should be sorted
by decreasing ranking metric.}

\item{digits}{Minimal number of significant digits, see
\code{\link{print.default}}.}

\item{\dots}{Additional arguments (ignored).}

\item{object}{An object of class \code{summary.poth}.}
}
\value{
An object of class \code{poth} with corresponding \code{print}
function. The object is a list containing the following components:
\item{poth}{Separation in ranking metric.}
\item{ranking}{A named numeric vector with rankings, i.e.,
  SUCRAs or P-scores.}
\item{ranking.matrix}{A square matrix with the probabilities
  for each possible rank (if information is available).}
\item{pooled}{As defined above.}
}
\description{
Precision of treatment hierarchy (POTH) is a metric to quantify the uncertainty in
a treatment hierarchy in network meta-analysis
}
\details{
This function calculates the precision of treatment hierarchy (POTH) metric
to quantify the uncertainty in a treatment hierarchy in network
meta-analysis (Wigle et al., 2024).

Argument \code{x} providing information on the treatment hierarchy is the
only mandatory argument. The following input formats can be provided:
\enumerate{
 \item vector representing a ranking metric, i.e., SUCRAs or P-scores,
 \item square matrix with the probabilities for each possible rank
 (with treatments in rows and ranks in columns),
 \item MCMC samples (with samples in rows and treatments in columns),
 \item relative effect matrix,
 \item R object created with \code{\link[netmeta]{netmeta}},
   \code{\link[netmeta]{netrank}}, or \code{\link[netmeta]{rankogram}}
   object from R package \bold{netmeta}.
}

Argument \code{se} must be provided if argument \code{x} is a matrix with
relative effects. Otherwise, argument \code{se} is ignored.

Argument \code{small.values} must be provided if argument \code{x} contains
MCMC samples, relative effects, or is an object created with
\code{\link[netmeta]{netmeta}}. This argument can be provided for an R
object created with \code{\link[netmeta]{netrank}} or
\code{\link[netmeta]{rankogram}} and is ignored otherwise.

For a \code{\link[netmeta]{netmeta}}, \code{\link[netmeta]{netrank}}, or
\code{\link[netmeta]{rankogram}} object, argument \code{pooled} is by
default equal to "random" if only the random effects model was considered in
the analysis.

Argument \code{trts} is ignored for \code{\link[netmeta]{netmeta}},
\code{\link[netmeta]{netrank}}, and \code{\link[netmeta]{rankogram}} objects.
}
\examples{
\donttest{
library("netmeta")
data(smokingcessation)
p1 <- pairwise(list(treat1, treat2, treat3),
  event = list(event1, event2, event3), n = list(n1, n2, n3),
  data = smokingcessation, sm = "OR")
net1 <- netmeta(p1, random = FALSE)

# Calculate probabilities for each possible rank
set.seed(1909) # make results reproducible
rg1 <- rankogram(net1)
rg1

# Calculate POTH
s1 <- poth(rg1)
s1

# Also print probabilities for each possible rank
summary(s1)

# Use SUCRAs to calculate POTH
nr1 <- netrank(rg1)
nr1
poth(nr1)
poth(nr1$ranking.common)

data(Senn2013)
net2 <- netmeta(TE, seTE, treat1.long, treat2.long, studlab,
                data = Senn2013, sm = "MD", random = FALSE)

# Use P-scores to calculate POTH
nr2 <- netrank(net2)
nr2
poth(nr2)
}

}
\references{
Wigle A, Béliveau A, Salanti G, Rücker G, Schwarzer G, Mavridis D,
Nikolakopoulou A (2024):
Precision of treatment hierarchy: A metric for quantifying uncertainty in
treatment hierarchies in network meta-analysis.
Preprint on arXiv, \doi{10.48550/arXiv.2501.11596}
}
\author{
Augustine Wigle \email{amhwigle@uwaterloo.ca},
  Guido Schwarzer \email{guido.schwarzer@uniklinik-freiburg.de}
}
\keyword{print}
\keyword{summary}
