% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/power_spectral_density.R
\name{SingleBinPSDIntegrationOrDominantFreqComparison}
\alias{SingleBinPSDIntegrationOrDominantFreqComparison}
\title{Given sets of windows (dataframes) corresponding to different combos, see if the integration
or dominant frequency of a specific frequency range is significantly different between the combos}
\usage{
SingleBinPSDIntegrationOrDominantFreqComparison(
  list.of.windows,
  name.of.col.containing.time.series,
  level1.column.name,
  level2.column.name,
  level.combinations,
  level.combinations.labels,
  sampling_frequency,
  single.bin.boundary = NULL,
  x_start = NULL,
  x_end = NULL,
  x_increment = NULL,
  integration.or.dominant.freq
)
}
\arguments{
\item{list.of.windows}{A list of windows (dataframes).}

\item{name.of.col.containing.time.series}{A string that specifies the name of the column in the windows that correspond to the time series that should be used for making PSD.}

\item{level1.column.name}{A String that specifies the column name to use for the first level. This column should only contain one unique value within each window.}

\item{level2.column.name}{A String that specifies the column name to use for the second level. This column should only contain one unique value within each window.}

\item{level.combinations}{A List containing Lists. Each list that it contains has two vectors. The first vector specifying the values for level1 and the second vector specifying the values for level2. Each list element will correspond to a new line on the plot.}

\item{level.combinations.labels}{A vector of strings that labels each combination. This is used for naming the groups in integrals.with.combo.labels}

\item{sampling_frequency}{Numeric value specifying sampling frequency in hertz. If data is sampled once every second, then sampling frequency is 1 Hz. If data is sampled once every 2 seconds, then sampling frequency is 0.5 Hz.}

\item{single.bin.boundary}{A numeric vector with two elements. First element is the start frequency for the bin. Second element is the end frequency of the bin.}

\item{x_start}{Numeric value specifying start of the new x-axis for the averaged PSD. Default is 0 Hz.}

\item{x_end}{Numeric value specifying end of the new x-axis for the averaged PSD. Maximum value is the sampling_frequency divided by 2.}

\item{x_increment}{Numeric value specifying increment of the new x-axis for the averaged PSD.}

\item{integration.or.dominant.freq}{A string with two possible values for choosing whether integral or dominant frequency should be calculated and compared: "integration" or "dominant_freq".}
}
\value{
A list with 3 objects:
\enumerate{
\item integrals.with.combo.labels: Dataframe used for statistical testing.
\item kruskal.test.res: Results from Kruskal-Willis testing.
\item pairwise.wilcox.rest.res: Results from pairwise Wilcoxo testing
}
}
\description{
Just for a single frequency bin: For Each combination in level.combinations,
generate the integral or dominant frequency for each
window of each combination. At this point, we should have vectors of
integrals or dominant frequency with each vector
corresponding to a different combo. Now we want to see if the integrals or dominant frequencies
in each combo significantly differ from the other combos. Kruskal-Wallis test is used
as a non-parametric ANOVA test to see if the combos have integrals or dominant frequencies that
are significantly different.
}
\details{
Need to specify whether to compare integral or dominant frequency:
If integral (total power in frequency bin) is the value to compare, then
SingleBinPSDIntegrationForMultipleWindows() is used. If dominant frequency (
frequency corresponding to max PSD value in frequency bin) is the value to
compare, then PSDDominantFrequencyForMultipleWindows() is used.
}
\examples{


#Create a vector of time that represent times where data are sampled.
Fs = 100; #sampling frequency in Hz
T = 1/Fs; #sampling period
L = 1000; #length of time vector
t = (0:(L-1))*T; #time vector

#First signal
#1. 1 Hz with amplitude of 2
S1 <- 2*sin(2*pi*1*t)
level1.vals <- rep("a", length(S1))
level2.vals <- rep("1", length(S1))
S1.data.frame <- as.data.frame(cbind(t, S1, level1.vals, level2.vals))
colnames(S1.data.frame) <- c("Time", "Signal", "level1.ID", "level2.ID")
S1.data.frame[,"Signal"] <- as.numeric(S1.data.frame[,"Signal"])

#Second signal
#1. 1 Hz with amplitude of -4
#2. 2 Hz with amplitude of -2
S2 <- (-4)*sin(2*pi*1*t) - 2*sin(2*pi*2*t);
level1.vals <- rep("a", length(S2))
level2.vals <- rep("2", length(S2))
S2.data.frame <- as.data.frame(cbind(t, S2, level1.vals, level2.vals))
colnames(S2.data.frame) <- c("Time", "Signal", "level1.ID", "level2.ID")
S2.data.frame[,"Signal"] <- as.numeric(S2.data.frame[,"Signal"])

#Third signal
#1. 1 Hz with amplitude of 2
#2. 2 Hz with amplitude of 2
S3 <- 2*sin(2*pi*1*t) + 2*sin(2*pi*2*t);
level1.vals <- rep("a", length(S3))
level2.vals <- rep("3", length(S3))
S3.data.frame <- as.data.frame(cbind(t, S3, level1.vals, level2.vals))
colnames(S3.data.frame) <- c("Time", "Signal", "level1.ID", "level2.ID")
S3.data.frame[,"Signal"] <- as.numeric(S3.data.frame[,"Signal"])

#Fourth signal
#1. 1 Hz with amplitude of -2
S4 <- -2*sin(2*pi*1*t)
level1.vals <- rep("b", length(S4))
level2.vals <- rep("3", length(S4))
S4.data.frame <- as.data.frame(cbind(t, S4, level1.vals, level2.vals))
colnames(S4.data.frame) <- c("Time", "Signal", "level1.ID", "level2.ID")
S4.data.frame[,"Signal"] <- as.numeric(S4.data.frame[,"Signal"])

#Fifth signal
#1. 5 Hz with amplitude of -2
S5 <- -2*sin(2*pi*5*t)
level1.vals <- rep("c", length(S5))
level2.vals <- rep("1", length(S5))
S5.data.frame <- as.data.frame(cbind(t, S5, level1.vals, level2.vals))
colnames(S5.data.frame) <- c("Time", "Signal", "level1.ID", "level2.ID")
S5.data.frame[,"Signal"] <- as.numeric(S5.data.frame[,"Signal"])

#Extra representation of S2 dataframe to show an example that has enough samples
#to have significance for Kruskal-Wallis test
windows <- list(S1.data.frame, S2.data.frame, S2.data.frame, S2.data.frame, S2.data.frame,
S2.data.frame, S2.data.frame, S2.data.frame, S2.data.frame, S2.data.frame, S3.data.frame,
S4.data.frame,
S5.data.frame, S5.data.frame, S5.data.frame, S5.data.frame, S5.data.frame)

#Gets the composite of the first, second, and third signal. Should result in a flat signal.
FirstComboToUse <- list( c("a"), c(1, 2, 3) )

#Gets the composite of the third and fourth signal
SecondComboToUse <- list( c("a", "b"), c(3) )

#Gets the composite of fifth signal
ThirdComboToUse <- list( c("c"), c(1) )


#PSD-------------------------------------------------------------------------

PSD.results <- AutomatedCompositePlotting(list.of.windows = windows,
                           name.of.col.containing.time.series = "Signal",
                           x_start = 0,
                           x_end = 10,
                           x_increment = 0.01,
                           level1.column.name = "level1.ID",
                           level2.column.name = "level2.ID",
                           level.combinations = list(FirstComboToUse,
                                                    SecondComboToUse,
                                                    ThirdComboToUse),
                           level.combinations.labels = c("Signal 1 + 2 + 3",
                                                         "Signal 3 + 4",
                                                         "Signal 5"),
                           plot.title = "Example",
                           plot.xlab = "Hz",
                           plot.ylab = "(Original units)^2/Hz",
                           combination.index.for.envelope = 2,
                           TimeSeries.PSD.LogPSD = "PSD",
                           sampling_frequency = 100)

ggplot.obj.PSD <- PSD.results[[2]]

#Integration-------------------------------------------------------------------------

#Compare integration for the 1.5-2.5 Hz bin. P-value should not indicate
#significant difference
integration.compare.res <- SingleBinPSDIntegrationOrDominantFreqComparison(
list.of.windows = windows,
name.of.col.containing.time.series = "Signal",
level1.column.name = "level1.ID",
level2.column.name = "level2.ID",
level.combinations = list(FirstComboToUse, SecondComboToUse),
level.combinations.labels = c("Signal 1 + 2 + 3", "Signal 3 + 4"),
sampling_frequency = 100,
single.bin.boundary = c(1.5, 2.5),
integration.or.dominant.freq = "integration")

#Kruskal-Wallis test results
integration.compare.res[[2]]

#Compare integration for the 0.5-1.5 Hz bin. P-value should indicate
#significant difference
integration.compare.res2 <- SingleBinPSDIntegrationOrDominantFreqComparison(
list.of.windows = windows,
name.of.col.containing.time.series = "Signal",
level1.column.name = "level1.ID",
level2.column.name = "level2.ID",
level.combinations = list(FirstComboToUse, SecondComboToUse),
level.combinations.labels = c("Signal 1 + 2 + 3", "Signal 3 + 4"),
sampling_frequency = 100,
single.bin.boundary = c(0.5,1.5),
integration.or.dominant.freq = "integration")

#Kruskal-Wallis test results
integration.compare.res2[[2]]



#Dominant Frequency---------------------------------------------------------------------

#Compare dominant frequency P-value should not indicate
#significant difference
integration.compare.res3 <- SingleBinPSDIntegrationOrDominantFreqComparison(
list.of.windows = windows,
name.of.col.containing.time.series = "Signal",
level1.column.name = "level1.ID",
level2.column.name = "level2.ID",
level.combinations = list(FirstComboToUse, SecondComboToUse),
level.combinations.labels = c("Signal 1 + 2 + 3", "Signal 3 + 4"),
sampling_frequency = 100,
x_start = 0,
x_end = 10,
x_increment = 0.01,
integration.or.dominant.freq = "dominant_freq")

#Kruskal-Wallis test results
integration.compare.res3[[2]]


#Compare dominant frequency P-value should indicate
#significant difference
integration.compare.res4 <- SingleBinPSDIntegrationOrDominantFreqComparison(
list.of.windows = windows,
name.of.col.containing.time.series = "Signal",
level1.column.name = "level1.ID",
level2.column.name = "level2.ID",
level.combinations = list(SecondComboToUse, ThirdComboToUse),
level.combinations.labels = c("Signal 3 + 4", "Signal 5"),
sampling_frequency = 100,
x_start = 0,
x_end = 10,
x_increment = 0.01,
integration.or.dominant.freq = "dominant_freq")

#Kruskal-Wallis test results
integration.compare.res4[[2]]
#Values used in comparison of the two groups
integration.compare.res4[[1]]


}
