#' Adaptive Piecewise Linear Approximation of a Continuous Function
#'
#' Approximates a continuous function \eqn{f} on a domain \eqn{[a, b]} by
#' adaptively discretizing the domain and building a piecewise linear (PWL)
#' envelope until the maximum error between the PWL and \eqn{f} is within a given tolerance.
#'
#' @param f A continuous function \( f(x) \) to approximate.
#' @param domain Numeric vector of length 2 specifying the interval \eqn{[a, b]}.
#' @param tol Numeric tolerance for maximum allowed approximation error (default 1e-3).
#' @param max_iter Maximum number of refinement iterations (default 20).
#' @param initial_points Initial number of discretization points (default 5).
#' @param smallconst Numeric small constant used in building PWL envelope (default 1e-4).
#'
#' @return A list with components:
#' \describe{
#'   \item{PWL}{A matrix of piecewise linear segments: slope, intercept, lower bound, upper bound.}
#'   \item{data}{The final discretization points (x, y) used in fitting.}
#'   \item{max_error}{Maximum absolute error between \( f \) and the PWL approximation.}
#' }
#'
#' @examples
#' f <- function(x) log(x)
#' domain <- c(1, 10)
#' res <- adaptive_pwl_fit(f, domain, tol = 1e-4, initial_points = 10, smallconst = 0.01)
#'
#' cat("x,y\n")
#' for(i in 1:nrow(res$data)) {
#'   cat(paste(res$data[i, 1], res$data[i, 2], sep = ","), "\n")
#' }
#'
#' @export
adaptive_pwl_fit <- function(f, domain, tol = 1e-3, max_iter = 50, initial_points = 5, smallconst = 1e-4) {
  # Extract domain bounds
  a <- domain[1]
  b <- domain[2]

  # Initial discretization points and function values
  x_vals <- seq(a, b, length.out = initial_points)
  y_vals <- f(x_vals)
  data <- cbind(x_vals, y_vals)

  iter <- 1
  max_error <- Inf

  # Track the best (smallest error) PWL result
  best_pwl <- NULL
  best_data <- NULL
  min_error <- Inf

  while (iter <= max_iter && max_error > tol) {
    # Build piecewise linear envelope
    pwl_result <- build_pwl_envelope(data, smallconst)

    # Dense grid to evaluate error
    dense_x <- seq(a, b, length.out = 1000)
    dense_y <- f(dense_x)

    # Evaluate PWL approximation on dense_x
    seg_lower <- pwl_result$PWL[, 3]
    seg_upper <- pwl_result$PWL[, 4]

    # Find the segment index for each dense_x point (returns index of interval in seg_lower)
    idx_vec <- findInterval(dense_x, seg_upper)

    # Initialize pwl_y with NAs
    pwl_y <- rep(NA_real_, length(dense_x))

    # Compute pwl_y only for valid indices
    valid_idx <- which(idx_vec > 0 & idx_vec <= nrow(pwl_result$PWL))

    for(i in valid_idx) {
      idx <- idx_vec[i]
      slope <- pwl_result$PWL[idx, 1]
      intercept <- pwl_result$PWL[idx, 2]
      pwl_y[i] <- slope * dense_x[i] + intercept
    }

    valid <- !is.na(pwl_y)
    errors <- abs(pwl_y[valid] - dense_y[valid])
    max_error <- max(errors)

    message(sprintf("Iteration: %d - Max Error: %.5g", iter, max_error))

    # Track best result so far
    if (max_error < min_error) {
      min_error <- max_error
      best_pwl <- pwl_result$PWL
      best_data <- data
    }

    # Stop if error within tolerance
    message(sprintf("  Diff from tol: %.8g", max_error - tol))
    if (max_error <= tol) break

    # Add new discretization point where error is largest
    max_err_idx <- which.max(errors)
    new_x <- dense_x[max_err_idx]
    new_y <- dense_y[max_err_idx]

    data <- rbind(data, c(new_x, new_y))
    data <- data[order(data[,1]), , drop = FALSE]

    iter <- iter + 1
  }

  message(sprintf("Best max error achieved: %.5g", min_error))
  cat("Best PWL segments:\n")
  for (i in 1:nrow(best_pwl)) {
    slope <- best_pwl[i, 1]
    intercept <- best_pwl[i, 2]
    x0 <- best_pwl[i, 3]
    x1 <- best_pwl[i, 4]
    cat(sprintf("  y = %.6f * x + %.6f   on [%.6f, %.6f]\n", slope, intercept, x0, x1))
  }

  return(list(PWL = best_pwl, data = best_data, max_error = min_error))
}
