% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/pill_buttons.R
\name{pill_buttons}
\alias{pill_buttons}
\title{Surround text or numeric values in a colored pill button}
\usage{
pill_buttons(
  data,
  colors = "#15607A",
  color_ref = NULL,
  opacity = 1,
  number_fmt = NULL,
  show_text = TRUE,
  text_size = NULL,
  text_color = "black",
  text_color_ref = NULL,
  brighten_text = TRUE,
  brighten_text_color = "white",
  bold_text = FALSE,
  box_shadow = FALSE,
  tooltip = FALSE,
  animation = "background 1s ease"
)
}
\arguments{
\item{data}{Dataset containing either a text or numeric column.}

\item{colors}{The background color of the pill button.
Only a single color can be provided for columns containing text.
Multiple colors can be provided for columns containing values and
should be given in order from low values to high values.
If multiple colors are provided for columns containing text,
the first color in the vector will be assigned to the text.
The default color provided is "#15607A".
Can use R's built-in colors or other color packages.}

\item{color_ref}{Optionally assign colors to from another column
by providing the name of the column containing the colors in quotes.
Only one color can be provided per row.
Default is NULL.}

\item{opacity}{A value between 0 and 1 that adjusts the opacity in color(s).
A value of 0 is fully transparent, a value of 1 is fully opaque.
Default is 1.}

\item{number_fmt}{Optionally format numbers using formats from the scales package.
Default is NULL.}

\item{show_text}{Logical: show text or hide text.
Default is TRUE.}

\item{text_size}{Numeric value representing the size of the text labels.
Default is NULL.}

\item{text_color}{Assigns text color to values.
Default is black.}

\item{text_color_ref}{Optionally assign text color from another column
by providing the name of the column containing the text colors in quotes.
Only one color can be provided per cell.
Default is NULL.}

\item{brighten_text}{Logical: automatically assign color to text based on background color of the pill button.
Text within dark-colored backgrounds will turn white, text within light-colored backgrounds will be black.
Default is TRUE.}

\item{brighten_text_color}{Assigns text color to values if values are within a dark-colored pill button backgrounds.
Default is white.}

\item{bold_text}{Logical: bold text.
Default is FALSE.}

\item{box_shadow}{Logical: add a box shadow to the buttons.
Default is FALSE.}

\item{tooltip}{Logical: hover tooltip.
Default is FALSE.}

\item{animation}{Control the duration and timing function of the animation
when sorting/updating values shown on a page.
See [CSS transitions](https://developer.mozilla.org/en-US/docs/Web/CSS/transition)
for available timing functions and examples.
Animation can be turned off by setting to "none".
Default is "background 1s ease".}
}
\value{
a function that surrounds text/values in a column
    with a colored pill button.
}
\description{
The `pill_buttons()` function surrounds text or numeric values in a column in a colored pill button.
    If `pill_buttons()` is applied to a column containing text, the color of the pill button can be provided via a single color can be provided within `color`
    or via another column in the dataset by referencing the column name with `color_ref` (conditionally applying colors to text).
    If `pill_buttons` is applied to a numeric column, either a single color or a vector of colors can be provided within `color`
    or the colors can also be assigned via `color_ref`.
    The opacity of the color(s) provided can be adjusted by providing a value between 0 and 1 in `opacity`.
    The color of the text/values can be changed using `text_color`.
    If text/values are displayed within a dark-colored background, `brighten_text` will display the text/values in white so they are more visible.
    The color of `brighten_text_color` can be changed to a color other than white if desired.
    The horizontal alignment of `pill_buttons()` can be controlled using reactable::colDef(align = "center").
    `pill_buttons()` needs to placed within the cell argument in reactable::colDef.
}
\examples{
library(dplyr)
data <- iris[45:54, ]

## Surround text with pill buttons:
reactable(
data,
columns = list(
Species = colDef(cell = pill_buttons(data))))

## Conditionally apply colors from another column:
data \%>\%
mutate(color_assign = case_when(
Species == "setosa" ~ "red",
Species == "versicolor" ~ "forestgreen",
TRUE ~ "grey")) \%>\%
reactable(.,
columns = list(
Species = colDef(cell = pill_buttons(., color_ref = "color_assign"))))

## Surround numeric values with pill buttons:
reactable(
data,
columns = list(
Petal.Width = colDef(cell = pill_buttons(data))))

## Apply multiple colors to numeric values:
reactable(
data,
columns = list(
Petal.Width = colDef(
cell = pill_buttons(data,
                    colors = c("lightpink","white","lightgreen"),
                    opacity = 0.3))))

## Apply a box shadow to the buttons to give a 3-D effect:
reactable(
data,
columns = list(
Petal.Width = colDef(
cell = pill_buttons(data,
                    box_shadow = TRUE,
                    colors = c("lightpink","white","lightgreen"),
                    opacity = 0.3))))

}
