#' Projection-Weighted Canonical Correlation Analysis
#' 
#' Compute pairwise projection-weighted CCA (PWCCA) similarities between multiple representations.
#' PWCCA reweights canonical directions by the magnitude of each
#' representation’s projection onto those directions, emphasizing components
#' that are most used by the representation.
#'
#' @param mats A list of length \eqn{M} containing data matrices of size
#'   \eqn{(n_\mathrm{samples},\, p_k)}. All matrices must share the same number
#'   of rows for matching samples.
#'
#' @return An \eqn{M \times M} symmetric matrix of PWCCA similarities.
#' 
#' @references 
#' \insertRef{morcos_2018_InsightsRepresentationalSimilarity}{repsim}
#' 
#' @seealso \code{\link{cca}}
#' 
#' @examples
#' \donttest{
#' # --------------------------------------------------
#' # Use "iris" and "USArrests" datasets
#' #   1. apply scaling to reduce the effect of scales
#' #   2. add white noise to create multiple representations
#' #   3. generate 10 perturbations per each dataset
#' # --------------------------------------------------
#' # prepare the prototype
#' set.seed(1)
#' X = as.matrix(scale(as.matrix(iris[sample(1:150, 50, replace=FALSE),1:4])))
#' Y = as.matrix(scale(as.matrix(USArrests)))
#' n = nrow(X)
#' p_X = ncol(X)
#' p_Y = ncol(Y)
#' 
#' # generate 10 of each by perturbation
#' mats = vector("list", length=20L)
#' for (i in 1:10){
#'   mats[[i]] = X + matrix(rnorm(n*p_X, sd=1), nrow=n)
#' }
#' for (j in 11:20){
#'   mats[[j]] = Y + matrix(rnorm(n*p_Y, sd=1), nrow=n)
#' }
#' 
#' # compute the similarity
#' sim_pwcca = pwcca(mats)
#' 
#' # visualize
#' opar <- par(no.readonly=TRUE)
#' labs <- paste0("rep ",1:20)
#' par(pty="s")
#' 
#' image(sim_pwcca[,20:1], axes=FALSE, main="PWCCA")
#' axis(1, seq(0, 1, length.out=20), labels = labs, las = 2)
#' axis(2, at = seq(0, 1, length.out=20), labels = labs[20:1], las = 2)
#' 
#' par(opar)
#' }
#' 
#' @export
pwcca <- function(mats){
  # check the data
  aux_checker(mats, "pwcca")
  
  # pass onto C++ part
  return(cpp_PWCCA(mats))
}