% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ACG.R
\name{ACG}
\alias{ACG}
\alias{d_ACG}
\alias{c_ACG}
\alias{r_ACG}
\title{Angular central Gaussian distribution}
\usage{
d_ACG(x, Lambda, log = FALSE)

c_ACG(p, Lambda, log = FALSE)

r_ACG(n, Lambda)
}
\arguments{
\item{x}{locations in \eqn{S^{p-1}} to evaluate the density. Either a
matrix of size \code{c(nx, p)} or a vector of length \code{p}. Normalized
internally if required (with a \code{warning} message).}

\item{Lambda}{the shape matrix \eqn{\boldsymbol{\Lambda}}{\Lambda} of the
ACG. A symmetric and positive definite matrix of size \code{c(p, p)}.}

\item{log}{flag to indicate if the logarithm of the density (or the
normalizing constant) is to be computed.}

\item{p}{dimension of the ambient space \eqn{R^p} that contains
\eqn{S^{p-1}}. A positive integer.}

\item{n}{sample size, a positive integer.}
}
\value{
Depending on the function:
\itemize{
\item \code{d_ACG}: a vector of length \code{nx} or \code{1} with the
evaluated density at \code{x}.
\item \code{r_ACG}: a matrix of size \code{c(n, p)} with the random sample.
\item \code{c_ACG}: the normalizing constant.
}
}
\description{
Density and simulation of the Angular Central Gaussian (ACG)
distribution on
\eqn{S^{p-1}:=\{\mathbf{x}\in R^p:||\mathbf{x}||=1\}}{
S^{p-1} := \{x \in R^p : ||x|| = 1\}}, \eqn{p\ge 1}. The density at
\eqn{\mathbf{x} \in S^{p-1}}{x \in S^{p-1}}, \eqn{p\ge 2}, is given by
\deqn{c^{\mathrm{ACG}}_{p,\boldsymbol{\Lambda}}
(\mathbf{x}' \boldsymbol{\Lambda}^{-1} \mathbf{x})^{-p/2}
\quad\mathrm{with}\quad c^{\mathrm{ACG}}_{p,\boldsymbol{\Lambda}}:=
1 / (\omega_p |\boldsymbol{\Lambda}|^{1/2})}{
c^{ACG}_{p,\Lambda} (x' \Lambda^{-1} x)^{-p/2} with
c^{ACG}_{p,\Lambda} := 1 / (\omega_p |\Lambda|^{1/2}).}
where \eqn{\boldsymbol{\Lambda}}{\Lambda} is the shape matrix, a
\eqn{p\times p}{p x p} symmetric and positive definite matrix, and
\eqn{\omega_p} is the surface area of \eqn{S^{p-1}}.
}
\details{
Due to the projection of the ACG, the shape matrix
\eqn{\boldsymbol{\Lambda}}{\Lambda} is only identified up to a constant,
that is, \eqn{\boldsymbol{\Lambda}}{\Lambda} and
\eqn{c\boldsymbol{\Lambda}}{c\Lambda} give the same ACG distribution.
Usually, \eqn{\boldsymbol{\Lambda}}{\Lambda} is normalized to have trace
equal to \eqn{p}.

\code{c_ACG} is vectorized on \code{p}. If \eqn{p = 1}, then the ACG is the
uniform distribution in the set \eqn{\{-1, 1\}}{{-1, 1}}.
}
\examples{
# Simulation and density evaluation for p = 2
Lambda <- diag(c(5, 1))
n <- 1e3
x <- r_ACG(n = n, Lambda = Lambda)
col <- viridisLite::viridis(n)
r <- runif(n, 0.95, 1.05) # Radius perturbation to improve visualization
dens <- d_ACG(x = x, Lambda = Lambda)
plot(r * x, pch = 16, col = col[rank(dens)])

# Simulation and density evaluation for p = 3
Lambda <- rbind(c(5, 1, 0.5),
                c(1, 2, 1),
                c(0.5, 1, 1))
x <- r_ACG(n = n, Lambda = Lambda)
dens <- d_ACG(x = x, Lambda = Lambda)
scatterplot3d::scatterplot3d(x, xlim = c(-1, 1), ylim = c(-1, 1),
                             zlim = c(-1, 1), color = col[rank(dens)],
                             pch = 16, xlab = "", ylab = "", zlab = "",
                             angle = 20)
}
\references{
Tyler, D. E. (1987). Statistical analysis for the angular central Gaussian
distribution on the sphere. \emph{Biometrika}, 74(3):579--589.
\doi{10.1093/biomet/74.3.579}
}
\seealso{
\code{\link{tangent-elliptical}}.
}
\author{
Eduardo García-Portugués, Davy Paindaveine, and Thomas Verdebout.
}
