% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/cf_lav.R
\name{semPower.powerLav}
\alias{semPower.powerLav}
\title{semPower.powerLav}
\usage{
semPower.powerLav(
  type,
  modelPop = NULL,
  modelH0 = NULL,
  modelH1 = NULL,
  fitH1model = TRUE,
  Sigma = NULL,
  mu = NULL,
  fittingFunction = "ML",
  simulatedPower = FALSE,
  lavOptions = NULL,
  lavOptionsH1 = lavOptions,
  ...
)
}
\arguments{
\item{type}{type of power analysis, one of \code{'a-priori'}, \code{'post-hoc'}, \code{'compromise'}.}

\item{modelPop}{\code{lavaan} model string defining the true model. Can be omitted when \code{Sigma} is set.}

\item{modelH0}{\code{lavaan} model string defining the (incorrect) analysis model.}

\item{modelH1}{\code{lavaan} model string defining the comparison model. If omitted, the saturated model is the comparison model.}

\item{fitH1model}{whether to fit the H1 model. If \code{FALSE}, the H1 model is assumed to show the same fit as the saturated model, and only the delta df are computed.}

\item{Sigma}{can be used instead of \code{modelPop}: population covariance matrix. A list for multiple group models.}

\item{mu}{can be used instead of \code{modelPop}: vector of population means. Can be omitted for no meanstructure. A list for multiple group models.}

\item{fittingFunction}{one of \code{'ML'} (default), \code{'WLS'}, \code{'DWLS'}, \code{'ULS'}. Defines the fitting function used to obtain SigmaHat in analytical power analyses. This also implies a certain discrepancy function used to obtain Fmin.}

\item{simulatedPower}{whether to perform a simulated (\code{TRUE}, rather than analytical, \code{FALSE}) power analysis. See \code{\link[=simulate]{simulate()}} for additional options.}

\item{lavOptions}{a list of additional options passed to \code{lavaan}, e. g., \code{list(estimator = 'mlm')} to request robust ML estimation. Mostly useful in conjunction with \code{simulatedPower}.}

\item{lavOptionsH1}{alternative options passed to \code{lavaan} that are only used for the H1 model. If \code{NULL}, identical to \code{lavOptions}. Probably only useful for multigroup models.}

\item{...}{mandatory further parameters related to the specific type of power analysis requested, see \code{\link[=semPower.aPriori]{semPower.aPriori()}}, \code{\link[=semPower.postHoc]{semPower.postHoc()}}, and \code{\link[=semPower.compromise]{semPower.compromise()}}. See details.}
}
\value{
a list. Use the \code{summary} method to obtain formatted results. Beyond the results of the power analysis and a number of effect size measures, the list contains the following components:
\item{\code{Sigma}}{the population covariance matrix. A list for multiple group models.}
\item{\code{mu}}{the population mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{SigmaHat}}{the H0 model implied covariance matrix. A list for multiple group models.}
\item{\code{muHat}}{the H0 model implied mean vector or \code{NULL} when no meanstructure is involved. A list for multiple group models.}
\item{\code{modelH0}}{\code{lavaan} H0 model string.}
\item{\code{modelH1}}{\code{lavaan} H1 model string or \code{NULL} when the comparison refers to the saturated model.}
\item{\code{simRes}}{detailed simulation results when a simulated power analysis (\code{simulatedPower = TRUE}) was performed.}
}
\description{
Perform a power analysis given \code{lavaan} model strings defining the H0 and the H1 model based on either
a \code{lavaan} model string defining the population model or the population covariance matrix Sigma and the population means mu.
This requires the \code{lavaan} package.
}
\details{
Generic function to perform a power analysis based  on a true population covariance matrix Sigma
and a model implied covariance matrix SigmaHat (and optionally the associated mean vectors),
where SigmaHat (and muHat) is determined by fitting a respective H0 model using \code{lavaan},
and Sigma (and mu) can also be provided through a corresponding \code{lavaan} model string.

All \code{semPower} convenience functions internally call this function.

Beyond the arguments explicitly contained in the function call, additional arguments
are required specifying the requested type of \strong{power analysis}:
\itemize{
\item \code{alpha}: The alpha error probability. Required for \code{type = 'a-priori'} and \code{type = 'post-hoc'}.
\item Either \code{beta} or \code{power}: The beta error probability and the statistical power (1 - beta), respectively. Only for \code{type = 'a-priori'}.
\item \code{N}: The sample size. Always required for \code{type = 'post-hoc'} and \code{type = 'compromise'}. For \code{type = 'a-priori'} and multiple group analysis, \code{N} is a list of group weights.
\item \code{abratio}: The ratio of alpha to beta. Only for \code{type = 'compromise'}.
}

If a \strong{simulated power analysis} (\code{simulatedPower = TRUE}) is requested, optional arguments can be provided as a list to \code{simOptions}:
\itemize{
\item \code{nReplications}: The targeted number of simulation runs. Defaults to 250, but larger numbers greatly improve accuracy at the expense of increased computation time.
\item \code{minConvergenceRate}:  The minimum convergence rate required, defaults to .5. The maximum actual simulation runs are increased by a factor of 1/minConvergenceRate.
\item \code{type}: specifies whether the data should be generated from a population assuming multivariate normality (\code{'normal'}; the default), or based on an approach generating non-normal data (\code{'IG'}, \code{'mnonr'}, \code{'RC'}, or \code{'VM'}).
The approaches generating non-normal data require additional arguments detailed below.
\item \code{missingVars}: vector specifying the variables containing missing data (defaults to NULL).
\item \code{missingVarProp}: can be used instead of \code{missingVars}: The proportion of variables containing missing data (defaults to zero).
\item \code{missingProp}: The proportion of missingness for variables containing missing data (defaults to zero), either a single value or a vector giving the probabilities for each variable.
\item \code{missingMechanism}: The missing data mechanism, one of \code{MCAR} (the default), \code{MAR}, or \code{NMAR}.
\item \code{nCores}: The number of cores to use for parallel processing. Defaults to 1 (= no parallel processing). This requires the \code{doSNOW} package.
}

\code{type = 'IG'} implements the independent generator approach (IG, Foldnes & Olsson, 2016) approach
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors. This requires the \code{covsim} package.

\code{type = 'mnonr'} implements the approach suggested by Qu, Liu, & Zhang (2020) and requires provision of  Mardia's multivariate skewness (\code{skewness})  and kurtosis  (\code{kurtosis}), where
skewness must be non-negative and kurtosis must be at least 1.641 skewness + p (p + 0.774), where p is the number of variables. This requires the \code{mnonr} package.

\code{type = 'RK'} implements the approach suggested by Ruscio & Kaczetow (2008) and requires provision of the population distributions
of each variable (\code{distributions}). \code{distributions} must be a list (if all variables shall be based on the same population distribution) or a list of lists.
Each component must specify the population distribution (e.g. \code{rchisq}) and additional arguments (\code{list(df = 2)}).

\code{type = 'VM'} implements the third-order polynomial method (Vale & Maurelli, 1983)
specifying third and fourth moments of the marginals, and thus requires that skewness (\code{skewness}) and excess kurtosis (\code{kurtosis}) for each variable are provided as vectors.
}
\examples{
\dontrun{
# set up two CFA factors with a true correlation of .2
mPop <- '
  f1 =~ .5*x1 + .6*x2 + .4*x3
  f2 =~ .7*x4 + .8*x5 + .3*x6
  x1 ~~ .75*x1
  x2 ~~ .64*x2
  x3 ~~ .84*x3
  x4 ~~ .51*x4
  x5 ~~ .36*x5
  x6 ~~ .91*x6
  f1 ~~ 1*f1
  f2 ~~ 1*f2
  f1 ~~ .2*f2
'
# define the H0 analysis model (restricting the factor correlation to zero) 
mH0 <- '
  f1 =~ x1 + x2 + x3
  f2 =~ x4 + x5 + x6
  f1 ~~ 0*f2
'
# determine N to reject the H0 that the correlation is zero 
# with a power of 95\% on alpha = .05
powerLav <- semPower.powerLav(type = 'a-priori', 
                              modelPop = mPop, modelH0 = mH0,
                              alpha = .05, beta = .05)
summary(powerLav)

# same as above, but also define an H1 comparison model 
mH1 <- '
  f1 =~ x1 + x2 + x3
  f2 =~ x4 + x5 + x6
  f1 ~~ f2
'
powerLav <- semPower.powerLav(type = 'a-priori', 
                              modelPop = mPop, modelH0 = mH0, modelH1 = mH1,
                              alpha = .05, beta = .05)

# same as above, but use covariance matrix input instead of modelPop
gen <- semPower.genSigma(Phi = .2, 
                         loadings = list(c(.5, .6, .4), c(.7, .8, .3)))
Sigma <- gen$Sigma
powerLav <- semPower.powerLav(type = 'a-priori', 
                              Sigma = Sigma, modelH0 = mH0,
                              alpha = .05, beta = .05)

# note all of the above is identical to the output provided by the semPower.powerCFA function
powerCFA <- semPower.powerCFA(type = 'a-priori',
                              comparison = 'saturated',
                              Phi = .2, 
                              loadings = list(c(.5, .6, .4), c(.7, .8, .3)), 
                              alpha = .05, beta = .05)

# same as above, but perform simulated power analysis
# with 250 replications using a robust ML test-statistic
set.seed(300121)
powerLav <- semPower.powerLav(type = 'a-priori', 
                              Sigma = Sigma, modelH0 = mH0,
                              alpha = .05, beta = .05, 
                              simulatedPower = TRUE,
                              simOptions = list(nReplications = 250)
                              lavOptions = list(estimator = 'MLM'))
}
}
\seealso{
\code{\link[=semPower.aPriori]{semPower.aPriori()}} \code{\link[=semPower.postHoc]{semPower.postHoc()}} \code{\link[=semPower.compromise]{semPower.compromise()}}
}
