#' List Available Sensor Tower Categories
#'
#' Returns a tibble of app categories recognized by the Sensor Tower API,
#' mapping category IDs to category names for different platforms (iOS/Android).
#' Useful for finding valid inputs for the `category` parameter in other
#' functions.
#'
#' @param platform Optional character string. Filter results for a specific
#'   platform ("ios" or "android"). If NULL (default), returns categories for
#'   both platforms.
#'
#' @return A tibble with columns `platform` (character, "ios" or "android"),
#'   `category_id` (character, e.g., "6014"), and `category_name`
#'   (character, e.g., "Games").
#' @export
#' @examples
#' # Get all categories
#' all_cats <- st_categories()
#' head(all_cats)
#'
#' # Get only iOS categories
#' ios_cats <- st_categories(platform = "ios")
#' head(ios_cats)
#'
#' # Find game categories on iOS
#' ios_games <- subset(st_categories("ios"), grepl("Game", category_name))
#' head(ios_games)
st_categories <- function(platform = NULL) { # Renamed function
  # Access the internal data object st_category_data
  data <- st_category_data

  if (!is.null(platform)) {
    platform <- tolower(platform)
    if (!platform %in% c("ios", "android")) {
      warning(
        "Invalid platform specified. Choose 'ios' or 'android'. ",
        "Returning all categories.",
        call. = FALSE
      )
    } else {
      data <- data[data$platform == platform, ]
    }
  }
  return(data)
} 