% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/seqICP.R
\name{seqICP}
\alias{seqICP}
\title{Sequential Invariant Causal Prediction}
\usage{
seqICP(X, Y, test = "decoupled", par.test = list(grid = c(0,
  round(nrow(X)/2), nrow(X)), complements = FALSE, link = sum, alpha = 0.05, B =
  100, permutation = FALSE), model = "iid", par.model = list(pknown = FALSE,
  p = 0, max.p = 10), max.parents = ncol(X), stopIfEmpty = TRUE,
  silent = TRUE)
}
\arguments{
\item{X}{matrix of predictor variables. Each column corresponds to
one predictor variable.}

\item{Y}{vector of target variable, with length(Y)=nrow(X).}

\item{test}{string specifying the hypothesis test used to test for
invariance of a parent set S (i.e. the null hypothesis
H0_S). The following tests are available: "decoupled",
"combined", "trend", "variance", "block.mean", "block.variance",
"block.decoupled", "smooth.mean", "smooth.variance",
"smooth.decoupled" and "hsic".}

\item{par.test}{parameters specifying hypothesis test. The
following parameters are available: \code{grid},
\code{complements}, \code{link}, \code{alpha}, \code{B} and
\code{permutation}. The parameter \code{grid} is an increasing
vector of gridpoints used to construct enviornments for change
point based tests. If the parameter \code{complements} is 'TRUE'
each environment is compared against its complement if it is
'FALSE' all environments are compared pairwise. The parameter
\code{link} specifies how to compare the pairwise test
statistics, generally this is either max or sum. The parameter
\code{alpha} is a numeric value in (0,1) indicting the
significance level of the hypothesis test. The parameter
\code{B} is an integer and specifies the number of Monte-Carlo
samples (or permutations) used in the approximation of the null
distribution. If the parameter \code{permutation} is 'TRUE' a
permuatation based approach is used to approximate the null
distribution, if it is 'FALSE' the scaled residuals approach is
used.}

\item{model}{string specifying the underlying model class. Either
"iid" if Y consists of independent observations or "ar" if Y has
a linear time dependence structure.}

\item{par.model}{parameters specifying model. The following
parameters are available: \code{pknown}, \code{p} and
\code{max.p}. If \code{pknown} is 'FALSE' the number of lags will be
determined by comparing all fits up to \code{max.p} lags using
the AIC criterion. If \code{pknown} is 'TRUE' the procedure will fit
\code{p} lags.}

\item{max.parents}{integer specifying the maximum size for
admissible parents. Reducing this below the number of predictor
variables saves computational time but means that the confidence
intervals lose their coverage property.}

\item{stopIfEmpty}{if ‘TRUE’, the procedure will stop computing
confidence intervals if the empty set has been accepted (and
hence no variable can have a signicificant causal
effect). Setting to ‘TRUE’ will save computational time in these
cases, but means that the confidence intervals lose their
coverage properties for values different to 0.}

\item{silent}{If 'FALSE', the procedure will output progress
notifications consisting of the currently computed set S
together with the p-value resulting from the null hypothesis
H0_S}
}
\value{
object of class 'seqICP' consisting of the following
  elements

\item{parent.set}{vector of the estimated causal parents.}

\item{test.results}{matrix containing the result from each
individual test as rows.}

\item{S}{list of all the sets that were
tested. The position within the list corresponds to the index in
the first column of the test.results matrix.}

\item{p.values}{p-value for being not included in the set of true
causal parents. (If a p-value is smaller than alpha, the
corresponding variable is a member of parent.set.)}

\item{coefficients}{vector of coefficients resulting from a
regression based on the estimated parent set.}

\item{stopIfEmpty}{a boolean value indicating whether computations
stop as soon as intersection of accepted sets is empty.}

\item{modelReject}{a boolean value indicating if the whole model was
rejected (the p-value of the best fitting model is too low).}

\item{pknown}{a boolean value indicating whether the number of
lags in the model was known. Only relevant if model was set to
"ar".}

\item{alpha}{significance level at which the hypothesis tests were performed.}

\item{n.var}{number of predictor variables.}

\item{model}{either "iid" or "ar" depending on which model was selected.}
}
\description{
Estimates the causal parents S of the target variable Y using invariant causal prediction and fits a linear model of the form \cr Y = a X^S + N.
}
\details{
The function can be applied to two types of models\cr
(1) a linear model (model="iid")\cr Y_i = a X_i^S + N_i\cr with iid noise N_i and \cr
(2) a linear autoregressive model (model="ar")\cr Y_t = a_0 X_t^S + ... + a_p (Y_(t-p),X_(t-p)) + N_t\cr with iid noise N_t.

For both models the invariant prediction procedure is applied
using the hypothesis test specified by the \code{test} parameter
to determine whether a candidate model is invariant. For further
details see the references.
}
\examples{
set.seed(1)

# environment 1
na <- 140
X1a <- 0.3*rnorm(na)
X3a <- X1a + 0.2*rnorm(na)
Ya <- -.7*X1a + .6*X3a + 0.1*rnorm(na)
X2a <- -0.5*Ya + 0.5*X3a + 0.1*rnorm(na)

# environment 2
nb <- 80
X1b <- 0.3*rnorm(nb)
X3b <- 0.5*rnorm(nb)
Yb <- -.7*X1b + .6*X3b + 0.1*rnorm(nb)
X2b <- -0.5*Yb + 0.5*X3b + 0.1*rnorm(nb)

# combine environments
X1 <- c(X1a,X1b)
X2 <- c(X2a,X2b)
X3 <- c(X3a,X3b)
Y <- c(Ya,Yb)
Xmatrix <- cbind(X1, X2, X3)

# Y follows the same structural assignment in both environments
# a and b (cf. the lines Ya <- ... and Yb <- ...).
# The direct causes of Y are X1 and X3.
# A linear model considers X1, X2 and X3 as significant.
# All these variables are helpful for the prediction of Y.
summary(lm(Y~Xmatrix))

# apply seqICP to the same setting
seqICP.result <- seqICP(X = Xmatrix, Y,
par.test = list(grid = seq(0, na + nb, (na + nb)/10), complements = FALSE, link = sum,
alpha = 0.05, B =100), max.parents = 4, stopIfEmpty=FALSE, silent=FALSE)
summary(seqICP.result)
# seqICP is able to infer that X1 and X3 are causes of Y
}
\references{
Pfister, N., P. Bühlmann and J. Peters (2017).
Invariant Causal Prediction for Sequential Data. ArXiv e-prints (1706.08058).

Peters, J., P. Bühlmann, and N. Meinshausen (2016).
Causal inference using invariant prediction: identification and confidence intervals.
Journal of the Royal Statistical Society, Series B (with discussion) 78 (5), 947–1012.
}
\seealso{
The function \code{\link{seqICP.s}} allows to perform
  hypothesis test for individual sets S. For non-linear
  models the functions \code{\link{seqICPnl}} and
  \code{\link{seqICPnl.s}} can be used.
}
\author{
Niklas Pfister and Jonas Peters
}
