\name{simeventhistories}
\alias{simeventhistories}
\title{Simulate Event Histories}
\description{Simulates n individual event histories.}
\usage{
simeventhistories(n, mpl, max.time, change.times, X, states.at.origin = NULL, 
Xstruc, partial.markov.x = NULL, partial.markov.eta = NULL)
}
\arguments{
  \item{n}{number of individuals.}
  \item{mpl}{model parameter list as generated (only a skeleton that has 
  to be suitably completed) by the function \code{\link{mplskeleton}} 
  (see examples below).}
  \item{max.time}{maximum entry time.}
  \item{change.times}{vector giving the times of change of the time-change 
  covariates.}
  \item{X}{design matrix.}
  \item{states.at.origin}{state-types at origin (default is all possible 
  entry state-types, which is internally calculated).}
  \item{Xstruc}{X structure matrix. See Examples for more information.}
  \item{partial.markov.x}{function defining how the partial Markov covariates 
  are generated (see example below).}
  \item{partial.markov.eta}{list of lists (as generated by the function 
  \code{\link{pmeskeleton}} in close analogy to \code{mpl}) defining how 
  the partial Markov linear predictors are generated (see example below).}
}
\details{The example below provides an intuitive description of how to 
use the different input arguments. 
The idea of partial Markov covariates is based on the definition in 
Commenges (1991). 
A description of this idea directly in the context of illness-death models 
is described on pp. 224-225 in Beyersmann et al. (1999).}
\value{Three data frames named \code{msm.bascis}, \code{ttsce}, 
\code{tt.indicators} are returned organized within one list. 
The three data frames and their respective variables will be described 
in the next lines.

\code{msm.bascis} contains the following variables variables:
	\item{id}{id (1, ..., n) of the individual}
	\item{entry}{entry times}
	\item{exit}{exit times}
	\item{from}{values of initial states}
	\item{to}{values of final states}
	\item{delta}{non-censoring indicator function}
	\item{x1}{values of first covariate (additional covariates follow). 
	If partial Markov objects are supplied, the generated covariates are 
	attached as additional variables.}
	
The second data frame \code{ttsce} contains a transition-type specific 
covariate expansion (as well for partial Markov covariates in the case 
of a partial Markov set-up).

The third data frame \code{tt.indicators} contains the values of 
transition-type indicator functions. For censored observations, all 
values of one data line are equal to zero (as e.g. needed in a BayesX 
full likelihood analysis).}
\author{Holger Reulen}
\references{Daniel Commenges (1991)
Multi-state Models in Epidemiology. 
Lifetime Data Analysis, Vol. 5, No. 4.

Jan Beyersmann, Martin Schumacher, Arthur Allignol (2012)
Competing Risks and Multistate Models with R. 
Springer Series 'UseR!'.}
\seealso{\code{\link{mplskeleton}}}
\examples{
## An example for a time-varying setup without partial Markov effects:
tra2 <- matrix(ncol = 2, nrow = 2, data = FALSE)
tra2[1, 2] <- tra2[2, 1] <- TRUE
mpl <- mplskeleton(tmat = tra2)
mpl[[1]]$bhr[[2]] <- mpl[[2]]$bhr[[1]] <- function(t){return(0.5)}
mpl[[1]]$eta[[2]] <- function(x.i, t){ ## time-varying x2 and time-varying f(x2)
  ifelse(t < 5, 
         return(1.0 * x.i[1] + 0.5 * x.i[2]), 
         return(1.0 * x.i[1] + 1.0 * x.i[3]))}
mpl[[2]]$eta[[1]] <- function(x.i, t){ ## time-varying x2 and time-varying f(x1)
  ifelse(t < 5, 
         return(-0.5 * x.i[1] + 0.5 * x.i[2]), 
         return( 1.0 * x.i[1] + 0.5 * x.i[3]))}
set.seed(123)
N <- 2
X <- matrix(nrow = N, ncol = 2, rnorm(2 * N))
X <- cbind(X, X[, 2] + runif(N)/10)
colnames(X) <- c("x1", "x2.t1", "x2.t2")
Xstruc <- matrix(ncol = 2, nrow = 2, data = 0)
rownames(Xstruc) <- c("t1", "t2")
colnames(Xstruc) <- c("x1", "x2")
Xstruc[, 1] <- 1
Xstruc[, 2] <- c(2, 3)
d <- simeventhistories(n = N, mpl = mpl, X = X, max.time = 10, 
                       change.times = c(5), Xstruc = Xstruc)
head(d$msm.basics)
\dontrun{
## An Illness-Death model example with time-varying setup and partial Markov 
## effects:
traIDM <- matrix(nrow = 3, ncol = 3, FALSE)
traIDM[1, 2] <- traIDM[1, 3] <- traIDM[2, 1] <- traIDM[2, 3] <- TRUE
mpl <- mplskeleton(tmat = traIDM)
mpl[[1]]$bhr[[2]] <- mpl[[1]]$bhr[[3]] <- mpl[[2]]$bhr[[1]] <- 
                     mpl[[2]]$bhr[[3]] <- function(t){0.25}
mpl[[1]]$eta[[2]] <- mpl[[1]]$eta[[3]] <- mpl[[2]]$eta[[1]] <- 
  mpl[[2]]$eta[[3]] <- function(x.i, t){
    ifelse(t < 5, 
      return(0.5 * x.i[1]), 
      return(0.5 * x.i[2]))}
set.seed(123)
N <- 500
X <- matrix(nrow = N, ncol = 1, rnorm(N))
X <- cbind(X, X[, 1] + rnorm(N)/10)
colnames(X) <- c("x1.t1", "x1.t2")
Xstruc <- matrix(ncol = 1, nrow = 2, data = 0)
rownames(Xstruc) <- c("t1", "t2")
colnames(Xstruc) <- c("x1")
Xstruc[, 1] <- c(1, 2)
Xstruc
## Now set-up the partial Markov influences:
## Function 'partial.markov.x' has to take 5 input arguments representig vectors 
## of past history information. They have to take names 'entry', 'exit', 'from', 
## 'to', and 'delta':
partial.markov.x <- function(entry, exit, from, to, delta){
  count.12 <- sum(as.numeric((from == 1) & (to == 2) & (delta == 1)))
  count.21 <- sum(as.numeric((from == 2) & (to == 1) & (delta == 1)))
  return(c(count.12, count.21))}
## List 'partial.markov.eta' is a list of lists in analogy to 'mpl':
partial.markov.eta <- pmeskeleton(traIDM)
partial.markov.eta[[1]][[2]] <- function(x){return( 0.25 * x[1])}
partial.markov.eta[[1]][[3]] <- function(x){return( 0.50 * x[1])}
partial.markov.eta[[2]][[1]] <- function(x){return(-0.50 * x[1] + 0.25 * x[2])}
partial.markov.eta[[2]][[3]] <- function(x){return(0)}
## Event history simulation:
d <- simeventhistories(n = N, mpl = mpl, X = X, max.time = 10, 
                       change.times = c(5), Xstruc = Xstruc, 
                       partial.markov.x = partial.markov.x, 
                       partial.markov.eta = partial.markov.eta)
}
}