#' Calculate the Evaporative Fraction (EF)
#'
#' This function calculates the EF from two rasters: albedo
#' and the surface-air temperature difference (TS-TA). It saves the resulting
#' EF raster to the specified output path.
#'
#' @param albedo_path Character. File path to the albedo raster.Must have the same
#'        CRS and extent as the TS-TA raster.
#' @param TS_TA_path Character. File path to the raster of TS-TA (surface-air temperature difference).
#'        TS and TA must have the same unit of measurement (Kelvin preferably).
#' @details
#' - The input rasters (`albedo` and `TS-TA`) must have the same CRS (Coordinate Reference System) and extent.
#' - If they differ, the function will attempt to reproject and resample the rasters automatically.
#' @param output_path Character. File path where the EF raster will be saved.
#' @param n_intervals Integer. Number of intervals for splitting albedo values
#'        (default: 20).
#' @param percentile Numeric. Percentage used for identifying wet and dry edges
#'        (default: 0.01).
#'
#' @return A raster object representing the Evaporative Fraction (EF).
#'
#' @importFrom terra rast resample values varnames writeRaster
#' @importFrom dplyr "%>%"
#' @importFrom stats lm coef na.omit weighted.mean
#' @importFrom stats na.exclude
#'
#'
#' @examples
#' # Paths to example data included in the package
#' library(terra)
#'
#'albedo_raster <- rast(system.file("extdata", "albedo_reduced.tif", package = "ssebiEF"))
#'TS_TA_raster <- rast(system.file("extdata", "TS_TA_reduced.tif", package = "ssebiEF"))
#'
#' # Output path (temporary file for example purposes)
#' output_path <- tempfile(fileext = ".tif")
#'
#' # Run the function
#' calculate_EF(
#'   albedo_path = albedo_raster,
#'   TS_TA_path = TS_TA_raster,
#'   output_path = output_path,
#'   n_intervals = 20,
#'   percentile = 0.01
#' )
#'
#' # Print the output path
#' print(output_path)
#'
#' @export
calculate_EF <- function(albedo_path, TS_TA_path, output_path, n_intervals = 20, percentile = 0.01) {

  # Step 1: Load rasters if file paths are provided
  if (inherits(albedo_path, "character") && file.exists(albedo_path)) {
    albedo <- rast(albedo_path)
  } else if (inherits(albedo_path, "SpatRaster")) {
    albedo <- albedo_path
  } else {
    stop("Invalid albedo input.")
  }

  if (inherits(TS_TA_path, "character") && file.exists(TS_TA_path)) {
    TS_TA <- rast(TS_TA_path)
  } else if (inherits(TS_TA_path, "SpatRaster")) {
    TS_TA <- TS_TA_path
  } else {
    stop("Invalid TS_TA input.")
  }

  # Step 2: Align CRS and extent silently
  if (!terra::same.crs(albedo, TS_TA)) {
    TS_TA <- terra::project(TS_TA, terra::crs(albedo))
  }
  if (!terra::ext(albedo) == terra::ext(TS_TA)) {
    TS_TA <- terra::resample(TS_TA, albedo)
  }

  # Step 3: Clip albedo to [0,1] and normalize silently if maximum < 1
  albedo[albedo > 1] <- 1
  albedo[albedo < 0] <- 0
  max_albedo <- max(terra::values(albedo), na.rm = TRUE)
  if (max_albedo < 1) {
    albedo <- albedo / max_albedo
  }

  # Step 4: Convert rasters to vectors for processing
  values_albedo <- terra::values(albedo)
  values_TS_TA <- terra::values(TS_TA)

  # Step 5: Cut albedo range into intervals
  cut_albedo <- cut(values_albedo, breaks = n_intervals)
  R1 <- data.frame()

  for (i in 1:n_intervals) {
    idx <- which(cut_albedo == levels(cut_albedo)[i])
    ts_values <- values_TS_TA[idx]
    ts_values <- ts_values[!is.na(ts_values)]

    if (length(ts_values) > 0) {
      ts_sorted <- sort(ts_values)
      n <- length(ts_sorted)
      low_vals <- ts_sorted[1:floor(percentile * n)]
      high_vals <- ts_sorted[ceiling((1 - percentile) * n):n]

      R1[i, "Interval"] <- i
      R1[i, "Ts_wet"] <- mean(low_vals, na.rm = TRUE)
      R1[i, "Ts_dry"] <- mean(high_vals, na.rm = TRUE)
    } else {
      R1[i, "Interval"] <- i
      R1[i, "Ts_wet"] <- NA
      R1[i, "Ts_dry"] <- NA
    }
  }

  # Step 6: Assign albedo midpoints
  R1$albedo <- seq(0.5 / n_intervals, 1 - 0.5 / n_intervals, length.out = n_intervals)

  # Step 7: Remove first two intervals (bare soil)
  R1[1:2, c("Ts_wet", "Ts_dry")] <- NA

  # Step 8: Fit regression models
  reg_wet <- lm(Ts_wet ~ albedo, data = R1, na.action = na.exclude)
  reg_dry <- lm(Ts_dry ~ albedo, data = R1, na.action = na.exclude)

  a_wet <- coef(reg_wet)[2]; b_wet <- coef(reg_wet)[1]
  a_dry <- coef(reg_dry)[2]; b_dry <- coef(reg_dry)[1]

  # Step 9: Compute EF
  EF <- (((a_dry * albedo) + b_dry) - TS_TA) /
    (((a_dry * albedo) + b_dry) - ((a_wet * albedo) + b_wet))

  EF[EF > 1] <- 1
  EF[EF < 0] <- 0
  names(EF) <- "EF"
  terra::varnames(EF) <- "EF"

  # Step 10: Save and plot
  writeRaster(EF, output_path, overwrite = TRUE)
  terra::plot(EF, main = "Evaporative Fraction (EF)")

  return(EF)
}
