% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/counterFactuals.R, R/plotMethods.R,
%   R/printMethods.R
\name{cfact_hist}
\alias{cfact_hist}
\alias{plot.cfacthist}
\alias{print.cfacthist}
\title{Simulate historical counterfactual for structural STVAR models.}
\usage{
cfact_hist(
  stvar,
  cfact_type = c("fixed_path", "muted_response"),
  policy_var = 1,
  mute_var = NULL,
  cfact_start = 1,
  cfact_end = 1,
  cfact_path = NULL
)

\method{plot}{cfacthist}(x, ...)

\method{print}{cfacthist}(x, ..., digits = 3)
}
\arguments{
\item{stvar}{an object of class \code{'stvar'} defining a structural or reduced form
STVAR model. For a reduced form model (that is not readily identified statiscally),
the shocks are automatically identified by the lower triangular Cholesky decomposition.}

\item{cfact_type}{a character string indicating the type of counterfactual to be computed: should the path of the policy
variable be fixed to some hypothetical path (\code{cfact_type="fixed_path"}) in given points of time or should the responses
of the policy variable to lagged and contemporaneous movements of some given variable be muted (\code{cfact_type="muted_response"})?
See details for more information.}

\item{policy_var}{a positive integer between \eqn{1} and \eqn{d} indicating the index of the policy variable considered in the
counterfactual scenario.}

\item{mute_var}{a positive integer between \eqn{1} and \eqn{d} indicating the index of the variable to whose lagged movements the policy variable
specified in the argument \code{policy_var} should not react to in the counterfactual scenario. This indicates also the index of the shock
to which the policy variable should not react to. It is assumed that \code{mute_var != policy_var}. This argument is only used when
\code{cfact_type="muted_response"}.}

\item{cfact_start}{a positive integer between \eqn{1} and \eqn{T} indicating the starting period for the counterfactual behavior
of the specified policy variable. It is assumed that the observed time series in indexed as \eqn{y_{t-p+1},...,y_{0},y_1,...,y_T},
i.e., that the first \eqn{p} observations are the initial values, and the "time period one" observation is the \eqn{p+1}th row in
the data matrix.}

\item{cfact_end}{a positive integer between \code{cfact_start} and \eqn{T} indicating the ending period for the counterfactual
behavior of the specified policy variable.}

\item{cfact_path}{a numeric vector of length \code{cfact_end-cfact_start+1} indicating the hypothetical path of the policy variable
specified in the argument \code{policy_var}. This argument is only used when \code{cfact_type="fixed_path"}.}

\item{x}{object of class \code{'cfacthist'} created by the function \code{cfact_hist}.}

\item{...}{arguments passed to the function \code{window} to select the time periods to print.}

\item{digits}{how many significant digits to print?}
}
\value{
Returns a class \code{'cfacthist'} list with the following elements:
  \describe{
    \item{cfact_data}{A matrix of size \eqn{(T+p \times d)} containing the counterfactual time series. Note that the first \eqn{p} rows
     are for the initial values prior the time period \eqn{t=1}.}
    \item{cfact_shocks}{A matrix of size \eqn{(T \times d)} containing the counterfactual shocks.}
    \item{cfact_weights}{A matrix of size \eqn{(T \times M)} containing the counterfactual transition weights.}
    \item{stvar}{The original STVAR model object.}
    \item{input}{A list containing the arguments used to calculate the counterfactual.}
 }

Returns the input object \code{x} invisibly.
}
\description{
\code{cfact_hist} simulates historical counterfactual for structural STVAR models.
}
\details{
Two types of historical counterfactuals are accommodated where in given historical points of time
 either (1) the policy variable of interest takes some hypothetical path (\code{cfact_type="fixed_path"}), or (2)
 its responses to lagged movements of some given variable and to some given shock are shut off (\code{cfact_type="muted_response"}).
 In both cases, the counterfactual scenarios are simulated by creating hypothetical shocks to the policy variable of interest
 that yield the counterfactual outcome. This approach has the appealing feature that the counterfactual deviations from the
 policy reaction function are treated as policy surprises, allowing them to propagate normally, so that the dynamics of the model
 are not, per se, tampered but just the policy surprises are.

 \strong{Important:} This function assumes that when the policy variable of interest is the \eqn{i_1}th variable, the shock
 related to it that is manipulated is the \eqn{i_1}th shock. This should be automatically satisfied for recursively identified models,
 whereas for model identified by heteroskedasticity or non-Gaussianity, the ordering of the shocks can be generally changed
 without loss of generality with the function \code{reorder_B_columns}. In Type (2) counterfactuals it is additionally assumed
 that, if the variable to whose lagged movements the policy variable should not react to is the \eqn{i_2}th variable, the shock to which
 the policy variable does not reach to  is the \eqn{i_2}th shock.

 See the seminal paper of Bernanke et al (1997) for discussing about the "Type (1)" counterfactuals and
 Kilian and Lewis (2011) for discussion about the counterfactuals closely related to "Type (2)" counterfactuals.
 See Kilian and Lütkepohl (2017), Section 4.5 for further discussion about the historical counterfactuals.
 The literature cited about considers linear models, but it is explained in the vignette of this package how this
 function computes the historical counterfactuals for the STVAR models in a way that accommodates nonlinear time-varying dynamics.
}
\section{Functions}{
\itemize{
\item \code{plot(cfacthist)}: plot method

\item \code{print(cfacthist)}: print method

}}
\examples{
# Recursively identified logistic Student's t STVAR(p=3, M=2) model with the first
# lag of the second variable as the switching variable:
params32logt <- c(0.5959, 0.0447, 2.6279, 0.2897, 0.2837, 0.0504, -0.2188, 0.4008,
  0.3128, 0.0271, -0.1194, 0.1559, -0.0972, 0.0082, -0.1118, 0.2391, 0.164, -0.0363,
  -1.073, 0.6759, 3e-04, 0.0069, 0.4271, 0.0533, -0.0498, 0.0355, -0.4686, 0.0812,
  0.3368, 0.0035, 0.0325, 1.2289, -0.047, 0.1666, 1.2067, 7.2392, 11.6091)
mod32logt <- STVAR(gdpdef, p=3, M=2, params=params32logt, weight_function="logistic",
  weightfun_pars=c(2, 1), cond_dist="Student", identification="recursive")

# Simulate historical counterfactual where the first variable takes the values 5 and -5
# in the first and second time periods, respectively.
cfact1 <- cfact_hist(mod32logt, cfact_type="fixed_path", policy_var=1, cfact_start=1,
  cfact_end=2, cfact_path=c(5, -5))
print(cfact1, start=c(1959, 1), end=c(1960, 4)) # Print cfact data from 1959Q1 to 1960Q4
plot(cfact1) # Plot the observed and counterfactual data

# Simulate historical counterfactual where the first variable does not respond to lagged
# movements of the second variable nor to the second shock in time periods from 10 to 100.
cfact2 <- cfact_hist(mod32logt, cfact_type="muted_response", policy_var=1, mute_var=2,
 cfact_start=10, cfact_end=100)
print(cfact2, start=c(1960, 4), end=c(1963, 4)) # Print cfact data from 1960Q4 to 1963Q4
plot(cfact2) # plot the observed and counterfactual data
}
\references{
\itemize{
   \item Bernanke B., Gertler M., Watson M. 1997. Systematic monetary policy and the effects of oilprice shocks.
     \emph{Brookings Papers on Economic Activity}, 1, 91—142.
   \item Kilian L., Lewis L. 2011. Does the fed respond to oil price shocks? \emph{The Economic Journal}, \strong{121}:555.
   \item Kilian L., Lütkepohl H. 2017. Structural Vector Autoregressive Analysis. 1st edition.
     \emph{Cambridge University Press}, Cambridge.
 }
}
\seealso{
\code{\link{GIRF}}, \code{\link{GFEVD}}, \code{\link{linear_IRF}}, \code{\link{hist_decomp}}, \code{\link{cfact_fore}},
 \code{\link{cfact_girf}}, \code{\link{fitSSTVAR}}
}
