% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/geom-spatraster.R, R/stat-spatraster.R
\name{geom_spatraster}
\alias{geom_spatraster}
\alias{stat_spatraster}
\title{Visualise \code{SpatRaster} objects}
\source{
Based on the \code{layer_spatial()} implementation on \CRANpkg{ggspatial} package.
Thanks to \href{https://github.com/paleolimbot}{Dewey Dunnington} and \href{https://github.com/paleolimbot/ggspatial/graphs/contributors}{ggspatial contributors}.
}
\usage{
geom_spatraster(
  mapping = aes(),
  data,
  na.rm = TRUE,
  show.legend = NA,
  inherit.aes = FALSE,
  interpolate = FALSE,
  maxcell = 5e+05,
  use_coltab = TRUE,
  mask_projection = FALSE,
  ...
)

stat_spatraster(
  mapping = aes(),
  data,
  geom = "raster",
  na.rm = TRUE,
  show.legend = NA,
  inherit.aes = FALSE,
  maxcell = 5e+05,
  ...
)
}
\arguments{
\item{mapping}{Set of aesthetic mappings created by \code{\link[ggplot2:aes]{ggplot2::aes()}}. See
\strong{Aesthetics} specially in the use of \code{fill} aesthetic.}

\item{data}{A \code{SpatRaster} object.}

\item{na.rm}{If \code{TRUE}, the default, missing values are silently removed. If
\code{FALSE}, missing values are removed with a warning.}

\item{show.legend}{logical. Should this layer be included in the legends?
\code{NA}, the default, includes if any aesthetics are mapped.
\code{FALSE} never includes, and \code{TRUE} always includes.
It can also be a named logical vector to finely select the aesthetics to
display. To include legend keys for all levels, even
when no data exists, use \code{TRUE}.  If \code{NA}, all levels are shown in legend,
but unobserved levels are omitted.}

\item{inherit.aes}{If \code{FALSE}, overrides the default aesthetics, rather
than combining with them.}

\item{interpolate}{If \code{TRUE} interpolate linearly, if \code{FALSE}
(the default) don't interpolate.}

\item{maxcell}{positive integer. Maximum number of cells to use for
the plot.}

\item{use_coltab}{Logical. Only applicable to \code{SpatRaster} objects that have
an associated \link[terra:colors]{coltab}. Should the coltab be used on the
plot? See also \code{\link[=scale_fill_coltab]{scale_fill_coltab()}}.}

\item{mask_projection}{logical, defaults to \code{FALSE}. If \code{TRUE}, mask out
areas outside the input extent. For example, to avoid data wrapping
around the date-line in Equal Area projections. This argument is passed
to \code{\link[terra:project]{terra::project()}} when reprojecting the \code{SpatRaster}.}

\item{...}{Other arguments passed on to \code{\link[ggplot2:layer]{layer()}}'s \code{params} argument. These
arguments broadly fall into one of 4 categories below. Notably, further
arguments to the \code{position} argument, or aesthetics that are required
can \emph{not} be passed through \code{...}. Unknown arguments that are not part
of the 4 categories below are ignored.
\itemize{
\item Static aesthetics that are not mapped to a scale, but are at a fixed
value and apply to the layer as a whole. For example, \code{colour = "red"}
or \code{linewidth = 3}. The geom's documentation has an \strong{Aesthetics}
section that lists the available options. The 'required' aesthetics
cannot be passed on to the \code{params}. Please note that while passing
unmapped aesthetics as vectors is technically possible, the order and
required length is not guaranteed to be parallel to the input data.
\item When constructing a layer using
a \verb{stat_*()} function, the \code{...} argument can be used to pass on
parameters to the \code{geom} part of the layer. An example of this is
\code{stat_density(geom = "area", outline.type = "both")}. The geom's
documentation lists which parameters it can accept.
\item Inversely, when constructing a layer using a
\verb{geom_*()} function, the \code{...} argument can be used to pass on parameters
to the \code{stat} part of the layer. An example of this is
\code{geom_area(stat = "density", adjust = 0.5)}. The stat's documentation
lists which parameters it can accept.
\item The \code{key_glyph} argument of \code{\link[ggplot2:layer]{layer()}} may also be passed on through
\code{...}. This can be one of the functions described as
\link[ggplot2:draw_key]{key glyphs}, to change the display of the layer in the legend.
}}

\item{geom}{The geometric object to use display the data. Recommended \code{geom}
for \code{SpatRaster} are \code{"raster"} (the default), \code{"point"},\code{"text"} and
\code{"label"}.}
}
\value{
A \CRANpkg{ggplot2} layer
}
\description{
This geom is used to visualise \code{SpatRaster} objects (see \code{\link[terra:rast]{terra::rast()}}).
The geom is designed for visualise the object by layers, as \code{\link[terra:plot]{terra::plot()}}
does.

For plotting \code{SpatRaster} objects as map tiles (i.e. RGB \code{SpatRaster}), use
\code{\link[=geom_spatraster_rgb]{geom_spatraster_rgb()}}.

The underlying implementation is based on \code{\link[ggplot2:geom_tile]{ggplot2::geom_raster()}}.

\code{stat_spatraster()} is provided as a complementary function, so the \code{geom}
can be modified.
}
\section{\CRANpkg{terra} equivalent}{


\code{\link[terra:plot]{terra::plot()}}
}

\section{Coords}{


When the \code{SpatRaster} does not present a crs (i.e.,
\code{terra::crs(rast) == ""}) the geom does not make any assumption on the
scales.

On \code{SpatRaster} that have a crs, the geom uses \code{\link[ggplot2:ggsf]{ggplot2::coord_sf()}} to
adjust the scales. That means that also the
\strong{\code{SpatRaster} may be reprojected}.
}

\section{Aesthetics}{


\code{geom_spatraster()} understands the following aesthetics:
\itemize{
\item \code{\link[ggplot2:aes_colour_fill_alpha]{fill}}
\item \code{\link[ggplot2:aes_colour_fill_alpha]{alpha}}
}

If \code{fill} is not provided, \code{geom_spatraster()} creates a \CRANpkg{ggplot2}
layer with all the layers of the \code{SpatRaster} object. Use \code{facet_wrap(~lyr)}
to display properly the \code{SpatRaster} layers.

If \code{fill} is used, it should contain the name of one layer that is present
on the \code{SpatRaster} (i.e.
\verb{geom_spatraster(data = rast, aes(fill = <name_of_lyr>)}). Names of the
layers can be retrieved using \code{names(rast)}.

Using \code{geom_spatraster(..., mapping = aes(fill = NULL))} or
\verb{geom_spatraster(..., fill = <color value(s)>)} would create a layer with no
mapped \code{fill} aesthetic.

\code{fill} can use computed variables.

For \code{alpha} use computed variable. See section \strong{Computed variables}.


\subsection{\code{stat_spatraster()}}{

\code{stat_spatraster()} understands the same aesthetics than \code{geom_spatraster()}
when using \code{geom = "raster"} (the default):
\itemize{
\item \code{\link[ggplot2:aes_colour_fill_alpha]{fill}}
\item \code{\link[ggplot2:aes_colour_fill_alpha]{alpha}}
}

When \code{geom = "raster"} the \code{fill} argument would behave as in
\code{geom_spatraster()}. If another \code{geom} is used \code{stat_spatraster()} would
understand the aesthetics of the required \code{geom} and
\verb{aes(fill = <name_of_lyr>)} would not be applicable.

Note also that mapping of aesthetics \code{x} and \code{y} is provided by default,
so the user does not need to add those aesthetics on \code{aes()}. In all the
cases the aesthetics should be mapped by using computed variables. See
section \strong{Computed variables} and \strong{Examples}.
}
}

\section{Facets}{


You can use \code{ facet_wrap(~lyr)} for creating a faceted plot by each layer of
the \code{SpatRaster} object. See \code{\link[ggplot2:facet_wrap]{ggplot2::facet_wrap()}} for details.
}

\section{Computed variables}{


This geom computes internally some variables that are available for use as
aesthetics, using (for example) \code{aes(alpha = after_stat(value))} (see
\code{\link[ggplot2:aes_eval]{ggplot2::after_stat()}}).
\itemize{
\item \code{after_stat(value)}: Values of the \code{SpatRaster.}
\item \code{after_stat(lyr)}: Name of the layer.
}
}

\examples{
\donttest{
# Avg temperature on spring in Castille and Leon (Spain)
file_path <- system.file("extdata/cyl_temp.tif", package = "tidyterra")

library(terra)
temp_rast <- rast(file_path)

library(ggplot2)

# Display a single layer
names(temp_rast)

ggplot() +
  geom_spatraster(data = temp_rast, aes(fill = tavg_04)) +
  # You can use coord_sf
  coord_sf(crs = 3857) +
  scale_fill_grass_c(palette = "celsius")

# Display facets
ggplot() +
  geom_spatraster(data = temp_rast) +
  facet_wrap(~lyr, ncol = 2) +
  scale_fill_grass_b(palette = "celsius", breaks = seq(0, 20, 2.5))

# Non spatial rasters

no_crs <- rast(crs = NA, extent = c(0, 100, 0, 100), nlyr = 1)
values(no_crs) <- seq_len(ncell(no_crs))


ggplot() +
  geom_spatraster(data = no_crs)

# Downsample

ggplot() +
  geom_spatraster(data = no_crs, maxcell = 25)
}

\donttest{
# Using stat_spatraster
# Default
ggplot() +
  stat_spatraster(data = temp_rast) +
  facet_wrap(~lyr)

# Using points
ggplot() +
  stat_spatraster(
    data = temp_rast,
    aes(color = after_stat(value)),
    geom = "point", maxcell = 250
  ) +
  scale_colour_viridis_c(na.value = "transparent") +
  facet_wrap(~lyr)

# Using points and labels

r_single <- temp_rast |> select(1)

ggplot() +
  stat_spatraster(
    data = r_single,
    aes(color = after_stat(value)),
    geom = "point",
    maxcell = 2000
  ) +
  stat_spatraster(
    data = r_single,
    aes(label = after_stat(round(value, 2))),
    geom = "label",
    alpha = 0.85,
    maxcell = 20
  ) +
  scale_colour_viridis_c(na.value = "transparent")
}
}
\seealso{
\code{\link[ggplot2:geom_tile]{ggplot2::geom_raster()}}, \code{\link[ggplot2:ggsf]{ggplot2::coord_sf()}},
\code{\link[ggplot2:facet_wrap]{ggplot2::facet_wrap()}}

Recommended \code{geoms}:
\itemize{
\item \code{\link[ggplot2:geom_point]{ggplot2::geom_point()}}.
\item \code{\link[ggplot2:geom_text]{ggplot2::geom_label()}}.
\item \code{\link[ggplot2:geom_text]{ggplot2::geom_text()}}.
}

Other \CRANpkg{ggplot2} utils:
\code{\link{autoplot.Spat}},
\code{\link{fortify.Spat}},
\code{\link{geom_spat_contour}},
\code{\link{geom_spatraster_rgb}()},
\code{\link{ggspatvector}},
\code{\link{stat_spat_coordinates}()}
}
\concept{ggplot2.utils}
