% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/HACLasso.r
\name{rlassoLoad}
\alias{rlassoLoad}
\title{\code{rlassoLoad} performs Lasso estimation under heteroscedastic and autocorrelated non-Gaussian disturbances
                  with predefined penalty loadings.}
\source{
Victor Chernozhukov, Chris Hansen, Martin Spindler (2016). hdm: High-Dimensional Metrics,
R Journal, 8(2), 185-199. URL https://journal.r-project.org/archive/2016/RJ-2016-040/index.html.
}
\usage{
rlassoLoad(
  x,
  y,
  load,
  bns = 10,
  lns = NULL,
  nboot = 5000,
  post = TRUE,
  intercept = TRUE,
  model = TRUE,
  X.dependent.lambda = FALSE,
  c = 2,
  gamma = NULL,
  numIter = 15,
  tol = 10^-5,
  threshold = NULL,
  ...
)
}
\arguments{
\item{x}{Regressors (vector, matrix or object can be coerced to matrix).}

\item{y}{Dependent variable (vector, matrix or object can be coerced to matrix).}

\item{load}{Penalty loadings, vector of length p (no. of regressors).}

\item{bns}{Block length with default bns=10.}

\item{lns}{Number of blocks with default lns = floor(T/bns).}

\item{nboot}{Number of bootstrap iterations with default nboot=5000.}

\item{post}{Logical. If TRUE (default), post-Lasso estimation is conducted, i.e. a refit of the model with the selected variables.}

\item{intercept}{Logical. If TRUE, intercept is included which is not penalized.}

\item{model}{Logical. If TRUE (default), model matrix is returned.}

\item{X.dependent.lambda}{Logical, TRUE, if the penalization parameter depends on the design
of the matrix x. FALSE (default), if independent of the design matrix.}

\item{c}{Constant for the penalty default is 2.}

\item{gamma}{Constant for the penalty default gamma=0.1/log(T) with T=data length.}

\item{numIter}{Number of iterations for the algorithm for the estimation of the variance and data-driven penalty.}

\item{tol}{Constant tolerance for improvement of the estimated variances.}

\item{threshold}{Constant applied to the final estimated lasso coefficients. Absolute values 
below the threshold are set to zero.}

\item{...}{further parameters}
}
\value{
rlassoLoad returns an object of class "rlasso". An object of class "rlasso" is a list containing at least the
following components:
\item{coefficients}{Parameter estimates.}
\item{beta}{Parameter estimates (named vector of coefficients without intercept). }
\item{intercept}{Value of the intercept. }
\item{index}{Index of selected variables (logical vector). }
\item{lambda}{Data-driven penalty term for each variable, product of lambda0 (the penalization parameter) and the loadings. }
\item{lambda0}{Penalty term. }
\item{loadings}{Penalty loadings, vector of lenght p (no. of regressors). }
\item{residuals}{Residuals, response minus fitted values. }
\item{sigma}{Root of the variance of the residuals. }
\item{iter}{Number of iterations. }
\item{call}{Function call. }
\item{options}{Options. }
\item{model}{Model matrix (if model = TRUE in function call). }
}
\description{
\code{rlassoLoad} performs Lasso estimation under heteroscedastic and autocorrelated non-Gaussian disturbances
                  with predefined penalty loadings.
}
\examples{
\donttest{  
set.seed(1)
T = 100 #sample size
p = 20 # number of variables
b = 5 # number of variables with non-zero coefficients
beta0 = c(rep(10,b), rep(0,p-b))
rho = 0.1 #AR parameter
Cov = matrix(0,p,p)
for(i in 1:p){
  for(j in 1:p){
     Cov[i,j] = 0.5^(abs(i-j))
  }
} 
C <- chol(Cov)
X <- matrix(rnorm(T*p),T,p)\%*\%C
eps <- arima.sim(list(ar=rho), n = T+100)
eps <- eps[101:(T+100)] 
Y = X\%*\%beta0 + eps

fit1 =  rlasso(X, Y, penalty = list(homoscedastic = "none",
              lambda.start = 2*0.5*sqrt(T)*qnorm(1-0.1/(2*p))), post=FALSE)
beta = fit1$beta
intercept = fit1$intercept
res = Y - X \%*\% beta - intercept * rep(1, length(Y))

load = rep(0,p)
for(i in 1:p){
  load[i] = sqrt(lrvar(X[,i]*res)*T)
  }
reg.lasso.load1 <- rlassoLoad(X,Y,load) #lambda is chosen independent of regressor 
                                             #matrix X by default.

bn = 10 # block length
reg.lasso.load2 <- rlassoLoad(X, Y,load, bns=bn, nboot=5000,
                            X.dependent.lambda = TRUE, c=2.7)
} 
}
