% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/frenet_frame3d.R
\name{frenet_frame3d}
\alias{frenet_frame3d}
\title{Frenet-Serret frame for a 3D parametric curve}
\usage{
frenet_frame3d(
  X,
  Y,
  Z,
  a,
  b,
  t_points,
  h = 1e-04,
  plot = FALSE,
  n_samples = 400,
  vec_scale = NULL,
  curve_line = list(color = "blue", width = 2, dash = "solid"),
  T_line = list(color = "red", width = 4, dash = "solid"),
  N_line = list(color = "green", width = 4, dash = "solid"),
  B_line = list(color = "black", width = 4, dash = "solid"),
  show_curve = TRUE,
  show_points = TRUE,
  point_marker = list(color = "blue", size = 3, symbol = "circle"),
  scene = list(aspectmode = "data", xaxis = list(title = "x(t)"), yaxis = list(title =
    "y(t)"), zaxis = list(title = "z(t)")),
  bg = list(paper = "white", plot = "white"),
  tol = 1e-10
)
}
\arguments{
\item{X}{Function returning the \code{x} coordinate of the curve as a
function of the parameter \code{t}.}

\item{Y}{Function returning the \code{y} coordinate of the curve.}

\item{Z}{Function returning the \code{z} coordinate of the curve.}

\item{a}{Lower endpoint of the parameter interval.}

\item{b}{Upper endpoint of the parameter interval.}

\item{t_points}{Numeric vector with the parameter values where the frame
is computed and optionally plotted.}

\item{h}{Step size for centered finite difference approximations.}

\item{plot}{Logical; if \code{TRUE}, shows a 3D \pkg{plotly} visualization
of the curve together with the three vector fields.}

\item{n_samples}{Number of points used to sample the curve for plotting.}

\item{vec_scale}{Base scaling factor for the vector segments. If
\code{NULL}, it is estimated from the overall size of the sampled curve.}

\item{curve_line}{Style options for drawing the base curve.}

\item{T_line}{Style options for tangent vector segments.}

\item{N_line}{Style options for normal vector segments.}

\item{B_line}{Style options for binormal vector segments.}

\item{show_curve}{Logical; if \code{TRUE}, the base curve appears in the
plot.}

\item{show_points}{Logical; if \code{TRUE}, the evaluation points are
marked on the curve.}

\item{point_marker}{Plotly marker style for the evaluation points.}

\item{scene}{Plotly 3D scene configuration.}

\item{bg}{Background settings for the \pkg{plotly} figure.}

\item{tol}{Numeric tolerance used to detect degenerate derivative
situations.}
}
\value{
A tibble containing the parameter values and the coordinates of:
\itemize{
\item the point on the curve,
\item the tangent vector,
\item the normal vector,
\item the binormal vector,
\item a numerical estimate of the curvature.
}
Columns are named
\code{t, x, y, z, Tx, Ty, Tz, Nx, Ny, Nz, Bx, By, Bz, kappa}.
}
\description{
Computes the Frenet-Serret frame, that is, the tangent, normal and
binormal vectors of a three dimensional parametric curve at selected
values of the parameter. The frame is obtained from numerical
approximations of the first and second derivatives of the curve.
Optionally, the curve and the three vector fields can be displayed in a
3D interactive visualization using \pkg{plotly}.
}
\details{
At each parameter value in \code{t_points}, the function:
\itemize{
\item computes finite difference approximations of the first and second
derivatives of the curve,
\item normalizes the first derivative to obtain the unit tangent
direction,
\item uses the first and second derivatives to construct a principal
normal direction,
\item constructs the binormal direction as a unit vector orthogonal to
both the tangent and the normal,
\item evaluates a numerical estimate of the curvature using the same
derivative information.
}

If the derivative information is too small or nearly degenerate (for
example, when the tangent direction cannot be reliably obtained), some
components of the frame may be set to \code{NA}. The tolerance parameter
\code{tol} controls how these situations are detected.

When \code{plot = TRUE}, the function displays:
\itemize{
\item a sampled representation of the curve,
\item the evaluation points,
\item short line segments indicating the tangent, normal and binormal
directions at each evaluation point.
}

All visual elements can be styled or shown selectively through the
corresponding arguments.
}
\examples{
X <- function(t) t*cos(t)
Y <- function(t) t*sin(3*t)
Z <- function(t) t
frenet_frame3d(
  X, Y, Z, a = 0, b = 2*pi,
  t_points = c(pi/3, pi, 5*pi/3)
)

}
