% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/lagrange_check.R
\name{lagrange_check}
\alias{lagrange_check}
\title{Optimality check with Lagrange multipliers and bordered Hessian}
\usage{
lagrange_check(f, g, x, h = NULL, tol = 1e-06)
}
\arguments{
\item{f}{Objective function. It must be \code{function(x)} and
return a single numeric value. The argument \code{x} is a
numeric vector of length \code{n}.}

\item{g}{Equality constraints. Either a single function \code{function(x)}
returning a numeric vector of length \code{m}, or a list of scalar
functions \code{function(x)}, one per constraint.}

\item{x}{Numeric vector giving the candidate point at which the
optimality conditions are evaluated.}

\item{h}{Step size for finite differences. It can be:
\itemize{
\item a single numeric value used for all coordinates,
\item a numeric vector of length \code{n} with one step per
coordinate,
\item or \code{NULL}, in which case step sizes are chosen as
\code{1e-4 * (1 + abs(x))} componentwise.
}}

\item{tol}{Numeric tolerance used to judge feasibility of the
constraints, the effective rank of the Jacobian, near singularity
of matrices and very small principal minors.}
}
\value{
A list with components:
\itemize{
\item \code{ok_stationarity}: numeric value of the norm of the
stationarity residual. When constraints are present, this
measures how close the gradient of \code{f} is to the
linear combination given by the Jacobian transpose and the
Lagrange multipliers.
\item \code{ok_feasible}: maximum absolute value of the
constraint vector \code{g(x)} at the candidate point.
\item \code{lambda}: numeric vector of length \code{m} with the
Lagrange multipliers.
\item \code{J}: Jacobian matrix of the constraints at \code{x},
with dimension \code{m x n}.
\item \code{H_f}: Hessian matrix of the objective function
at \code{x}, of size \code{n x n}.
\item \code{H_g}: list of Hessian matrices corresponding to each
constraint function, each of size \code{n x n}.
\item \code{H_L}: Hessian matrix of the Lagrangian at \code{x}.
\item \code{B}: bordered Hessian matrix, of size
\code{(m + n) x (m + n)} when constraints are present.
\item \code{minors}: \code{data.frame} with one row per leading
principal minor. It contains the order of the minor, its
signed value and the logarithm of the absolute determinant
used in the computation.
\item \code{clasificacion}: character value equal to
\code{"minimo"}, \code{"maximo"} or \code{"indeterminado"},
according to the bordered Hessian criterion.
\item \code{notas}: character vector with diagnostic messages
about the rank of the Jacobian, near singularity of matrices
or any numerical issues detected.
}
}
\description{
Evaluates first- and second-order optimality conditions for a smooth
constrained optimization problem with equality constraints at a given
candidate point. The function checks the Lagrange conditions,
builds the bordered Hessian, and classifies the candidate as a
minimum, maximum or indeterminate/saddle according to the signs of
the leading principal minors.
}
\details{
Consider a problem of minimizing or maximizing a scalar function
\code{f(x)} subject to \code{m} equality constraints collected in
\code{g(x) = 0}, where \code{x} is a vector in R^n and
\code{g(x)} is a vector in R^m.

At the candidate point \code{x}, the function:
\itemize{
\item Approximates the gradient of \code{f} and the gradients
of each constraint using second-order central finite
differences.
\item Builds the Jacobian matrix \code{J} of the constraints
(rows are gradients of each constraint).
\item Approximates the Hessian matrix of \code{f} and the Hessian
of each constraint, also by central finite differences.
\item Forms the Hessian of the Lagrangian by combining the Hessian
of \code{f} and the Hessians of the constraints with the
Lagrange multipliers.
\item Builds the bordered Hessian matrix using the Jacobian and
the Hessian of the Lagrangian.
\item Computes leading principal minors of the bordered Hessian
and uses their signs to classify the candidate.
}

The classification is based on the standard bordered Hessian test:
after multiplying each leading principal minor by (-1)^m, if all
resulting values are positive the point is classified as a minimum;
if their signs alternate (negative, positive, negative, and so on),
the point is classified as a maximum. In any other case, the result
is reported as indeterminate.
All derivatives (gradients and Hessians) are computed numerically
with central finite differences of second order. The step sizes can
be given explicitly or chosen automatically from the scale of
the point \code{x}.
}
\examples{
## 1) Minimum with one constraint:
##    f(x,y) = x^2 + y^2,  g(x,y) = x + y - 1 = 0  -> (0.5, 0.5)
f1 <- function(x) x[1]^2 + x[2]^2
g1 <- function(x) x[1] + x[2] - 1
lagrange_check(f1, g1, x = c(0.5, 0.5))

## 2) Maximum with one constraint:
##    f(x,y) = -(x^2 + y^2),  g(x,y) = x + y - 1 = 0  -> (0.5, 0.5)
f2 <- function(x) -(x[1]^2 + x[2]^2)
lagrange_check(f2, g1, x = c(0.5, 0.5))

## 3) Two constraints in R^3 (minimum norm with two planes)
f3 <- function(x) sum(x^2)
g3 <- list(
  function(x) x[1] + x[2] + x[3] - 1,
  function(x) x[1] - x[3]
)
## Candidate solution: x1 = x3, 2*x1 + x2 = 1  ->  x = (1/3, 1/3, 1/3)
lagrange_check(f3, g3, x = c(1, 1, 1) / 3)

}
