#' @title Cut.off
#' @description Computes the cutoff threshold using the descriptive values generated by the bootstrap, also plots the empirical distribution and the sample wTO.
#' @param wTO_value is the table returned by the wTO with all the simulated values.
#' @param type type of plot
#' @param plot logical value indicating whether to plot the data.
#' @return plots in a new device the cutoff value, and the amount of links in each one of the bands.
#' @keywords internal
#' @import graphics
#' @importFrom stats density quantile
#' @importFrom methods is
#' @importFrom plyr arrange

Cut.off <- function(wTO_value, type, plot) {
  `%ni%` <- Negate(`%in%`)
  wTO_value <- plyr::arrange(wTO_value, wTO_value$Var1)
  wTO_value[is.na(wTO_value)] <- 0
  wTO_value$relstar <- wTO_value[, 2] / sum(wTO_value[, 2], na.rm = TRUE)
  wTO_value$relreal <- wTO_value[, 3] / sum(wTO_value[, 3], na.rm = TRUE)
  
  quantile.from.freq <- function(vals, freq, quant) {
    ord <- order(vals)
    cs <- cumsum(freq[ord])
    if (any(cs < quant)) {
      return(vals[max(which(cs < quant)) + 1])
    } else {
      return(min(vals))
    }
  }
  
  wTO_value$Var1 <- as.numeric(wTO_value$Var1)
  quantile_star <- data.frame(
    `0.1%` = quantile.from.freq(wTO_value$Var1, wTO_value$relstar, 0.001),
    `2.5%` = quantile.from.freq(wTO_value$Var1, wTO_value$relstar, 0.025),
    `10%` = quantile.from.freq(wTO_value$Var1, wTO_value$relstar, 0.1),
    `90%` = quantile.from.freq(wTO_value$Var1, wTO_value$relstar, 0.9),
    `97.5%` = quantile.from.freq(wTO_value$Var1, wTO_value$relstar, 0.975),
    `99.9%` = quantile.from.freq(wTO_value$Var1, wTO_value$relstar, 0.999)
  )
  
  quantile_real <- data.frame(
    `0.1%` = quantile.from.freq(wTO_value$Var1, wTO_value$relreal, 0.001),
    `2.5%` = quantile.from.freq(wTO_value$Var1, wTO_value$relreal, 0.025),
    `10%` = quantile.from.freq(wTO_value$Var1, wTO_value$relreal, 0.1),
    `90%` = quantile.from.freq(wTO_value$Var1, wTO_value$relreal, 0.9),
    `97.5%` = quantile.from.freq(wTO_value$Var1, wTO_value$relreal, 0.975),
    `99.9%` = quantile.from.freq(wTO_value$Var1, wTO_value$relreal, 0.999)
  )
  
  if (plot) {
    PLOT <- function(wTO_value) {
      graphics::par(xpd = FALSE)
      graphics::plot(wTO_value$relstar ~ wTO_value$Var1, type = "l",
                     xlim = c(floor(min(wTO_value$Var1)), 1),
                     main = type,
                     ylim = c(0, max(wTO_value$relstar)), axes = FALSE,
                     xlab = "wTO", ylab = "Density", col.main = "steelblue2", col.lab = "steelblue2")
      graphics::lines(wTO_value$Var1, wTO_value$relreal, type = "l", col = "violet")
      graphics::abline(h = 0, col = "gray", lty = 4)
      graphics::abline(v = as.numeric(quantile_real), col = c("red", "orange", "yellow", "yellow", "orange", "red"), lty = 2)
      graphics::axis(1, las = 1, cex.axis = 0.6, col = "steelblue", col.ticks = "steelblue3", col.axis = "steelblue")
      graphics::axis(2, las = 1, cex.axis = 0.6, col = "steelblue", col.ticks = "steelblue3", col.axis = "steelblue")
      graphics::par(xpd = TRUE)
      graphics::legend(c(0.9, max(wTO_value$relstar)), c("wTO - Data set", "wTO - Reshuffle", "99.9%", "95%", "80%"),
                       inset = c(-0.8, 0), lwd = 2, lty = 1,
                       col = c("violet", "black", "yellow", "orange", "red"), bty = "n", cex = 0.5)
    }
    
    res <- try(PLOT(wTO_value))
    if (!methods::is(res, 'try-error')) {
      res
    }
  }
  
  return(list(Empirical.Quantile = quantile_star, Quantile = quantile_real))
}
