library(wkb)
library(sp)
context("Conversion to and from WKB Polygon representations")

# create an object of class SpatialPolygons
# in which each Polygons object contains a single Polygon object
# with only one exterior boundary
triangle <- Polygons(
  list(
    Polygon(data.frame(x = c(2, 2.5, 3, 2), y = c(2, 3, 2, 2)))
  ), "1")
rectangle <- Polygons(
  list(
    Polygon(data.frame(x = c(0, 0, 1, 1, 0), y = c(0, 1, 1, 0, 0)))
  ), "2")
refobj <- SpatialPolygons(list(triangle, rectangle))

# create an AsIs list of little-endian WKB geometry representations of type Polygon
refwkb <- I(list(
  as.raw(c(0x01, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x04, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08,
           0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x40, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x40)),
  as.raw(c(0x01, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00, 0x00, 0x05, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0,
           0x3f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf0, 0x3f, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0xf0, 0x3f, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0xf0, 0x3f, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00))
))

# create an AsIs list of big-endian WKB geometry representations of type Polygon
refwkbbe <- I(list(
  as.raw(c(0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
           0x00, 0x04, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x04, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x40, 0x08, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00)),
  as.raw(c(0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x01, 0x00, 0x00,
           0x00, 0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3f, 0xf0, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
           0x00, 0x00, 0x00, 0x00, 0x00))
))

test_that("little-endian WKB Polygon representation converts to SpatialPolygons object", {
  # convert little-endian WKB Polygon representation to SpatialPolygons object
  obj <- readWKB(refwkb)

  # test
  expect_equal(obj, refobj)
})

test_that(paste(
  "SpatialPolygons object in which each Polygons object contains a single Polygon",
  "object with only one exterior boundary converts to little-endian WKB Polygon representation"), {
  # convert SpatialPolygons object to little-endian WKB Polygon representation
  wkb <- writeWKB(refobj)

  # test
  expect_equal(wkb, refwkb)
})

test_that("big-endian WKB Polygon representation converts to SpatialPolygons object", {
  # convert big-endian WKB Polygon representation to SpatialPolygons object
  obj <- readWKB(refwkbbe)

  # test
  expect_equal(obj, refobj)
})

test_that(paste(
  "SpatialPolygons object in which each Polygons object contains a single Polygon",
  "object with only one exterior boundary converts to big-endian WKB Polygon representation"), {
  # convert SpatialPolygons object to big-endian WKB Polygon representation
  wkbbe <- writeWKB(refobj, endian = "big")

  # test
  expect_equal(wkbbe, refwkbbe)
})
