/* pbmtogf.c -- convert a pbm format file to a gf foft file,
   also generate a pl for the font.

Copyright (C) 1998, 1999 Wai Wong

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  

Version 0.1 1998/03/31 Initial version
Version 1.0 1999/03/25 First public release version
*/

#include <stdio.h>
#include <string.h>

#define PBMTOGF

#include "pbmtogf.h"

#define isoptchar(c) (c == '-')

extern char *readpbm(FILE *);
extern char *get_num(char *, unsigned short *);
extern void write_pl(FILE *);

char progname[] = "pbmtogf";
char version[] = "1.0";

int debug = 0;
int verbose = 1;

char infilename[MAXLINELEN] = "";
char outfilename[MAXLINELEN] = "pbmtogf.600gf";
char plfilename[MAXLINELEN] = "pbmtogf.pl";
char metfilename[MAXLINELEN] = "";
FILE *infile = NULL;
FILE *outfile = NULL;

/* global parameters */
unsigned short img_wid = 0;
unsigned short img_ht = 0;
unsigned short ch_wid = 0;
unsigned short ch_ht = 0;
unsigned short cols = 1;
unsigned short rows = 1;
unsigned short ccode1 = 0;
unsigned short ccode2 = 0;
unsigned short design_size = 12; /* 12582912 = 12 * 1048576 (2^20) */
unsigned long checksum = 0;
unsigned long hppp = 544093L; /* (600 /72.27) * 2^16 */
unsigned long vppp = 544093L;
unsigned short res = 600;
unsigned short nchars = 0;

char resstr[15];

void error(int kind, char *mess)
{
  fprintf(stderr, "%s: %s\n", progname, mess);

  if(kind >= SERIOUS) exit(1);
}

void usage()
{
  fprintf(stderr, "Usage: readpbm [options] infilename\n");
  fprintf(stderr, "The following options are accepted:\n");
  fprintf(stderr, "  -help  display this message\n");
  fprintf(stderr, "  -d[n]  enable debugging\n");
  fprintf(stderr, "  -c<n>  n characters per row\n");
  fprintf(stderr, "  -r<n>  n rows in the image\n");
  fprintf(stderr, "  -w<n>  character width in pixels\n");
  fprintf(stderr, "  -h<n>  character height in pixels\n");
  fprintf(stderr, "  -f<n>  character code of first character\n");
  fprintf(stderr, "  -n<n>  number of characters \n");
  fprintf(stderr, "  -s<n>  output resolution in pixels per inch\n");
  fprintf(stderr, "  -z<n>  designe size in points\n");
  fprintf(stderr, "  -m<s>  metrics file name (not yet impletemented)\n");

}/* end of usage () */

main(int argc, char *argv[])
{
  char *img, *cp, *cp2;
  int i = 0;
  int f = 0;
  unsigned short d = 0;

  if(verbose)
    fprintf(stderr, "This is %s, version %s\n", progname, version);

  while(argc-- > 1){
    i++;
    cp = argv[i];
    if(debug > 1) fprintf(stderr,"Argv[%d]=%s\n", i, cp);
    if(isoptchar(*cp)){
      switch(*(++cp)){
	case'd':{ /* debugging */
	  if(isdigit(*(++cp))) debug = *cp -'0';
	  else debug = 1;
	  break;
	}
	case'c':{ /* number of characters per row*/
	  cols = 0;
	  cp = get_num(++cp, &cols);
	  break;
	}
	case'r':{ /* number of rows */
	  rows = 0;
	  cp = get_num(++cp, &rows);
	  break;
	}
	case'w':{ /* character width in pixels */
	  ch_wid = 0;
	  cp = get_num(++cp, &ch_wid);
	  break;
	}
	case'h':{ /* character height in pixels */
	  if(strcmp(cp,"help") == 0){
	    usage(); exit(0);
	  }
	  ch_ht = 0;
	  cp = get_num(++cp, &ch_ht);
	  break;
	}
	case'm':{ /* metrics file name */
	  strcpy(metfilename, ++cp);
	  break;
	}
	case'f':{ /* Character code of first character */
	  ccode1 = 0;
	  cp = get_num(++cp, &ccode1);
	  break;
	}
	case'n':{ /* Number of characters */
	  ccode2 = 0;
	  cp = get_num(++cp, &ccode2);
	  break;
	}
	case's':{ /* output resolution in pixel per inch */
	  res = 0;
	  cp = get_num(++cp, &res);
	  break;
	}
	case'z':{ /* design size in points */
	  design_size = 0;
	  cp = get_num(++cp, &design_size);
	  break;
	}
      default:{
	  error(WARN, "Unknown option");
	  usage();
	  exit(1);
	} 
      }
      }else{
	if(f == 0){
	  strcpy(infilename, cp);
	  f++;
	}else{
	  error(WARN, "Too many file names, ignored!\n");
	  usage();
	}
      }    
  }

  /* validate the character code and number of characters */
  if(ccode2 == 0)    ccode2 = rows * cols;

  if(ccode2 > MAXNCHARS)
    error(FATAL, "Too many characters, break the bitmap to several smaller ones");
  if((ccode1 + ccode2) > MAXNCHARS)
    error(FATAL, "Character code too large");

  if(debug){
    fprintf(stderr,"Args:cols=%d,rows=%d,ch_wid=%d,ch_ht=%d,size=%d,in=%s,out=%s\n",
	    cols, rows, ch_wid, ch_ht, design_size, infilename, outfilename);
    fprintf(stderr,"Args:char_code1=%d, char_code2=%d,met=%s\n",
	    ccode1, ccode2, metfilename);
  }



  if(f == 0){
    usage();
    exit(1);
  }
  
  if((infile = fopen(infilename, "rb")) == NULL)
    error(FATAL, "error opening input file");

  img = readpbm(infile);

  if(f > 0) fclose(infile);

  /* Validate the charcter size */
  if((cols > 1) || (rows > 1)){ /* at least one of them have been set */
    if(ch_wid != 0){ /* It has been set */
      if((ch_wid * cols) > img_wid){
	error(SERIOUS, "Invalid character width or number of columns");
      }
    }
    if(ch_ht != 0){ /* It has been set */
      if((ch_ht * rows) > img_ht){
	error(SERIOUS, "Invalid character heigth or number of rows");
      }
    }
  }
  if(cols == 0) cols = 1;
  if(rows == 0) rows = 1;
  if(ch_wid == 0) ch_wid = img_wid / cols;
  if(ch_ht == 0) ch_ht = img_ht / rows;

  if(debug)
    fprintf(stderr,"Args:cols=%d,rows=%d,ch_wid=%d,ch_ht=%d\n",
	    cols, rows, ch_wid, ch_ht);

  if(f == 1){ /* work out output file names */
    strcpy(outfilename, infilename);
    cp = strrchr(outfilename, '.');
    *cp = '\0';
    strcpy(plfilename, outfilename);
    sprintf(resstr, ".%dgf", res);
    strcat(outfilename, resstr);
    strcat(plfilename, ".pl");
  }

  if(debug)
    fprintf(stderr,"Output to (GF) %s and (PL) %s\n", 
	    outfilename, plfilename);
   
  if((outfile = fopen(outfilename, "w")) == NULL)
    error(FATAL, "error opening output GF file");
  else{
    proc_image(outfile, img, img_wid, img_ht, ch_wid, ch_ht);

    if(f >= 2) fclose(outfile);
  }

  if((outfile = fopen(plfilename, "w")) == NULL)
    error(FATAL, "error opening output PL file");
  else{
    write_pl(outfile);

    if(f >= 2) fclose(outfile);
  }

  exit(0);
  }/* end of main () */
