/*
  This is the file loadtabl.c of the CJK macro package ver. 4.1.3
  (20-Jun-1997).
*/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ttf.h"
#include "loadtabl.h"

#define SEEK_SET 0

/*
   This module is very long. It just reads the data file according to the
   True Type Font Manual. Many parts of the file are unused code at the 
   moment.
*/

typedef struct {
  char tag[4];
  ULONG cksum;
  ULONG off;
  ULONG len;
} dir_chunk;


static int load_maxp(TTF *f, FILE *fp, int off, int len);
static int load_head(TTF *f, FILE *fp, int off, int len);
static int load_cmap(TTF *f, FILE *fp, int off, int len);
static int load_hhea(TTF *f, FILE *fp, int off, int len);
static int load_hmtx(TTF *f, FILE *fp, int off, int len);
static int load_loca(TTF *f, FILE *fp, int off, int len);
static int load_name(TTF *f, FILE *fp, int off, int len);
static int load_fpgm(TTF *f, FILE *fp, int off, int len);
static int load_cvt(TTF *f, FILE *fp, int off, int len);
static int load_hdmx(TTF *f, FILE *fp, int off, int len);
static int load_kern(TTF *f, FILE *fp, int off, int len);
static int load_LTSH(TTF *f, FILE *fp, int off, int len);
static int load_prep(TTF *f, FILE *fp, int off, int len);
static int load_PCLT(TTF *f, FILE *fp, int off, int len);

static void get_simple_glyph(OUTLINE *ol, FILE *fp, int recursion);


/*@@*/

/* prototype declaration in ttf.h */

TTF *ttf_init(name)
  char *name;
{
  FILE *fp;
  TTF *f;
  PROFILE *prof;
  int ntabs, i;
  char *tag;
  dir_chunk *ch;

  if ((fp = fopen(name, READ_BIN)) == NULL)
    ttf_error(ttf_err_invalid_filename, name);
  f = TTF_ALLOC(TTF, 1, "ttf_init");
  f->name = strdup(name);
  f->fp = fp;
  f->version = get_Fixed(fp);
  ntabs = get_USHORT(fp);
  /* skip search optimize data, I don't use it at the moment */
  fseek(fp, 12L, SEEK_SET);

  /* load ntabs tables pointer */
  ch = TTF_ALLOC(dir_chunk, ntabs, "ttf_init");
  for (i = 0; i < ntabs; i++)
  {
    ch[i].tag[0] = get_BYTE(fp);
    ch[i].tag[1] = get_BYTE(fp);
    ch[i].tag[2] = get_BYTE(fp);
    ch[i].tag[3] = get_BYTE(fp);
    ch[i].cksum = get_ULONG(fp);
    ch[i].off = get_ULONG(fp);
    ch[i].len = get_ULONG(fp);
  }

  /* 
     load `maxp' & `head' tables first because they will be used by other
     tables.
  */

  for (i = 0; i < ntabs; i++)
  {
    tag = ch[i].tag;
    if (*tag == 'm' && strncmp(tag, "maxp", 4) == 0)
    {
      if (load_maxp(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else if (*tag =='h' && strncmp(tag, "head", 4) == 0)
    {
      if (load_head(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
  }  
  /* allocate outline data from `maxp' table */
  prof = f->profile;
  f->glyf = TTF_ALLOC(OUTLINE, 1, "ttf_init");
  f->glyf->endCtrOfGlyphs =
    TTF_ALLOC(SHORT, prof->maxComponentElements * prof->maxComponentDepth,
    "ttf_init");
  f->glyf->endPtsOfContours =
    TTF_ALLOC(USHORT, __max(prof->maxContours, prof->maxCompositeContours),
    "ttf_init");
  f->glyf->xCoordinates =
    TTF_ALLOC(LONG, __max(prof->maxPoints, prof->maxCompositePoints),
    "ttf_init");
  f->glyf->yCoordinates =
    TTF_ALLOC(LONG, __max(prof->maxPoints, prof->maxCompositePoints),
    "ttf_init");
  f->glyf->flags =
    TTF_ALLOC(BYTE, __max(prof->maxPoints, prof->maxCompositePoints),
    "ttf_init");
  f->glyf->instruction =
    TTF_ALLOC(BYTE, prof->maxSizeOfInstructions, "ttf_init");

  /* 
     read the other tables now except the `glyf' table. It will be read
     on demand.
  */

  for (i = 0; i < ntabs; i++)
  {
    tag = ch[i].tag;
    if (*tag == 'c' && !strncmp(tag, "cmap", 4))
    {
      if (load_cmap(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else if (*tag == 'g' && !strncmp(tag, "glyf", 4))
    {
      f->glyfoff = ch[i].off;
    }
    else if (*tag == 'h' && !strncmp(tag, "head", 4))
    {
      ; /* already read */
    }
    else if (*tag == 'h' && !strncmp(tag, "hhea", 4))
    {
      if (load_hhea(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else if (*tag == 'h' && !strncmp(tag, "hmtx", 4))
    {
      if (load_hmtx(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else if (*tag == 'l' && !strncmp(tag, "loca", 4))
    {
      if (load_loca(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else if (*tag == 'm' && !strncmp(tag, "maxp", 4))
    {
      ; /* already read */
    }
    else if (*tag == 'n' && !strncmp(tag, "name", 4))
    {
      if (load_name(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else if (*tag == 'p' && !strncmp(tag, "post", 4))
    {
#if 0
      if (load_post(f, fp, ch[i].off, ch[i].len))
        goto return_error;
#endif
    }
    else if (*tag == 'f' && !strncmp(tag, "fpgm", 4))
    {
      if (load_fpgm(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else if (*tag == 'O' && !strncmp(tag, "OS/2", 4))
    {
#if 0
      if (load_OS2(f, fp, ch[i].off, ch[i].len))
        goto return_error;
#endif
    }
    else if (*tag == 'c' && !strncmp(tag, "cvt", 3))
    {
      if (load_cvt(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else if (*tag == 'h' && !strncmp(tag, "hdmx", 4))
    {
      if (load_hdmx(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else if (*tag == 'k' && !strncmp(tag, "kern", 4))
    {
      if (load_kern(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else if (*tag == 'L' && !strncmp(tag, "LTSH", 4))
    {
      if (load_LTSH(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else if (*tag == 'p' && !strncmp(tag, "prep", 4))
    {
      if (load_prep(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else if (*tag == 'W' && !strncmp(tag, "WIN", 3))
    {
#if 0
      if (load_WIN(f, fp, ch[i].off, ch[i].len))
        goto return_error;
#endif
    }
    else if (*tag == 'V' && !strncmp(tag, "VDMX", 4))
    {
#if 0
      if (load_VDMX(f, fp, ch[i].off, ch[i].len))
        goto return_error;
#endif
    }
    else if (*tag == 'F' && !strncmp(tag, "FOCA", 4))
    {
#if 0
      if (load_FOCA(f,fp,ch[i].off,ch[i].len))
        goto return_error;
#endif
    }
    else if (*tag == 'P' && !strncmp(tag, "PCLT", 4))
    {
      if (load_PCLT(f, fp, ch[i].off, ch[i].len))
        goto return_error;
    }
    else
    {
      /* we ignore unknown tables */
#if 0
      ttf_error(ttf_err_invalid_table, tag);
#endif
    }
  }
  return f;

return_error:
  if (f)
    TTF_FREE(f, "ttf_init");
  return NULL;
}

    
static int load_cmap(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  CMAP0 *m0;
  CMAP2 *m2;
  CMAP4 *m4;
  int i, j, length, num_var;
  int version;
  ULONG *offs;

  fseek(fp, off, SEEK_SET);
  /* ignore version */
  version = get_USHORT(fp);
  f->numberOfEncoding = get_USHORT(fp);
  f->cmap = TTF_ALLOC(CMAP, f->numberOfEncoding, "load_cmap");
  offs = TTF_ALLOC(ULONG, f->numberOfEncoding, "load_cmap");
  for (i = 0; i < f->numberOfEncoding; i++)
  {
    f->cmap[i].PlatformID = get_USHORT(fp);
    f->cmap[i].EncodingID = get_USHORT(fp);
    offs[i] = get_ULONG(fp);
  }
  for (i = 0; i < f->numberOfEncoding; i++)
  {
    fseek(fp, off + offs[i], SEEK_SET);
    f->cmap[i].format = get_USHORT(fp);
    switch (f->cmap[i].format)
    {
    case 0:
      m0 = f->cmap[i].map.cmap0 = TTF_ALLOC(CMAP0, 1, "load_cmap");
      /* ignore length and version */
      get_USHORT(fp);
      get_USHORT(fp);
      fread(m0->glyphIdArray, 1, 256, fp);
      break;
    case 2:
      m2 = f->cmap[i].map.cmap2 = TTF_ALLOC(CMAP2, 1, "load_cmap");
      num_var = 0;
      length = get_USHORT(fp);
      get_USHORT(fp);
      for (j = 0; j < 256; j++)
      {
        m2->subHeaderKeys[j] = get_USHORT(fp) / 8;
        if (num_var < m2->subHeaderKeys[j])
          num_var = m2->subHeaderKeys[j];
      }
      length -= (256 + 3) * 2 + num_var * 8;
      if (length <= 0)
        length += 65536;
      /* ignore version */
      m2->data = TTF_ALLOC(USHORT, num_var * 8, "load_cmap");
      for (j = 0; j <= num_var; j++)
      {
        m2->data[j * 4] = get_USHORT(fp);
        m2->data[j * 4 + 1] = get_USHORT(fp);
        m2->data[j * 4 + 2] = get_USHORT(fp);
        m2->data[j * 4 + 3] = get_USHORT(fp) - (num_var - j) * 8 - 2;
      }
      length /= 2;
      m2->glyphIdArray = TTF_ALLOC(USHORT, length, "load_cmap");
      for (j = 0; j < length; j++)
        m2->glyphIdArray[j] = get_USHORT(fp);
#if 0
      for (j = 0; j < 256; j++)
      {
        if (m2->subHeaderKeys[j] != 0)
        {
          k = m2->subHeaderKeys[j];
          printf(
            "%dth Offset %d firstcode %x entry %d idoff %d idrange %d\n",
            j, k, m2->data[k * 4], m2->data[k * 4 + 1],
            m2->data[k * 4 + 2], m2->data[k * 4 + 3]);
        }
      }
      for (j = 0; j < length; j++)
      {
        if (j && (j % 16 == 0))
        {
          printf("\n");
        }
        printf("%3d ", m2->glyphIdArray[j]);
      }
#endif
      break;
    case 4:
      m4 = f->cmap[i].map.cmap4 = TTF_ALLOC(CMAP4, 1, "load_cmap");
      length = get_USHORT(fp);
      get_USHORT(fp);

      m4->segCount = get_USHORT(fp) / 2;

      get_USHORT(fp);
      get_USHORT(fp);
      get_USHORT(fp);

      m4->endCount = TTF_ALLOC(USHORT, m4->segCount, "load_cmap");
      for (j = 0; j < m4->segCount; j++)
        m4->endCount[j] = get_USHORT(fp);

      get_USHORT(fp);

      m4->startCount = TTF_ALLOC(USHORT, m4->segCount, "load_cmap");
      for (j = 0; j < m4->segCount; j++)
        m4->startCount[j] = get_USHORT(fp);

      m4->idDelta = TTF_ALLOC(USHORT, m4->segCount, "load_cmap");
      for (j = 0; j < m4->segCount; j++)
        m4->idDelta[j] = get_USHORT(fp);

      m4->idRangeOffset = TTF_ALLOC(USHORT, m4->segCount, "load_cmap");
      for (j = 0; j < m4->segCount; j++)
        m4->idRangeOffset[j] = get_USHORT(fp);

      length -= (8 + 4 * m4->segCount) * 2;
      if (length <= 0)
        length += 65536;
      length /= 2;
      m4->glyphIdArray = TTF_ALLOC(USHORT, length, "load_cmap");
      for (j = 0; j < length; j++)
        m4->glyphIdArray[j] = get_USHORT(fp);
      break;
    case 6:
      f->numberOfEncoding--;
      break;
    default:
      ttf_error(ttf_err_invalid_format);
    }
  }
  TTF_FREE(offs, "load_cmap");
  return 0;
}


int load_glyf(f, fp, off, recursion)
  TTF *f;
  FILE *fp;
  int off, recursion;
{
  OUTLINE *ol;
  USHORT flags, glyphIndex;
  USHORT tmp;
  SHORT match_point_new = 0, match_point_old = 0;
  FWord x_offset = 0, y_offset = 0;
  SHORT x_scale = 1 << 14, y_scale = 1 << 14;
  SHORT scale_01 = 0, scale_10 = 0;
  LONG x, y;
  long file_pos;
  USHORT point_pos;
  FWord xMin, yMin, xMax, yMax;
  int i;

  fseek(fp, f->glyfoff + off, SEEK_SET);
  ol = f->glyf;

  ol->valid = 0;
  if (!recursion)
  {
    ol->curPoint = 0;
    ol->curContour = 0;
    ol->curGlyph = 0;
  }
  ol->endCtrOfGlyphs[ol->curGlyph] = get_SHORT(fp);

  xMin = get_FWord(fp); /* these values may be unreliable */
  yMin = get_FWord(fp); /* and will not be used. */
  xMax = get_FWord(fp); /* we compute the maxima from the point directly */
  yMax = get_FWord(fp);

  if (ol->endCtrOfGlyphs[ol->curGlyph] > 0)
  {
    if ((USHORT)(ol->endCtrOfGlyphs[ol->curGlyph]) >
        f->profile->maxContours)
      ttf_error(ttf_err_invalid_table);
    ol->endCtrOfGlyphs[ol->curGlyph] += ol->curContour;
    ol->numberOfCompGlyphs = 1;
    get_simple_glyph(ol, fp, recursion);
  }
  else
  {
#if 0
    /* composite fonts not implemented yet */
    ttf_error(ttf_err_not_implemented);
#endif
    do
    {
      flags = get_USHORT(fp);
      glyphIndex = get_USHORT(fp);

      /* some fonts don't follow the TTF specs exactly... */

      if (glyphIndex > f->profile->numglyphs)
      {
        fprintf(stderr, "ttf2pk: Invalid TTF index\n");
        return -1;
      }

      if (flags & FLAG_ARG_WORD)
      {
        if (flags & FLAG_XY_VALUES)
        {
          x_offset = get_FWord(fp);
          y_offset = get_FWord(fp);
        }
        else
        {
          match_point_new = get_SHORT(fp);
          match_point_old = get_SHORT(fp);
        }
      }
      else
      {
        tmp = get_USHORT(fp);
        if (flags & FLAG_XY_VALUES)
        {
          x_offset = (CHAR)(tmp >> 8);
          y_offset = (CHAR)(tmp & 0xFF);
        }
        else
        {
          match_point_new = tmp >> 8;
          match_point_old = tmp & 0xFF;
        }
      }

      /* get_F2Dot14() scales with 2^14 */
      if (flags & FLAG_SCALE)
        x_scale = get_F2Dot14(fp);
      else if (flags & FLAG_XY_SCALE)
      {
        x_scale = get_F2Dot14(fp);
        y_scale = get_F2Dot14(fp);
      }
      else if (flags & FLAG_2X2_SCALE)
      {
        x_scale = get_F2Dot14(fp);
        scale_01 = get_F2Dot14(fp);
        scale_10 = get_F2Dot14(fp);
        y_scale = get_F2Dot14(fp);
      }

      /* recursion */
      file_pos = ftell(fp);
      point_pos = ol->curPoint;
      load_glyf(f, fp, f->indexToLocation[glyphIndex], 1);
      fseek(fp, file_pos, SEEK_SET);

      for (i = point_pos; i < ol->curPoint; i++)
      {
        x = ol->xCoordinates[i];
        y = ol->yCoordinates[i];
        ol->xCoordinates[i] = (x * x_scale + y * scale_01) >> 14;
        ol->yCoordinates[i] = (x * scale_10 + y * y_scale) >> 14;
      }

      if (!(flags & FLAG_XY_VALUES))
      {
        x_offset = ol->xCoordinates[match_point_new + point_pos] -
                   ol->xCoordinates[match_point_old];
        y_offset = ol->yCoordinates[match_point_new + point_pos] -
                   ol->yCoordinates[match_point_old];
      }

      if (point_pos == 0)
      {
        ol->xMax = ol->xMin = ol->xCoordinates[0];
        ol->yMax = ol->yMin = ol->yCoordinates[0];
      }

      for (i = point_pos; i < ol->curPoint; i++)
      {
        /* we ignore FLAG_ROUND */
        ol->xCoordinates[i] += x_offset;
        ol->yCoordinates[i] += y_offset;

        if (ol->xCoordinates[i] > ol->xMax)
          ol->xMax = ol->xCoordinates[i];
        if (ol->xCoordinates[i] < ol->xMin)
          ol->xMin = ol->xCoordinates[i];
        if (ol->yCoordinates[i] > ol->yMax)
          ol->yMax = ol->yCoordinates[i];
        if (ol->yCoordinates[i] < ol->yMin)
          ol->yMin = ol->yCoordinates[i];
      }

#if 0
      /* we currently ignore FLAG_METRICS */
      if (flags & FLAG_METRICS)
      {
      }
#endif

      if (flags & FLAG_INSTR)
      {
        ol->instructionLength = get_USHORT(fp);
        if (ol->instructionLength)
          fread(ol->instruction, 1, ol->instructionLength, fp);
      }

      ol->numberOfCompGlyphs = ++(ol->curGlyph);
    } while (flags & FLAG_MORE_COMP);
  }

  f->glyf->valid = 1;
  return 1;
}


static void get_simple_glyph(ol, fp, recursion)
  OUTLINE *ol;
  FILE *fp;
  int recursion;
{
  int i, totpoint;
  BYTE b, fl, c, st;

  for (i = ol->curContour; i < ol->endCtrOfGlyphs[ol->curGlyph]; i++)
    ol->endPtsOfContours[i] = get_USHORT(fp) + ol->curPoint;
  totpoint =
   ol->endPtsOfContours[ol->endCtrOfGlyphs[ol->curGlyph] - 1] + 1;
  ol->instructionLength = get_USHORT(fp);
  if (ol->instructionLength)
    fread(ol->instruction, 1, ol->instructionLength, fp);

  for (i = ol->curPoint; i < totpoint;)
  {
    ol->flags[i++] = c = get_BYTE(fp);
    if (c & FLAG_REPEAT)
    {
      st = get_BYTE(fp);
      while (st--)
        ol->flags[i++] = c;
    }
  }

  for (i = ol->curPoint; i < totpoint; i++)
  {
    fl = ol->flags[i];
    if (fl & FLAG_X_SHORT_VECTOR)
    {
      b = get_BYTE(fp);
      if (fl & FLAG_X_SAME)
        ol->xCoordinates[i] = b;
      else
        ol->xCoordinates[i] = -b;
    }
    else if (fl & FLAG_X_SAME)
      ol->xCoordinates[i] = 0;
    else
      ol->xCoordinates[i] = get_SHORT(fp);
  }

  for (i = ol->curPoint; i < totpoint; i++)
  {
    fl = ol->flags[i];
    if (fl & FLAG_Y_SHORT_VECTOR)
    {
      b = get_BYTE(fp);
      if (fl & FLAG_Y_SAME)
        ol->yCoordinates[i] = b;
      else
        ol->yCoordinates[i] = -b;
    }
    else if (fl & FLAG_Y_SAME)
      ol->yCoordinates[i] = 0;
    else
      ol->yCoordinates[i] = get_SHORT(fp);
  }

  if (!recursion)
  {
    ol->xMax = ol->xMin = ol->xCoordinates[0];
    ol->yMax = ol->yMin = ol->yCoordinates[0];
  }
  
  for (i = 1 + ol->curPoint; i < totpoint; i++)
  {
    ol->xCoordinates[i] += ol->xCoordinates[i - 1];
    ol->yCoordinates[i] += ol->yCoordinates[i - 1];

    if (!recursion)
    {
      if (ol->xCoordinates[i] > ol->xMax)
        ol->xMax = ol->xCoordinates[i];
      if (ol->xCoordinates[i] < ol->xMin)
        ol->xMin = ol->xCoordinates[i];
      if (ol->yCoordinates[i] > ol->yMax)
        ol->yMax = ol->yCoordinates[i];
      if (ol->yCoordinates[i] < ol->yMin)
        ol->yMin = ol->yCoordinates[i];
    }
  }

  ol->curPoint = totpoint;
  ol->curContour = ol->endCtrOfGlyphs[ol->curGlyph];
}

    
static int load_head(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  Fixed version;
  HEAD *h;

  fseek(fp, off, SEEK_SET);
  version = get_Fixed(fp);
  if (version != 0x00010000)
    ttf_error(ttf_err_invalid_version);
  f->head = h = TTF_ALLOC(HEAD, 1, "load_head");

  h->fontRevision = get_Fixed(fp);
  /* forget checksum and magicnumber */
  get_ULONG(fp);
  get_ULONG(fp);
  h->flags = get_USHORT(fp);
  h->unitsPerEm = get_USHORT(fp);
  /* forget date data */
  get_ULONG(fp);
  get_ULONG(fp);
  get_ULONG(fp);
  get_ULONG(fp);
  
  h->xMin = get_FWord(fp);
  h->yMin = get_FWord(fp);
  h->xMax = get_FWord(fp);
  h->yMax = get_FWord(fp);
  h->macStyle = get_USHORT(fp);
  h->lowestRecPPEM = get_USHORT(fp);
  h->fontDirectionHint = get_SHORT(fp);
  h->indexToLocFormat = get_SHORT(fp);
  h->glyphDataFormat = get_SHORT(fp);

  return 0;
}


static int load_hhea(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  int version;
  HORIZON *h;

  fseek(fp, off, SEEK_SET);
  version = get_Fixed(fp);
  if (version != 0x00010000)
    ttf_error(ttf_err_invalid_version);
  
  f->horz = h = TTF_ALLOC(HORIZON, 1, "load_hhea");

  h->ascender = get_FWord(fp);
  h->descender = get_FWord(fp);
  h->lineGap = get_FWord(fp);
  h->advanceWidthMax = get_uFWord(fp);
  h->minLeftSideBearing = get_FWord(fp);
  h->minRightSideBearing= get_FWord(fp);
  h->xMaxExtent = get_FWord(fp);
  h->caretSlopeRise = get_SHORT(fp);
  h->caretSlopeRun  = get_SHORT(fp);
  /* stub bytes */
  get_SHORT(fp);
  get_SHORT(fp);
  get_SHORT(fp);
  get_SHORT(fp);
  get_SHORT(fp);
  h->metricDataFormat = get_SHORT(fp);
  h->numberOfHMetrics = get_USHORT(fp);
  h->longHorMetrics = NULL;
  h->leftSideBearing = NULL;
  return 0;
}


static int load_hmtx(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  int i;
  HORIZON *h;

  h = f->horz;

  fseek(fp, off, SEEK_SET);
  h->longHorMetrics = 
    TTF_ALLOC(HorMetric, h->numberOfHMetrics, "load_hmtx");
  len = f->profile->numglyphs - h->numberOfHMetrics;
  h->leftSideBearing = TTF_ALLOC(FWord, len, "load_hmtx");
  for (i = 0; i < h->numberOfHMetrics; i++)
  {
    h->longHorMetrics[i].advanceWidth = get_uFWord(fp);
    h->longHorMetrics[i].lsb = get_FWord(fp);
  }
  for (i = 0; i < len; i++)
    h->leftSideBearing[i] = get_FWord(fp);

  return 0;
}


static int load_loca(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  int i;

  fseek(fp, off, SEEK_SET);
  if (f->head == NULL)
    ttf_error(ttf_err_need_head_table);
  if (f->profile == NULL)
    ttf_error(ttf_err_need_maxp_table);
  f->indexToLocation =
    TTF_ALLOC(ULONG, f->profile->numglyphs + 1, "load_loca");
  if (f->head->indexToLocFormat == 0)
    for (i = 0; i <= f->profile->numglyphs; i++)
      f->indexToLocation[i] = get_USHORT(fp) * 2;
  else
    for (i = 0; i <= f->profile->numglyphs; i++)
      f->indexToLocation[i] = get_ULONG(fp);
  return 0;
}


static int load_maxp(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  int version;
  PROFILE *p;

  fseek(fp, off, SEEK_SET);
  version = get_Fixed(fp);
  if (version != 0x00010000)
    ttf_error(ttf_err_invalid_version);
  
  p = f->profile = TTF_ALLOC(PROFILE, 1, "load_maxp");
  p->numglyphs = get_USHORT(fp);
  p->maxPoints = get_USHORT(fp);
  p->maxContours = get_USHORT(fp);
  p->maxCompositePoints = get_USHORT(fp);
  p->maxCompositeContours = get_USHORT(fp);
  p->maxZones = get_USHORT(fp);
  p->maxTwilightPoints = get_USHORT(fp);
  p->maxStorage = get_USHORT(fp);
  p->maxFunctionDefs = get_USHORT(fp);
  p->maxInstructionDefs = get_USHORT(fp);
  p->maxStackElements = get_USHORT(fp);
  p->maxSizeOfInstructions = get_USHORT(fp);
  p->maxComponentElements = get_USHORT(fp);
  p->maxComponentDepth = get_USHORT(fp);
  return 0;
}


static int load_name(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  return 0;
}


static int load_cvt(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  int i;

  fseek(fp, off, SEEK_SET);
  f->cvt = TTF_ALLOC(USHORT, len / 2, "load_cvt");
  f->cvtlen = len / 2;
  for (i = 0; i < len / 2; i++)
    f->cvt[i] = get_USHORT(fp);
  return 0;
}


static int load_fpgm(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  int i;

  fseek(fp, off, SEEK_SET);
  f->fpgm = TTF_ALLOC(BYTE, len, "load_fpgm");
  f->fpgmlen = len;
  for (i = 0;i < len; i++)
    f->fpgm[i] = get_BYTE(fp);
  return 0;
}


static int load_kern(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  return 0;
}


static int load_LTSH(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  return 0;
}


static int load_prep(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  int rlen;

  fseek(fp, off, SEEK_SET);
  f->prep = TTF_ALLOC(BYTE, len, "load_prep");
  rlen = fread(f->prep, 1, len, fp);
  if (rlen != len)
    ttf_error(ttf_err_invalid_prep);
  f->preplen = len;
  return 0;
}


static int load_hdmx(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  return 0;
}


static int load_PCLT(f, fp, off, len)
  TTF *f;
  FILE *fp;
  int off, len;
{
  return 0;
}


/* end of loadtabl.c */
