IMPLEMENTATION MODULE pxlreader;

(* Author:         Andrew Trevorrow
   Implementation: Modula-2 under VAX/UNIX 4.2 BSD
   Date Started:   June, 1986

   Description:
   All the routines in this module are of course highly SYSDEP!
   See pxl.inf for a description of PXL files.
   Although a PXL file is considered to be a stream of 32-bit words,
   we will be randomly positioning to PXL bytes and then reading
   bytes via GetPXLByte, SignedPXLPair etc.
*)

FROM io IMPORT
   File, Open, Close;

FROM unix IMPORT
   fseek, fgetc;

FROM screenio IMPORT
   WriteString, WriteCard, WriteLn, RestoreTerminal;

VAR
   PXLfile : File;

(******************************************************************************)

PROCEDURE OpenPXLFile (filename : ARRAY OF CHAR) : BOOLEAN;

(* Return TRUE if given file can be opened. *)

BEGIN
PXLfile := Open(filename,"r");
RETURN PXLfile <> NIL;           (* PXLfile open if not NIL *)
END OpenPXLFile;

(******************************************************************************)

PROCEDURE MoveToPXLByte (offset : CARDINAL);

(* Set current position in PXLfile to given byte offset from start of file,
   where first byte is at offset 0.
*)

BEGIN
IF fseek(PXLfile,INTEGER(offset),0) < 0 THEN
   WriteString('fseek failed in MoveToPXLByte!'); WriteLn;
   RestoreTerminal; HALT;
END;
END MoveToPXLByte;

(******************************************************************************)

PROCEDURE MoveToPXLDirectory;

(* The word offset of the font directory is given in the 2nd last word.
   The last word should contain 1001 in TeX82 PXL files.
*)

VAR dirptr : CARDINAL;

BEGIN
(* move to byte at start of 2nd last PXL word *)
IF fseek(PXLfile,-8,2) < 0 THEN
   WriteString('fseek failed in MoveToPXLDirectory!'); WriteLn;
   RestoreTerminal; HALT;
END;
dirptr := 4 * CARDINAL(SignedPXLQuad());   (* byte offset of font directory *)
(* check last word contains 1001 *)
IF SignedPXLQuad() <> 1001 THEN
   WriteString('Bad PXL file! id word <> 1001.'); WriteLn;
   RestoreTerminal; HALT;
END;
MoveToPXLByte(dirptr);   (* move to starting byte of font directory *)
END MoveToPXLDirectory;

(******************************************************************************)

(* Here are the functions used to get byte/s from PXLfile. *)

PROCEDURE GetPXLByte () : INTEGER;

(* Return unsigned value of next byte in PXLfile and advance. *)

BEGIN
RETURN ORD(fgetc(PXLfile));
END GetPXLByte;

(******************************************************************************)

PROCEDURE GetTwoPXLBytes () : INTEGER;

(* Return unsigned value of next 2 bytes and advance. *)

VAR a, b : CARDINAL;

BEGIN
a := ORD(fgetc(PXLfile));
b := ORD(fgetc(PXLfile));
RETURN a * 256 + b;
END GetTwoPXLBytes;

(******************************************************************************)

PROCEDURE SignedPXLPair () : INTEGER;

(* Return the next 2 bytes, possibly signed, and advance. *)

VAR a, b : CARDINAL;

BEGIN
a := ORD(fgetc(PXLfile));
b := ORD(fgetc(PXLfile));
IF a < 128 THEN
   RETURN a * 256 + b;
ELSE
   RETURN (a - 256) * 256 + b;
END;
END SignedPXLPair;

(******************************************************************************)

PROCEDURE SignedPXLQuad () : INTEGER;

(* Return the value (possibly signed) of the next 4 bytes and advance. *)

VAR overlay : RECORD
                 CASE BOOLEAN OF
                    TRUE  : i : INTEGER |
                    FALSE : a,b,c,d : CHAR
                 END;
              END;

BEGIN
WITH overlay DO
   d := fgetc(PXLfile);
   c := fgetc(PXLfile);
   b := fgetc(PXLfile);
   a := fgetc(PXLfile);
END;
RETURN overlay.i;   (* SYSDEP: compiler did not allow RETURN i in WITH stmt! *)
END SignedPXLQuad;

(******************************************************************************)

PROCEDURE ClosePXLFile;

(* Close the currently open PXL file. *)

BEGIN
Close(PXLfile);
END ClosePXLFile;

(******************************************************************************)

BEGIN
END pxlreader.
