(* VDU routines for a VT640 terminal. *)

#include 'globals.h';
#include 'screenio.h';
#include 'vdu.h';
#include 'tek4010vdu.h';

(******************************************************************************)

PROCEDURE StartText;

(* We are about to draw text in dialogue region. *)

BEGIN
TEK4010StartText;
END; (* StartText *)

(******************************************************************************)

PROCEDURE MoveAbs (row, col : INTEGER);

(* Move cursor to given screen position. *)

BEGIN
WriteChar(ESC); WriteChar('[');
WriteInt(row);
WriteChar(';');
WriteInt(col);
WriteChar('H');
END; (* MoveAbs *)

(******************************************************************************)

PROCEDURE MoveToTextLine (line : INTEGER);

(* Move current position to start of given line. *)

BEGIN
WriteChar(GS);    (* cannot send CAN if VT640 is already in VT100 mode *)
WriteChar(CAN);   (* switch to VT100 mode *)
MoveAbs(line,1);
END; (* MoveToTextLine *)

(******************************************************************************)

PROCEDURE ClearTextLine (line : INTEGER);

(* Erase given line; note that DVItoVDU does not assume anything about the
   current position at the end of this routine.
*)

BEGIN
WriteChar(GS);
WriteChar(CAN);
MoveAbs(line,1);
WriteChar(ESC);
WriteString('[K');   (* erase to end of line *)
END; (* ClearTextLine *)

(******************************************************************************)

PROCEDURE ClearScreen;

BEGIN
WriteChar(GS);
WriteChar(CAN);
WriteChar(ESC);
WriteString('[2J');   (* erase all alphanumerics *)
TEK4010ClearScreen;
END; (* ClearScreen *)

(******************************************************************************)

PROCEDURE StartGraphics;

(* We are about to draw in window region. *)

BEGIN
TEK4010StartGraphics;
END; (* StartGraphics *)

(******************************************************************************)

PROCEDURE LoadFont (fontname : string;
                    fontsize : INTEGER;
                    mag, hscale, vscale : REAL);

BEGIN
TEK4010LoadFont(fontname,fontsize,mag,hscale,vscale);
END; (* LoadFont *)

(******************************************************************************)

PROCEDURE ShowChar (screenh, screenv : INTEGER;
                    ch : CHAR);

BEGIN
TEK4010ShowChar(screenh, screenv, ch);
END; (* ShowChar *)

(******************************************************************************)

PROCEDURE ShowRectangle (screenh, screenv,          (* top left pixel *)
                         width, height : INTEGER;   (* size of rectangle *)
                         ch : CHAR);                (* black pixel *)

BEGIN
TEK4010ShowRectangle(screenh, screenv, width, height, ch);
END; (* ShowRectangle *)

(******************************************************************************)

PROCEDURE ResetVDU;

BEGIN
WriteChar(GS);    (* cannot send CAN if VT640 is already in VT100 mode *)
WriteChar(CAN);   (* switch to VT100 mode *)
END; (* ResetVDU *)

(******************************************************************************)

PROCEDURE InitVDU;

(* The dialog region will be the top 4 text lines in VT100 mode:
      Line 1 = DVI status line,
      Line 2 = window status line,
      Line 3 = message line,
      Line 4 = command line.
   The window region will be text lines 5 to 24 in VT100 mode.
*)

BEGIN
InitTEK4010VDU;
DVIstatusl    := 1;      (* DVItoVDU assumes top text line = 1 *)
windowstatusl := 2;
messagel      := 3;
commandl      := 4;
bottoml       := 24;     (* also number of text lines on screen *)
(* The above values assume the VT640 is in VT100 mode;
   the following values assume it is emulating a Tektronix 4010.
   Note that windowv must be given a value using DVItoVDU's coordinate scheme
   where top left pixel is (0,0).
*)
windowv  := 130;         (* approx. height in TEK4010 pixels of 4 text lines
                            i.e. 4 * 780/24 *)
windowh  := 0;
windowht := 780 - windowv;
windowwd := 1024;
END; (* InitVDU *)
