/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains the Tektronix 4014 image-paste functionality.
 *
 *	A graphics file containing raw output generated for a Tektronix
 *	4014 is read in and interpreted. The output primitives (lines,
 *	characters, etc.) are transformed from a 4096 by 3120 device
 *	to the box specified as input, that is, the entire Tektronix
 *	screen is mapped to the box.
 *
 *	There are a few instances where output can be generated outside
 *	the designated box. If the beam Y location is near zero and alpha
 *	mode is entered, characters with descenders will go below the
 *	Y=0 axis. If the beam X location is near 4095 and alpha mode is
 *	entered, characters can exceed the X=4095 bound (this can also
 *	happen if the maximum characters allowed per line for a given
 *	character size is used). And, of course, Y coordinates greater than
 *	3119 will exceed the top bound (if you've ever done this, you
 *	can see the beam bouncing around off the top of the storage tube
 *	on the 4014). This is simply how the 4014 works, and is emulated
 *	faithfully.
 *
 *	The input graphics file is interpreted raw. If for some reason
 *	there is some implied carriage control (i.e. cr-lf) in lines
 *	produced while in 'alpha' mode, no cr-lf will be emulated to the
 *	output. The actual carriage returns and line feeds MUST be within
 *	the file records, not implied by carriage control attributes.
 *
 *	It is assumed the input file is reasonably well behaved. There
 *	are a number of unusual conditions that can occur that may not
 *	be emulated correctly by this software, although every attempt
 *	has been made to accurately emulate the Tektronix hardware.
 *	(For example, if you enter write-thru mode and point plot mode
 *	with a high intensity character, the beam will actually store
 *	on the tube. It is assumed that the intent of entering
 *	write-thru mode is for the output NOT to store, so is does
 *	not produce any output, regardless of the intensity character.
 *	There are other examples of combinations of various seldom-used
 *	capabilities that a reasonable person would not use to accom-
 *	plish a particular function.)
 */

#include "types.h"
#include "devtab.h"
#include "font.h"
#include "tk.h"

#include "arith.p"
#include "raster.p"
#include "fileio.p"
#include "font.p"
#include "strng.p"

/*
 *	Here are definitions of the various hardware straps and keyboard
 *	switch settings. These options are controlled by the /OPTIONS
 *	switch to the PASTE command. The defaults for these should serve
 *	adequately for most situations.
 */

unsigned short TKP_Options;
#define ALTDEL   0x0001		/* ESC ? can be used in place of DEL */
#define CRLF     0x0002		/* CR implies LF */
#define EGM      0x0004		/* Enhanced Graphics Module present */
#define LFCR     0x0008		/* LF implies CR */
#define MARGIN   0x0010		/* Margin control option specified */
#define SELECT   0x0020		/* Character set selection specified */
#define ALTCHR   0x0040		/* Alternate character set (APL) selected */
#define ASCIICHR 0x0080		/* ASCII character set selected */
#define MRGONE   0x0100		/* Page full on margin one overflow */
#define MRGTWO   0x0200		/* Page full on margin two overflow */

/*
 *	The following definition is necessary due to the lack of the
 *	APL character set raster descriptions:
 */

#define ALT_CHARSET 0	/* 0 = ASCII only, 1 = ASCII+APL */

/*
 *	Screen size definitions:
 */

#define HOME_X      0
#define HOME_Y   3071
#define RASTER_X 4096
#define RASTER_Y 3120
#define MAX_CPL   133

/*
 *	Incremental mode definitions:
 */

#define INCR_EAST  0x01
#define INCR_WEST  0x02
#define INCR_NORTH 0x04
#define INCR_SOUTH 0x08
#define INCR_PDOWN 0x10
#define INCR_PUP   0x20

/*
 *	Graphics mode attributes structure:
 */

struct Attributes {
	struct Ratio Matrix[4];		/* Transformation matrix */
	long Offset[2];			/* Offset to origin */
	long Cur_Mapped_X;		/* Mapped current x */
	long Cur_Mapped_Y;		/* Mapped current y */
	unsigned short CurX;		/* Current X position */
	unsigned short CurY;		/* Current Y position */
	unsigned short Flags;		/* Various on-off flags */
#define SEEN_EXB   0x0001
#define SEEN_LOY   0x0002
#define CROSSHAIRS 0x0004
#define PAGE_FULL  0x0008
#define MARGIN_2   0x0010
#define FIRST_OP   0x0020
#define ALT_CHARS  0x0040
#define PEN_DOWN   0x0080
#define BYPASS     0x0100
	unsigned char HiY, ExB, LoY, HiX, LoX;/* Address bytes */
	unsigned char Linestyle;	/* Current line type (0-7) */
	unsigned char Charsize;		/* Character size (0-3) */
	unsigned char Writing_Mode;	/* One of three writing modes */
#define NORMAL     0
#define DEFOCUSED  1
#define WRITE_THRU 2
};

/*
 *	Tektronix font struct definition:
 */

struct Tek_Font_Def {
	struct Font_Definition *Font;
	unsigned char (*Char_Set)[12];
};

/*
 *	Define all font and character set data structures statically:
 */

#define eight_defs(array,k)     &array[k], &array[k+1], &array[k+2], &array[k+3],\
				&array[k+4], &array[k+5], &array[k+6], &array[k+7]
#define sixtyfour_defs(array,k) eight_defs(array,k), eight_defs(array,k+8),\
				eight_defs(array,k+16), eight_defs(array,k+24),\
				eight_defs(array,k+32), eight_defs(array,k+40),\
				eight_defs(array,k+48), eight_defs(array,k+56)
#define ninetyfour_defs(array)  sixtyfour_defs(array,0), eight_defs(array,64),\
				eight_defs(array,72), eight_defs(array,80),\
				&array[88], &array[89], &array[90], &array[91],\
				&array[92], &array[93]

struct Char_Definition TKP_Extralarge_Char_Def[94];
struct Char_Definition TKP_Large_Char_Def[94];
struct Char_Definition TKP_Medium_Char_Def[94];
struct Char_Definition TKP_Small_Char_Def[94];

struct Char_Definition *TKP_Extralarge_Directory[127] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	ninetyfour_defs (TKP_Extralarge_Char_Def)
}, *TKP_Large_Directory[127] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	ninetyfour_defs (TKP_Large_Char_Def)
}, *TKP_Medium_Directory[127] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	ninetyfour_defs (TKP_Medium_Char_Def)
}, *TKP_Small_Directory[127] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	ninetyfour_defs (TKP_Small_Char_Def)
};

struct Font_Definition TKP_Extralarge_Font = {
	0,					/* Link */
	&TKP_Extralarge_Directory[0],		/* Pointer to directory */
	0,					/* Pointer to font encoding */
	"", "Tektronix_Extralarge",		/* Font area and name pointers */
	0,					/* Font number (filled in) */
	(ID_INTERNAL << ID_V_TYPE) + 1,		/* Pixel identifier */
	0,					/* Checksum */
	0,					/* Design size in scaled points (filled in) */
	0,					/* Design size in DVI units (filled in) */
	0,					/* At size (same as design size) */
	1000,					/* Magnification */
	0, 0, 0,				/* Maximum size, packed, unpacked, compressed */
	0,					/* Font index number (filled in) */
	arraysize(TKP_Extralarge_Directory),	/* Size of character directory array */
	0, 0,					/* Maximum width, height */
	{ '\0' },				/* Coding style */
	{ '\0' },				/* Font family name */
	0,					/* Font face */
	0					/* Flags byte (filled in) */
}, TKP_Large_Font = {
	0,					/* Link */
	&TKP_Large_Directory[0],		/* Pointer to directory */
	0,					/* Pointer to font encoding */
	"", "Tektronix_Large",			/* Font area and name pointers */
	0,					/* Font number (filled in) */
	(ID_INTERNAL << ID_V_TYPE) + 1,		/* Pixel identifier */
	0,					/* Checksum */
	0,					/* Design size in scaled points (filled in) */
	0,					/* Design size in DVI units (filled in) */
	0,					/* At size (same as design size) */
	1000,					/* Magnification */
	0, 0, 0,				/* Maximum size, packed, unpacked, compressed */
	0,					/* Font index number (filled in) */
	arraysize(TKP_Large_Directory),		/* Size of character directory array */
	0, 0,					/* Maximum width, height */
	{ '\0' },				/* Coding style */
	{ '\0' },				/* Font family name */
	0,					/* Font face */
	0					/* Flags byte (filled in) */
}, TKP_Medium_Font = {
	0,					/* Link */
	&TKP_Medium_Directory[0],		/* Pointer to directory */
	0,					/* Pointer to font encoding */
	"", "Tektronix_Medium",			/* Font area and name pointers */
	0,					/* Font number (filled in) */
	(ID_INTERNAL << ID_V_TYPE) + 1,		/* Pixel identifier */
	0,					/* Checksum */
	0,					/* Design size in scaled points (filled in) */
	0,					/* Design size in DVI units (filled in) */
	0,					/* At size (same as design size) */
	1000,					/* Magnification */
	0, 0, 0,				/* Maximum size, packed, unpacked, compressed */
	0,					/* Font index number (filled in) */
	arraysize(TKP_Medium_Directory),	/* Size of character directory array */
	0, 0,					/* Maximum width, height */
	{ '\0' },				/* Coding style */
	{ '\0' },				/* Font family name */
	0,					/* Font face */
	0					/* Flags byte (filled in) */
}, TKP_Small_Font = {
	0,					/* Link */
	&TKP_Small_Directory[0],		/* Pointer to directory */
	0,					/* Pointer to font encoding */
	"", "Tektronix_Small",			/* Font area and name pointers */
	0,					/* Font number (filled in) */
	(ID_INTERNAL << ID_V_TYPE) + 1,		/* Pixel identifier */
	0,					/* Checksum */
	0,					/* Design size in scaled points (filled in) */
	0,					/* Design size in DVI units (filled in) */
	0,					/* At size (same as design size) */
	1000,					/* Magnification */
	0, 0, 0,				/* Maximum size, packed, unpacked, compressed */
	0,					/* Font index number (filled in) */
	arraysize(TKP_Small_Directory),		/* Size of character directory array */
	0, 0,					/* Maximum width, height */
	{ '\0' },				/* Coding style */
	{ '\0' },				/* Font family name */
	0,					/* Font face */
	0					/* Flags byte (filled in) */
};

#if ALT_CHARSET
struct Char_Definition TKP_Extralarge_APL_Char_Def[94];
struct Char_Definition TKP_Large_APL_Char_Def[94];
struct Char_Definition TKP_Medium_APL_Char_Def[94];
struct Char_Definition TKP_Small_APL_Char_Def[94];

struct Char_Definition *TKP_Extralarge_APL_Directory[127] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	ninetyfour_defs (TKP_Extralarge_APL_Char_Def)
}, *TKP_Large_APL_Directory[127] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	ninetyfour_defs (TKP_Large_APL_Char_Def)
}, *TKP_Medium_APL_Directory[127] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	ninetyfour_defs (TKP_Medium_APL_Char_Def)
}, *TKP_Small_APL_Directory[127] = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	ninetyfour_defs (TKP_Small_APL_Char_Def)
};

struct Font_Definition TKP_Extralarge_APL_Font = {
	0,					/* Link */
	&TKP_Extralarge_APL_Directory[0],	/* Pointer to directory */
	0,					/* Pointer to font encoding */
	"", "Tektronix_Extralarge_APL",		/* Font area and name pointers */
	0,					/* Font number (filled in) */
	(ID_INTERNAL << ID_V_TYPE) + 1,		/* Pixel identifier */
	0,					/* Checksum */
	0,					/* Design size in scaled points (filled in) */
	0,					/* Design size in DVI units (filled in) */
	0,					/* At size (same as design size) */
	1000,					/* Magnification */
	0, 0, 0,				/* Maximum size, packed, unpacked, compressed */
	0,					/* Font index number (filled in) */
	arraysize(TKP_Extralarge_APL_Directory),/* Size of character directory array */
	0, 0,					/* Maximum width, height */
	{ '\0' },				/* Coding style */
	{ '\0' },				/* Font family name */
	0,					/* Font face */
	0					/* Flags byte (filled in) */
}, TKP_Large_APL_Font = {
	0,					/* Link */
	&TKP_Large_APL_Directory[0],		/* Pointer to directory */
	0,					/* Pointer to font encoding */
	"", "Tektronix_Large_APL",		/* Font area and name pointers */
	0,					/* Font number (filled in) */
	(ID_INTERNAL << ID_V_TYPE) + 1,		/* Pixel identifier */
	0,					/* Checksum */
	0,					/* Design size in scaled points (filled in) */
	0,					/* Design size in DVI units (filled in) */
	0,					/* At size (same as design size) */
	1000,					/* Magnification */
	0, 0, 0,				/* Maximum size, packed, unpacked, compressed */
	0,					/* Font index number (filled in) */
	arraysize(TKP_Large_APL_Directory),	/* Size of character directory array */
	0, 0,					/* Maximum width, height */
	{ '\0' },				/* Coding style */
	{ '\0' },				/* Font family name */
	0,					/* Font face */
	0					/* Flags byte (filled in) */
}, TKP_Medium_APL_Font = {
	0,					/* Link */
	&TKP_Medium_APL_Directory[0],		/* Pointer to directory */
	0,					/* Pointer to font encoding */
	"", "Tektronix_Medium_APL",		/* Font area and name pointers */
	0,					/* Font number (filled in) */
	(ID_INTERNAL << ID_V_TYPE) + 1,		/* Pixel identifier */
	0,					/* Checksum */
	0,					/* Design size in scaled points (filled in) */
	0,					/* Design size in DVI units (filled in) */
	0,					/* At size (same as design size) */
	1000,					/* Magnification */
	0, 0, 0,				/* Maximum size, packed, unpacked, compressed */
	0,					/* Font index number (filled in) */
	arraysize(TKP_Medium_APL_Directory),	/* Size of character directory array */
	0, 0,					/* Maximum width, height */
	{ '\0' },				/* Coding style */
	{ '\0' },				/* Font family name */
	0,					/* Font face */
	0					/* Flags byte (filled in) */
}, TKP_Small_APL_Font = {
	0,					/* Link */
	&TKP_Small_APL_Directory[0],		/* Pointer to directory */
	0,					/* Pointer to font encoding */
	"", "Tektronix_Small_APL",		/* Font area and name pointers */
	0,					/* Font number (filled in) */
	(ID_INTERNAL << ID_V_TYPE) + 1,		/* Pixel identifier */
	0,					/* Checksum */
	0,					/* Design size in scaled points (filled in) */
	0,					/* Design size in DVI units (filled in) */
	0,					/* At size (same as design size) */
	1000,					/* Magnification */
	0, 0, 0,				/* Maximum size, packed, unpacked, compressed */
	0,					/* Font index number (filled in) */
	arraysize(TKP_Small_APL_Directory),	/* Size of character directory array */
	0, 0,					/* Maximum width, height */
	{ '\0' },				/* Coding style */
	{ '\0' },				/* Font family name */
	0,					/* Font face */
	0					/* Flags byte (filled in) */
};
#endif

/*
 *	The following are the hardware character definitions. A minimal
 *	hardware character on the 4014 is a 7 by 9 character raster
 *	inside a 9 by 12 character cell. The origin of the raster is at
 *	a position at the left edge of the cell and three units up from
 *	the bottom (for descenders). APL character set is missing because
 *	the author's Tektronix doesn't have it.
 */

unsigned char TKP_Ascii[94][12] = {
	{ 0x10,0x10,0x10,0x10,0x10,0x10,0x00,0x00,0x10,0x00,0x00,0x00 }, /* ! */
	{ 0x28,0x28,0x28,0x28,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 }, /* " */
	{ 0x28,0x28,0xFE,0x28,0x28,0x28,0xFE,0x28,0x28,0x00,0x00,0x00 }, /* # */
	{ 0x28,0x7E,0xA8,0xA8,0x7C,0x2A,0x2A,0xFC,0x28,0x00,0x00,0x00 }, /* $ */
	{ 0x40,0xA2,0xA4,0x48,0x10,0x24,0x4A,0x8A,0x04,0x00,0x00,0x00 }, /* % */
	{ 0x60,0x90,0x90,0x60,0x60,0x94,0x8C,0x84,0x7A,0x00,0x00,0x00 }, /* & */
	{ 0x08,0x08,0x10,0x20,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 }, /* ' */
	{ 0x08,0x10,0x20,0x20,0x20,0x20,0x20,0x10,0x08,0x00,0x00,0x00 }, /* ( */
	{ 0x20,0x10,0x08,0x08,0x08,0x08,0x08,0x10,0x20,0x00,0x00,0x00 }, /* ) */
	{ 0x10,0x92,0x54,0x38,0x10,0x38,0x54,0x92,0x10,0x00,0x00,0x00 }, /* * */
	{ 0x00,0x10,0x10,0x10,0xFE,0x10,0x10,0x10,0x00,0x00,0x00,0x00 }, /* + */
	{ 0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x38,0x10,0x60,0x00,0x00 }, /* , */
	{ 0x00,0x00,0x00,0x00,0xFE,0x00,0x00,0x00,0x00,0x00,0x00,0x00 }, /* - */
	{ 0x00,0x00,0x00,0x00,0x00,0x00,0x10,0x38,0x10,0x00,0x00,0x00 }, /* . */
	{ 0x00,0x02,0x04,0x08,0x10,0x20,0x40,0x80,0x00,0x00,0x00,0x00 }, /* / */
	{ 0x38,0x44,0x86,0x8A,0x92,0xA2,0xC2,0x44,0x38,0x00,0x00,0x00 }, /* 0 */
	{ 0x10,0x30,0x50,0x10,0x10,0x10,0x10,0x10,0x7C,0x00,0x00,0x00 }, /* 1 */
	{ 0x7C,0x82,0x02,0x02,0x7C,0x80,0x80,0x82,0xFE,0x00,0x00,0x00 }, /* 2 */
	{ 0x7C,0x82,0x02,0x02,0x3C,0x02,0x02,0x82,0x7C,0x00,0x00,0x00 }, /* 3 */
	{ 0x04,0x0C,0x14,0x24,0x44,0xFE,0x04,0x04,0x04,0x00,0x00,0x00 }, /* 4 */
	{ 0xFE,0x80,0x80,0x80,0xFC,0x02,0x02,0x84,0x78,0x00,0x00,0x00 }, /* 5 */
	{ 0x3C,0x42,0x80,0x80,0xBC,0xC2,0x82,0x82,0x7C,0x00,0x00,0x00 }, /* 6 */
	{ 0xFE,0x82,0x02,0x04,0x08,0x10,0x10,0x10,0x10,0x00,0x00,0x00 }, /* 7 */
	{ 0x7C,0x82,0x82,0x44,0x38,0x44,0x82,0x82,0x7C,0x00,0x00,0x00 }, /* 8 */
	{ 0x7C,0x82,0x82,0x82,0x42,0x3E,0x02,0x84,0x78,0x00,0x00,0x00 }, /* 9 */
	{ 0x00,0x10,0x38,0x10,0x00,0x10,0x38,0x10,0x00,0x00,0x00,0x00 }, /* : */
	{ 0x00,0x00,0x00,0x10,0x38,0x10,0x00,0x10,0x38,0x10,0x60,0x00 }, /* ; */
	{ 0x06,0x08,0x10,0x20,0x40,0x20,0x10,0x08,0x06,0x00,0x00,0x00 }, /* < */
	{ 0x00,0x00,0x00,0x7C,0x00,0x7C,0x00,0x00,0x00,0x00,0x00,0x00 }, /* = */
	{ 0xC0,0x20,0x10,0x08,0x04,0x08,0x10,0x20,0xC0,0x00,0x00,0x00 }, /* > */
	{ 0x7C,0x82,0x82,0x02,0x1C,0x10,0x10,0x00,0x10,0x00,0x00,0x00 }, /* ? */
	{ 0x3C,0x42,0x9A,0xAA,0xAA,0xBE,0x80,0x42,0x3C,0x00,0x00,0x00 }, /* @ */
	{ 0x10,0x28,0x44,0x82,0x82,0xFE,0x82,0x82,0x82,0x00,0x00,0x00 }, /* A */
	{ 0xF8,0x44,0x42,0x44,0x78,0x44,0x42,0x44,0xF8,0x00,0x00,0x00 }, /* B */
	{ 0x38,0x44,0x82,0x80,0x80,0x80,0x82,0x44,0x38,0x00,0x00,0x00 }, /* C */
	{ 0xF8,0x44,0x42,0x42,0x42,0x42,0x42,0x44,0xF8,0x00,0x00,0x00 }, /* D */
	{ 0xFE,0x80,0x80,0x80,0xF8,0x80,0x80,0x80,0xFE,0x00,0x00,0x00 }, /* E */
	{ 0xFE,0x80,0x80,0x80,0xF8,0x80,0x80,0x80,0x80,0x00,0x00,0x00 }, /* F */
	{ 0x38,0x44,0x82,0x80,0x80,0x8E,0x82,0x46,0x3A,0x00,0x00,0x00 }, /* G */
	{ 0x82,0x82,0x82,0x82,0xFE,0x82,0x82,0x82,0x82,0x00,0x00,0x00 }, /* H */
	{ 0x7C,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x7C,0x00,0x00,0x00 }, /* I */
	{ 0x0E,0x04,0x04,0x04,0x04,0x04,0x84,0x84,0x78,0x00,0x00,0x00 }, /* J */
	{ 0x86,0x88,0x90,0xA0,0xC0,0xA0,0x90,0x88,0x86,0x00,0x00,0x00 }, /* K */
	{ 0xE0,0x40,0x40,0x40,0x40,0x40,0x40,0x42,0x7E,0x00,0x00,0x00 }, /* L */
	{ 0xC6,0xAA,0x92,0x92,0x92,0x82,0x82,0x82,0x82,0x00,0x00,0x00 }, /* M */
	{ 0x82,0xC2,0xA2,0x92,0x8A,0x86,0x82,0x82,0x82,0x00,0x00,0x00 }, /* N */
	{ 0x38,0x44,0x82,0x82,0x82,0x82,0x82,0x44,0x38,0x00,0x00,0x00 }, /* O */
	{ 0xFC,0x82,0x82,0x82,0xFC,0x80,0x80,0x80,0x80,0x00,0x00,0x00 }, /* P */
	{ 0x38,0x44,0x82,0x82,0x82,0x8A,0x8A,0x44,0x3A,0x00,0x00,0x00 }, /* Q */
	{ 0xFC,0x82,0x82,0x82,0xFC,0x90,0x88,0x84,0x82,0x00,0x00,0x00 }, /* R */
	{ 0x3C,0x42,0x80,0x80,0x7C,0x02,0x02,0x84,0x78,0x00,0x00,0x00 }, /* S */
	{ 0xFE,0x92,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x00,0x00,0x00 }, /* T */
	{ 0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x82,0x7C,0x00,0x00,0x00 }, /* U */
	{ 0x82,0x82,0x82,0x82,0x82,0x82,0x44,0x28,0x10,0x00,0x00,0x00 }, /* V */
	{ 0x82,0x82,0x82,0x82,0x82,0x92,0x92,0x92,0x6C,0x00,0x00,0x00 }, /* W */
	{ 0x82,0x82,0x44,0x28,0x10,0x28,0x44,0x82,0x82,0x00,0x00,0x00 }, /* X */
	{ 0x82,0x82,0x82,0x44,0x28,0x10,0x10,0x10,0x10,0x00,0x00,0x00 }, /* Y */
	{ 0xFE,0x82,0x04,0x08,0x10,0x20,0x40,0x82,0xFE,0x00,0x00,0x00 }, /* Z */
	{ 0xFE,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0xC0,0xFE,0x00,0x00,0x00 }, /* [ */
	{ 0x00,0x80,0x40,0x20,0x10,0x08,0x04,0x02,0x00,0x00,0x00,0x00 }, /* \ */
	{ 0xFE,0x06,0x06,0x06,0x06,0x06,0x06,0x06,0xFE,0x00,0x00,0x00 }, /* ] */
	{ 0x00,0x00,0x00,0x10,0x28,0x44,0x82,0x00,0x00,0x00,0x00,0x00 }, /* ^ */
	{ 0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0xFE,0x00,0x00,0x00 }, /* _ */
	{ 0x40,0x20,0x10,0x08,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00 }, /* ` */
	{ 0x00,0x00,0x00,0xF8,0x04,0x7C,0x84,0x84,0x7A,0x00,0x00,0x00 }, /* a */
	{ 0xC0,0x40,0x40,0x5C,0x62,0x42,0x42,0x62,0xDC,0x00,0x00,0x00 }, /* b */
	{ 0x00,0x00,0x00,0x7C,0x80,0x80,0x80,0x82,0x7C,0x00,0x00,0x00 }, /* c */
	{ 0x06,0x04,0x04,0x74,0x8C,0x84,0x84,0x8C,0x76,0x00,0x00,0x00 }, /* d */
	{ 0x00,0x00,0x00,0x7C,0x82,0xFE,0x80,0x82,0x7C,0x00,0x00,0x00 }, /* e */
	{ 0x18,0x24,0x20,0x20,0x70,0x20,0x20,0x20,0x70,0x00,0x00,0x00 }, /* f */
	{ 0x00,0x00,0x00,0x76,0x8C,0x84,0x84,0x8C,0x74,0x04,0x84,0x78 }, /* g */
	{ 0xC0,0x40,0x40,0x58,0x64,0x44,0x44,0x44,0xC6,0x00,0x00,0x00 }, /* h */
	{ 0x00,0x10,0x00,0x30,0x10,0x10,0x10,0x10,0x7C,0x00,0x00,0x00 }, /* i */
	{ 0x00,0x04,0x00,0x0C,0x04,0x04,0x04,0x04,0x04,0x44,0x44,0x38 }, /* j */
	{ 0xC0,0x40,0x40,0x46,0x48,0x70,0x50,0x48,0xC6,0x00,0x00,0x00 }, /* k */
	{ 0x30,0x10,0x10,0x10,0x10,0x10,0x10,0x10,0x0C,0x00,0x00,0x00 }, /* l */
	{ 0x00,0x00,0x00,0xB6,0xDA,0x92,0x92,0x92,0x92,0x00,0x00,0x00 }, /* m */
	{ 0x00,0x00,0x00,0xD8,0x64,0x44,0x44,0x44,0xC6,0x00,0x00,0x00 }, /* n */
	{ 0x00,0x00,0x00,0x38,0x44,0x82,0x82,0x44,0x38,0x00,0x00,0x00 }, /* o */
	{ 0x00,0x00,0x00,0xDC,0x62,0x42,0x42,0x62,0x5C,0x40,0x40,0xC0 }, /* p */
	{ 0x00,0x00,0x00,0x76,0x8C,0x84,0x84,0x8C,0x74,0x04,0x04,0x06 }, /* q */
	{ 0x00,0x00,0x00,0xCC,0x52,0x60,0x40,0x40,0xE0,0x00,0x00,0x00 }, /* r */
	{ 0x00,0x00,0x00,0x3C,0x40,0x3C,0x02,0x82,0x7C,0x00,0x00,0x00 }, /* s */
	{ 0x60,0x20,0x20,0xF8,0x20,0x20,0x22,0x22,0x1C,0x00,0x00,0x00 }, /* t */
	{ 0x00,0x00,0x00,0xC4,0x44,0x44,0x44,0x44,0x3A,0x00,0x00,0x00 }, /* u */
	{ 0x00,0x00,0x00,0x82,0x82,0x82,0x44,0x28,0x10,0x00,0x00,0x00 }, /* v */
	{ 0x00,0x00,0x00,0x82,0x82,0x92,0x92,0xAA,0x44,0x00,0x00,0x00 }, /* w */
	{ 0x00,0x00,0x00,0xC6,0x28,0x10,0x10,0x28,0xC6,0x00,0x00,0x00 }, /* x */
	{ 0x00,0x00,0x00,0x86,0x84,0x84,0x84,0x8C,0x74,0x04,0x84,0x78 }, /* y */
	{ 0x00,0x00,0x00,0xFC,0x88,0x10,0x20,0x44,0xFC,0x00,0x00,0x00 }, /* z */
	{ 0x08,0x10,0x10,0x10,0x20,0x10,0x10,0x10,0x08,0x00,0x00,0x00 }, /* { */
	{ 0x10,0x10,0x10,0x10,0x00,0x10,0x10,0x10,0x10,0x00,0x00,0x00 }, /* | */
	{ 0x20,0x10,0x10,0x10,0x08,0x10,0x10,0x10,0x20,0x00,0x00,0x00 }, /* } */
	{ 0x00,0x00,0x00,0x64,0x98,0x00,0x00,0x00,0x00,0x00,0x00,0x00 }  /* ~ */
};
#if ALT_CHARSET
unsigned char TKP_Apl[94][12] = {
};
#endif

struct Tek_Font_Def TKP_Fonts[2][4] = {
	{ { &TKP_Extralarge_Font, &TKP_Ascii[0][0] },
	  { &TKP_Large_Font, &TKP_Ascii[0][0] },
	  { &TKP_Medium_Font, &TKP_Ascii[0][0] },
	  { &TKP_Small_Font, &TKP_Ascii[0][0] } },
#if ALT_CHARSET
	{ { &TKP_Extralarge_APL_Font, &TKP_Apl[0][0] },
	  { &TKP_Large_APL_Font, &TKP_Apl[0][0] },
	  { &TKP_Medium_APL_Font, &TKP_Apl[0][0] },
	  { &TKP_Small_APL_Font, &TKP_Apl[0][0] } }
#else
	{ { &TKP_Extralarge_Font, &TKP_Ascii[0][0] },
	  { &TKP_Large_Font, &TKP_Ascii[0][0] },
	  { &TKP_Medium_Font, &TKP_Ascii[0][0] },
	  { &TKP_Small_Font, &TKP_Ascii[0][0] } }
#endif
};

unsigned short TKP_Width[4], TKP_Height[4];	/* Scaled character widths and heights (pixels) */
unsigned long TKP_SP_Width[4], TKP_LF_Height[4];/* Scaled character cell sizes (1/1024 pixels) */
int TKP_Charset_Attr;				/* Orientation + mirror-font flag */

/*
 *	Some additional character definition parameters. This information
 *	from tables 3-1 and 3-2 in the Tektronix manual. 3.5 pixels is
 *	equivalent to 1 Mil.
 */

struct {
	unsigned short Cell_Width;	/* Width of space (pixels) */
	unsigned short Cell_Height;	/* Height of line feed (pixels) */
	unsigned short Char_Width;	/* Width of character raster within cell (Mils) */
	unsigned short Char_Height;	/* Height of character raster within cell (Mils) */
	unsigned short Chars_Per_Line;	/* Number of characters per line */
	unsigned short Lines_Per_Page;	/* Number of lines per page */
} TKP_Charsizes[4] = {
	{ 56, 88, 160, 195,  74, 35 },	/* Extralarge */
	{ 51, 82, 145, 175,  81, 38 },	/* Large */
	{ 34, 53,  95, 115, 121, 58 },	/* Medium */
	{ 31, 48,  90, 110, 133, 64 }	/* Small */
};

/*
 *	The hardware linestyle definitions:
 */

unsigned int TKP_Solid_Line[1] = { 0 };
unsigned int TKP_Dotted_Line[] = { 2, 8, 8 };
unsigned int TKP_Dotdash_Line[] = { 4, 40, 8, 8, 8 };
unsigned int TKP_Shortdash_Line[] = { 2, 24, 8 };
unsigned int TKP_Longdash_Line[] = { 2, 48, 16 };
unsigned int TKP_Dotdotdotdash_Line[] = { 8, 8, 8, 8, 8, 8, 8, 80, 8 }; /* 4100 series linestyles */
unsigned int TKP_Long_Dotdash_Line[] = { 4, 14, 14, 86, 14 };
unsigned int TKP_Widedash_Line[] = { 2, 32, 32 };
unsigned int *TKP_Linestyles[8] = {
	&TKP_Solid_Line[0],
	&TKP_Dotted_Line[0],
	&TKP_Dotdash_Line[0],
	&TKP_Shortdash_Line[0],
	&TKP_Longdash_Line[0],
	&TKP_Solid_Line[0],
	&TKP_Solid_Line[0],
	&TKP_Solid_Line[0]
};
unsigned long TKP_Linewidth[2];
unsigned long TKP_Pointscale;

/*
 *	Tektronix operating mode definitions. These numbers should NOT
 *	be changed, as they represent the relative heirarchy of the
 *	modes:
 */

int TKP_Mode;
#define ALPHA_MODE         0
#define GRAPH_MODE         1
#define POINT_MODE         2
#define SPECIAL_POINT_MODE 3
#define INCR_MODE          4

pointer TKP_File;
int TKP_PB_Char;

/*
 *	The parameters to TK_Paste specify the output device descriptor,
 *	the location of the upper left corner of the output box, the
 *	width, depth and orientation of the output box, the relative
 *	picture number and the file name containing the graphical input
 *	data. All dimensions are in 1/1024 pixels.
 *
 *	The 'orientation' option allows the picture to be pasted into
 *	the box in any of four orientations, corresponding to rotations
 *	of multiples of 90 degrees.
 */

int TK_Paste (Device_Ptr, File, X0, Y0, X_Scale, Y_Scale, Orientation, Picture,
	      Optv, Optc)
struct Device_Table *Device_Ptr;
char *File;
long X0, Y0;
long X_Scale, Y_Scale;
unsigned int Orientation, Picture;
char *Optv[];
int Optc;
{
	auto   struct Device_Table *Dev_Ptr;
	auto   struct Attributes *Attr_Ptr;
	auto   long Beam_X, Beam_Y, Temp;
	auto   int Index, Jndex;
	auto   unsigned char c, Point_Int_Char;
	static struct Attributes Current_Attributes;
	extern int TKP_Rcv_Addr_Byte();
	extern unsigned short TKP_Parse_Options();
	msgcode DVIOUT_INOPENFAIL;

	Dev_Ptr = Device_Ptr;
	Attr_Ptr = &Current_Attributes;
/*
 *	Parse the input options; add defaults:
 */
	TKP_Options = TKP_Parse_Options (Optv, Optc);
/*
 *	Open the graphics file for input:
 */
	if ((TKP_File = Open_Gen_File_M (File, ".TEK", "r", 0)) == 0) {
		Message (DVIOUT_INOPENFAIL, File, 1);
		return (0);
	}
/*
 *	Set transformation values:
 */
	switch (Orientation) {

	case 0:		/* Up normal */
		Attr_Ptr->Matrix[0].Numerator = X_Scale;
		Attr_Ptr->Matrix[0].Denominator = RASTER_X;
		Attr_Ptr->Matrix[1].Numerator = 0;
		Attr_Ptr->Matrix[1].Denominator = 1;
		Attr_Ptr->Matrix[2].Numerator = 0;
		Attr_Ptr->Matrix[2].Denominator = 1;
		Attr_Ptr->Matrix[3].Numerator = -Y_Scale;
		Attr_Ptr->Matrix[3].Denominator = RASTER_Y;
		Attr_Ptr->Offset[0] = X0;
		Attr_Ptr->Offset[1] = Y_Scale + Y0;
		break;

	case 1:		/* Up is left */
		Attr_Ptr->Matrix[0].Numerator = 0;
		Attr_Ptr->Matrix[0].Denominator = 1;
		Attr_Ptr->Matrix[1].Numerator = -Y_Scale;
		Attr_Ptr->Matrix[1].Denominator = RASTER_X;
		Attr_Ptr->Matrix[2].Numerator = -X_Scale;
		Attr_Ptr->Matrix[2].Denominator = RASTER_Y;
		Attr_Ptr->Matrix[3].Numerator = 0;
		Attr_Ptr->Matrix[3].Denominator = 1;
		Attr_Ptr->Offset[0] = X_Scale + X0;
		Attr_Ptr->Offset[1] = Y_Scale + Y0;
		break;

	case 2:		/* Up is down */
		Attr_Ptr->Matrix[0].Numerator = -X_Scale;
		Attr_Ptr->Matrix[0].Denominator = RASTER_X;
		Attr_Ptr->Matrix[1].Numerator = 0;
		Attr_Ptr->Matrix[1].Denominator = 1;
		Attr_Ptr->Matrix[2].Numerator = 0;
		Attr_Ptr->Matrix[2].Denominator = 1;
		Attr_Ptr->Matrix[3].Numerator = Y_Scale;
		Attr_Ptr->Matrix[3].Denominator = RASTER_Y;
		Attr_Ptr->Offset[0] = X_Scale + X0;
		Attr_Ptr->Offset[1] = Y0;
		break;

	case 3:		/* Up is right */
		Attr_Ptr->Matrix[0].Numerator = 0;
		Attr_Ptr->Matrix[0].Denominator = 1;
		Attr_Ptr->Matrix[1].Numerator = Y_Scale;
		Attr_Ptr->Matrix[1].Denominator = RASTER_X;
		Attr_Ptr->Matrix[2].Numerator = X_Scale;
		Attr_Ptr->Matrix[2].Denominator = RASTER_Y;
		Attr_Ptr->Matrix[3].Numerator = 0;
		Attr_Ptr->Matrix[3].Denominator = 1;
		Attr_Ptr->Offset[0] = X0;
		Attr_Ptr->Offset[1] = Y0;
		break;
	}
/*
 *	Set line width and point scale factors. This is set up so that
 *	de-focused lines and points are 50 percent wider than regular
 *	lines and points. The normal Tektronix beam width is about 3-4
 *	pixels wide, and gets wider as the terminal ages. Two pixels
 *	wide seems to be a good compromise between terminal specifications
 *	and reality. For points output using point plot, special point
 *	plot and incremental plot modes, the maximum point size (100 percent
 *	intensity) is set to twice as large as de-focused line widths.
 */
	TKP_Charset_Attr = Orientation & 0x03;
	if ((Temp = X_Scale) < 0) {
		Temp = -Temp;
		TKP_Charset_Attr |= 0x04;
	}
	TKP_Linewidth[0] = XN_Div_D_T_M (1, Temp, RASTER_X);
	TKP_Linewidth[1] = XN_Div_D_T_M (3, Temp, RASTER_X << 1);
	TKP_Pointscale = XN_Div_D_T_M (3, Temp, RASTER_X);
/*
 *	Determine size of hardware character cells; initialize
 *	font definitions:
 */
	if ((Orientation & 0x01) == 0)
		TKP_Set_Font_Params (Temp, Y_Scale);
	else
		TKP_Set_Font_Params (Y_Scale, Temp);
/*
 *	Initialize beam position at the home position; internal
 *	'registers':
 */
	Attr_Ptr->CurX = HOME_X;
	Attr_Ptr->CurY = HOME_Y;
	TKP_Map_XY (Attr_Ptr);
	Attr_Ptr->HiY = 0;
	Attr_Ptr->ExB = 0;
	Attr_Ptr->LoY = 0;
	Attr_Ptr->HiX = 0;
	Attr_Ptr->LoX = 0;
	Point_Int_Char = 0x57;	/* 6 percent */
/*
 *	Set initial mode as alpha, character size to 'extralarge',
 *	linestyle to 'solid':
 */
	TKP_Mode = ALPHA_MODE;
	Attr_Ptr->Charsize = 0;
	Attr_Ptr->Linestyle = 0;
	Attr_Ptr->Writing_Mode = NORMAL;
	Attr_Ptr->Flags = 0;
	if ((TKP_Options & ALTCHR) != 0)
		Attr_Ptr->Flags |= ALT_CHARS;
/*
 *	Set initial conditions at device:
 */
	(*Dev_Ptr->Set_Linewidth) (TKP_Linewidth[Attr_Ptr->Writing_Mode]);
	(*Dev_Ptr->Set_Linestyle) (TKP_Linestyles[Attr_Ptr->Linestyle][0],
				   &TKP_Linestyles[Attr_Ptr->Linestyle][1], 0);
/*
 *	Begin reading in characters. The control sequences US (or ESC US), CR,
 *	ESC FF and ESC SUB always have the same effect, no matter which mode the
 *	terminal is in. Control characters GS, FS and RS always change mode, if
 *	the current mode has a lower heirarchy.
 */
	TKP_PB_Char = -1;
	while ((c = (TKP_PB_Char >= 0) ? TKP_PB_Char : Read_Gen_Character_M (TKP_File) & 0x7F) != 0 ||
			Gen_At_EOF_M (TKP_File) == 0) {
		TKP_PB_Char = -1;
		switch (c) {

		case BEL:				/* Beep */
			Attr_Ptr->Flags &= ~BYPASS;
			break;

		case LF:
			if ((TKP_Options & LFCR) != 0)
				TKP_PB_Char = CR;
			Attr_Ptr->Flags &= ~BYPASS;
			break;

		case CR:
			TKP_Set_Alpha (Attr_Ptr);
			if ((TKP_Options & CRLF) != 0)
				TKP_PB_Char = LF;
			break;

		case SI:
			Attr_Ptr->Flags &= ~BYPASS;
			continue;

		case US:
			TKP_Set_Alpha (Attr_Ptr);
			continue;

		case GS:
			TKP_Set_Mode (Attr_Ptr, GRAPH_MODE);
			continue;

		case FS:
			if ((TKP_Options & EGM) != 0)
				TKP_Set_Mode (Attr_Ptr, POINT_MODE);
			continue;

		case RS:
			if ((TKP_Options & EGM) != 0)
				TKP_Set_Mode (Attr_Ptr, INCR_MODE);
			continue;

		case ESC:
			do {
				c = Read_Gen_Character_M (TKP_File) & 0x7F;
				if (c == 0 && Gen_At_EOF_M (TKP_File) != 0) {
					TKP_PB_Char = 0;
					break;
				}
			} while (c == ESC || c == LF || c == CR || c == 0);
			switch (c) {

			case ENQ:
				if ((Attr_Ptr->Flags & CROSSHAIRS) != 0)
					TKP_Set_Alpha (Attr_Ptr);
				Attr_Ptr->Flags |= BYPASS;
				continue;

			case BEL:
				Attr_Ptr->Flags &= ~BYPASS;
				break;

			case FF:
				Attr_Ptr->CurX = HOME_X;
				Attr_Ptr->CurY = HOME_Y;
				TKP_Map_XY (Attr_Ptr);
				Attr_Ptr->ExB = 0;
				Attr_Ptr->Flags &= ~(MARGIN_2 | PAGE_FULL);
				TKP_Set_Alpha (Attr_Ptr);
				continue;

			case SO:
				if ((TKP_Options & (ALTCHR | ASCIICHR)) == 0)
					Attr_Ptr->Flags |= ALT_CHARS;
				continue;

			case SI:
				if ((TKP_Options & (ALTCHR | ASCIICHR)) == 0)
					Attr_Ptr->Flags &= ~ALT_CHARS;
				continue;

			case ETB:
				Attr_Ptr->Flags &= ~BYPASS;
				continue;

			case CAN:
				Attr_Ptr->Flags |= BYPASS;
				continue;

			case SUB:
				Attr_Ptr->Flags |= (CROSSHAIRS | BYPASS);
				continue;

			case GS:
				TKP_Set_Mode (Attr_Ptr, GRAPH_MODE);
				continue;

			case FS:
				if ((TKP_Options & EGM) != 0)
					TKP_Set_Mode (Attr_Ptr, SPECIAL_POINT_MODE);
				continue;

			case RS:
				if ((TKP_Options & EGM) != 0)
					TKP_Set_Mode (Attr_Ptr, INCR_MODE);
				continue;

			case US:
				TKP_Set_Alpha (Attr_Ptr);
				continue;

			default:
				if (c == 0x3F && (TKP_Options & ALTDEL) != 0) {	/* Alternate DEL using ESC ? */
					c = DEL;
					break;
				}
				TKP_Set_LCE (c, Attr_Ptr, Dev_Ptr);
				continue;
			}
		}
/*
 *	Process the character in the context of the current mode:
 */
		switch (TKP_Mode) {

		case ALPHA_MODE:
			switch (c) {

			case BS:
				TKP_Advance_Left (Attr_Ptr);
				break;

			case HT: case SP:
				TKP_Advance_Right (Attr_Ptr);
				break;

			case LF:
				Attr_Ptr->Flags &= ~BYPASS;
				TKP_Advance_Down (Attr_Ptr);
				break;

			case VT:
				TKP_Advance_Up (Attr_Ptr);
				break;

			case CR:
				Attr_Ptr->Flags &= ~BYPASS;
				TKP_Advance_To_Margin (Attr_Ptr);
				break;

			default:
				if (c > SP)
					TKP_Process_Alpha_Input (c, Attr_Ptr, Dev_Ptr);
			}
			break;

		case SPECIAL_POINT_MODE:
			if ((Attr_Ptr->Flags & FIRST_OP) != 0) {
				if (c >= 0x20 && c <= 0x7D) {
					Point_Int_Char = c;
					Attr_Ptr->Flags &= ~FIRST_OP;
				}
				continue;
			}

		case POINT_MODE:
		case GRAPH_MODE:
			if (c == BEL)
				Attr_Ptr->Flags &= ~FIRST_OP;
			else if (c == LF) {		/* Mini line-feed */
				if (Attr_Ptr->CurY < 6) {
					Attr_Ptr->CurY = HOME_Y;
					Attr_Ptr->Flags ^= MARGIN_2;
				} else
					Attr_Ptr->CurY -= 6;
				TKP_Map_XY (Attr_Ptr);
			} else if (c >= SP && c <= DEL && TKP_Rcv_Addr_Byte (c, Attr_Ptr) != 0) {	/* Execute */
				Beam_X = Attr_Ptr->Cur_Mapped_X;
				Beam_Y = Attr_Ptr->Cur_Mapped_Y;
				Attr_Ptr->CurX = (((Attr_Ptr->HiX << 5) + Attr_Ptr->LoX) << 2) +
						  (Attr_Ptr->ExB & 0x03);
				Attr_Ptr->CurY = (((Attr_Ptr->HiY << 5) + Attr_Ptr->LoY) << 2) +
						  (Attr_Ptr->ExB >> 2);
				TKP_Map_XY (Attr_Ptr);
				if (TKP_Mode != GRAPH_MODE)	/* Point or Special Point */
					TKP_Plot_Point (Attr_Ptr, Point_Int_Char, Dev_Ptr);
				else if ((Attr_Ptr->Flags & FIRST_OP) == 0 && Attr_Ptr->Writing_Mode != WRITE_THRU)
					(*Dev_Ptr->Typeset_Line) (Beam_X, Beam_Y,
								  Attr_Ptr->Cur_Mapped_X, Attr_Ptr->Cur_Mapped_Y);
				Attr_Ptr->Flags &= ~(MARGIN_2 | PAGE_FULL | SEEN_EXB | SEEN_LOY | FIRST_OP | BYPASS);
			}
			break;

		case INCR_MODE:
			if (c >= SP && c <= 0x7D) {
				Beam_X = (long) Attr_Ptr->CurX;
				Beam_Y = (long) Attr_Ptr->CurY;
				if ((c & INCR_EAST) != 0)
					Beam_X += 1;
				if ((c & INCR_WEST) != 0)
					Beam_X -= 1;
				if ((c & INCR_NORTH) != 0)
					Beam_Y += 1;
				if ((c & INCR_SOUTH) != 0)
					Beam_Y -= 1;
				if (Beam_X < 0)
					Beam_X += RASTER_X;
				else if (Beam_X >= RASTER_X)
					Beam_X -= RASTER_X;
				if (Beam_Y < 0) {
					Beam_Y += RASTER_X;
					Attr_Ptr->Flags ^= MARGIN_2;
				} else if (Beam_Y >= RASTER_X)
					Beam_Y -= RASTER_X;
				if ((c & INCR_PUP) != 0)
					Attr_Ptr->Flags &= ~PEN_DOWN;
				if ((c & INCR_PDOWN) != 0)
					Attr_Ptr->Flags |= PEN_DOWN;
				Attr_Ptr->CurX = (unsigned short) Beam_X;
				Attr_Ptr->CurY = (unsigned short) Beam_Y;
				TKP_Map_XY (Attr_Ptr);
				if ((Attr_Ptr->Flags & PEN_DOWN) != 0)
					TKP_Plot_Point (Attr_Ptr, Point_Int_Char, Dev_Ptr);
			}
			break;
		}
	}
/*
 *	Finish up:
 */
	for (Index = 0; Index < 4; Index++) {
		if ((TKP_Fonts[0][Index].Font->Flags & DOWNLOADED) != 0) {
			for (Jndex = 33; Jndex < 127; Jndex++)
				Mem_Free (TKP_Fonts[0][Index].Font->Font_Directory[Jndex]->Pixel_Array);
		}
#if ALT_CHARSET
		if ((TKP_Fonts[1][Index].Font->Flags & DOWNLOADED) != 0) {
			for (Jndex = 33; Jndex < 127; Jndex++)
				Mem_Free (TKP_Fonts[1][Index].Font->Font_Directory[Jndex]->Pixel_Array);
		}
#endif
	}
	Close_Gen_File_M (TKP_File);
	return (1);
}

TKP_Set_Alpha (Attribute_Ptr)
struct Attributes *Attribute_Ptr;
{
	auto   struct Attributes *Attr_Ptr;

	Attr_Ptr = Attribute_Ptr;
	Attr_Ptr->Flags &= ~(BYPASS | PEN_DOWN);
	if (Attr_Ptr->CurY > HOME_Y) {
		Attr_Ptr->CurY = HOME_Y;
		TKP_Map_XY (Attr_Ptr);
	}
	TKP_Mode = ALPHA_MODE;
}

TKP_Set_LCE (c, Attr_Ptr, Dev_Ptr)
unsigned char c;
struct Attributes *Attr_Ptr;
struct Device_Table *Dev_Ptr;
{
	auto   unsigned char c1;

	if (TKP_Mode == ALPHA_MODE) {
		if (c == BS)
			TKP_Advance_Left (Attr_Ptr);
		else if (c == HT)
			TKP_Advance_Right (Attr_Ptr);
		else if (c == VT)
			TKP_Advance_Up (Attr_Ptr);
	}
	if (c >= 0x38 && c <= 0x3B)
		Attr_Ptr->Charsize = c - 0x38;
	else if (c >= 0x60 && c <= 0x77) {
		if ((c1 = (c >> 3) & 0x03) != Attr_Ptr->Writing_Mode) {
			(*Dev_Ptr->Set_Linewidth) (TKP_Linewidth[c1]);
			Attr_Ptr->Writing_Mode = c1;
		}
		if ((TKP_Options & EGM) != 0 && (c1 = c & 0x07) != Attr_Ptr->Linestyle) {
			(*Dev_Ptr->Set_Linestyle) (TKP_Linestyles[c1][0], &TKP_Linestyles[c1][1], 0);
			Attr_Ptr->Linestyle = c1;
		}
	}
}

TKP_Set_Mode (Attr_Ptr, Mode)
struct Attributes *Attr_Ptr;
int Mode;
{
	if (TKP_Mode <= Mode) {
		TKP_Mode = Mode;
		Attr_Ptr->Flags |= FIRST_OP;
	}
}

/*
 *	Plot a point for point plot, special point plot, or
 *	incremental mode. On the 4014 used to check out these
 *	functions, the de-focused bit of the intensity
 *	character did not seem to have any affect. Therefore,
 *	a writing mode set to de-focused was used instead to
 *	determine the point size, since this WAS observed to
 *	have an effect.
 */

TKP_Plot_Point (Attribute_Ptr, Int_Char, Dev_Ptr)
struct Attributes *Attribute_Ptr;
unsigned char Int_Char;
struct Device_Table *Dev_Ptr;
{
	auto   struct Attributes *Attr_Ptr;
	auto   unsigned long Point_Radius;
	auto   unsigned int Point_Intensity;
	static unsigned char Int_Settings[64] = {
		0, 1, 1, 1, 1, 1, 1, 2, 2, 2, 2, 2, 3, 3, 3, 3,
		4, 4, 4, 5, 5, 5, 6, 6, 7, 8, 9, 10, 11, 12, 12, 13,
		14, 16, 17, 19, 20, 22, 23, 25, 28, 31, 34, 38, 41, 44, 47, 50,
		56, 62, 69, 75, 81, 88, 94, 100, 56, 62, 69, 75, 81, 88, 94, 100
	};

	Attr_Ptr = Attribute_Ptr;
	if ((Point_Intensity = Int_Settings[Int_Char&0x3F]) != 0 && Attr_Ptr->Writing_Mode != WRITE_THRU) {
		Point_Radius = XN_Div_D_T_M (TKP_Pointscale, Point_Intensity, 100);
/*		if ((Int_Char & 0x40) == 0)	*/
		if (Attr_Ptr->Writing_Mode == DEFOCUSED)
			Point_Radius += Point_Radius >> 1;
		(*Dev_Ptr->Typeset_Point) (Attr_Ptr->Cur_Mapped_X, Attr_Ptr->Cur_Mapped_Y,
					   Point_Radius << 1);
	}
}

int TKP_Rcv_Addr_Byte (ac, Attribute_Ptr)
unsigned char ac;
struct Attributes *Attribute_Ptr;
{
	auto   struct Attributes *Attr_Ptr;
	auto   int Execute;
	auto   unsigned char c;

	Attr_Ptr = Attribute_Ptr;
	Execute = 0;
	c = ac;
	if (c >= 0x60) {		/* Extra byte or Lo-Y */
		if ((TKP_Options & EGM) == 0 || (Attr_Ptr->Flags & SEEN_LOY) == 0) {
			Attr_Ptr->LoY = c & ~0x60;
			Attr_Ptr->Flags |= SEEN_LOY;
		} else if ((Attr_Ptr->Flags & SEEN_EXB) == 0) {
			Attr_Ptr->ExB = Attr_Ptr->LoY & ~0x10;
			if ((Attr_Ptr->LoY & 0x10) != 0)
				Attr_Ptr->Flags |= MARGIN_2;
			Attr_Ptr->LoY = c & ~0x60;
			Attr_Ptr->Flags |= SEEN_EXB;
		} else {
			Attr_Ptr->LoY = c & ~0x60;
			Attr_Ptr->Flags &= ~SEEN_EXB;
		}
	} else if (c >= 0x40) {			/* Lo-X */
		Attr_Ptr->LoX = c & ~0x40;
		Execute++;
	} else if (c >= 0x20 && c <= 0x3F) {	/* Hi-Y or Hi-X */
		if ((Attr_Ptr->Flags & SEEN_LOY) == 0)
			Attr_Ptr->HiY = c & ~0x20;
		else {
			Attr_Ptr->HiX = c & ~0x20;
			Attr_Ptr->Flags &= ~(SEEN_EXB | SEEN_LOY);
		}
	}
	return (Execute);
}

/*
 *	Routine TKP_Process_Alpha_Input collects characters to be
 *	output in alpha mode, then outputs them to the device:
 */

TKP_Process_Alpha_Input (c, Attribute_Ptr, Dev_Ptr)
unsigned char c;
struct Attributes *Attribute_Ptr;
struct Device_Table *Dev_Ptr;
{
	auto   struct Attributes *Attr_Ptr;
	auto   long Save_X, Save_Y, Beam_X, Beam_Y;
	auto   int Index;
	auto   unsigned char c1;
	static unsigned char Alpha_String[MAX_CPL];

	Attr_Ptr = Attribute_Ptr;
	c1 = c;
	Beam_X = Attr_Ptr->Cur_Mapped_X;
	Beam_Y = Attr_Ptr->Cur_Mapped_Y;
	if ((Attr_Ptr->Flags & MARGIN_2) != 0) {
		Save_X = Beam_X;
		Save_Y = Beam_Y;
		Attr_Ptr->CurX += RASTER_X >> 1;
		TKP_Map_XY (Attr_Ptr);
		Beam_X = Attr_Ptr->Cur_Mapped_X;
		Beam_Y = Attr_Ptr->Cur_Mapped_Y;
		Attr_Ptr->Cur_Mapped_X = Save_X;
		Attr_Ptr->Cur_Mapped_Y = Save_Y;
		Attr_Ptr->CurX -= RASTER_X >> 1;
	}
	Index = 0;
	do
	if (c1 < DEL && (Attr_Ptr->Flags & (BYPASS | PAGE_FULL)) == 0) {
		if (Attr_Ptr->Writing_Mode != WRITE_THRU)
			Alpha_String[Index++] = c1;
		TKP_Advance_Right (Attr_Ptr);
		if (Attr_Ptr->Cur_Mapped_Y != Beam_Y || Index >= MAX_CPL) {
			if (Index > 0)
				TKP_Set_String (Beam_X, Beam_Y, Alpha_String, Index, Attr_Ptr, Dev_Ptr);
			Beam_X = Attr_Ptr->Cur_Mapped_X;
			Beam_Y = Attr_Ptr->Cur_Mapped_Y;
			if ((Attr_Ptr->Flags & MARGIN_2) != 0) {
				Save_X = Beam_X;
				Save_Y = Beam_Y;
				Attr_Ptr->CurX += RASTER_X >> 1;
				TKP_Map_XY (Attr_Ptr);
				Beam_X = Attr_Ptr->Cur_Mapped_X;
				Beam_Y = Attr_Ptr->Cur_Mapped_Y;
				Attr_Ptr->Cur_Mapped_X = Save_X;
				Attr_Ptr->Cur_Mapped_Y = Save_Y;
				Attr_Ptr->CurX -= RASTER_X >> 1;
			}
			Index = 0;
		}
	}
	while ((c1 = Read_Gen_Character_M (TKP_File) & 0x7F) > SP);
	TKP_PB_Char = c1;
	if (Index > 0)
		TKP_Set_String (Beam_X, Beam_Y, Alpha_String, Index, Attr_Ptr, Dev_Ptr);
}

/*
 *	Routine TKP_Set_String outputs the specified character
 *	string at the specified position:
 */

TKP_Set_String (X, Y, String, Count, Attribute_Ptr, Dev_Ptr)
long X, Y;
unsigned char String[];
int Count;
struct Attributes *Attribute_Ptr;
struct Device_Table *Dev_Ptr;
{
	auto   struct Attributes *Attr_Ptr;
	auto   struct Tek_Font_Def *Tek_Font_Ptr;
	auto   struct Font_Definition *Font_Ptr;
	auto   struct Char_Definition *Char_Ptr;
	auto   long Cur_X, Start_X;
	auto   int Index, Jndex;
	static struct Char_Definition *Char_Vector[MAX_CPL];

	Attr_Ptr = Attribute_Ptr;
	Tek_Font_Ptr = &TKP_Fonts[((Attr_Ptr->Flags & ALT_CHARS) == 0) ? 0 : 1][Attr_Ptr->Charsize];
	Font_Ptr = Tek_Font_Ptr->Font;
	if ((Font_Ptr->Flags & DOWNLOADED) == 0) {
		Index = Attr_Ptr->Charsize;
		TKP_Gen_Charset (Tek_Font_Ptr, TKP_Width[Index], TKP_Height[Index],
				 TKP_SP_Width[Index], TKP_LF_Height[Index], TKP_Charset_Attr & 0x03,
				 TKP_Charset_Attr & 0x04);
		(*Dev_Ptr->Download_Font) (Font_Ptr, &Font_Ptr->Font_Directory[33], 94);
		Font_Ptr->Flags |= DOWNLOADED;
	}
	Start_X = Cur_X = X;
	Jndex = 0;
	for (Index = 0; Index < Count; Index++) {
		Char_Ptr = Font_Ptr->Font_Directory[String[Index]];
		if (Char_Ptr->Driver_Id != 0)
			Char_Vector[Jndex++] = Char_Ptr;
		else {
			if (Jndex > 0) {
				(*Dev_Ptr->Typeset_String) (Start_X, Y, Font_Ptr, Char_Vector, Jndex);
				Jndex = 0;
			}
			(*Dev_Ptr->Typeset_Pixel) (Cur_X - ((long) Char_Ptr->X_Origin << 10),
						   Y - ((long) Char_Ptr->Y_Origin << 10),
						   Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height,
						   Char_Ptr->Pixel_Array);
			Start_X = Cur_X + Char_Ptr->H_Escapement;
		}
		Cur_X += Char_Ptr->H_Escapement;
	}
	if (Jndex > 0)
		(*Dev_Ptr->Typeset_String) (Start_X, Y, Font_Ptr, Char_Vector, Jndex);
}

/*
 *	Routines TKP_Advance_Up, TKP_Advance_Down, TKP_Advance_Left,
 *	TKP_Advance_Right and TKP_Advance_To_Margin move the alpha cursor
 *	position in alpha mode:
 */

TKP_Advance_Up (Attribute_Ptr)
struct Attributes *Attribute_Ptr;
{
	auto   struct Attributes *Attr_Ptr;
	auto   unsigned short y;

	Attr_Ptr = Attribute_Ptr;
	if ((Attr_Ptr->Flags & (PAGE_FULL | BYPASS)) == 0) {
		y = TKP_Charsizes[Attr_Ptr->Charsize].Cell_Height;
		if ((Attr_Ptr->CurY += y) > HOME_Y)
			Attr_Ptr->CurY = HOME_Y;
		TKP_Map_XY (Attr_Ptr);
	}
}

TKP_Advance_Down (Attribute_Ptr)
struct Attributes *Attribute_Ptr;
{
	auto   struct Attributes *Attr_Ptr;
	auto   unsigned short y;

	Attr_Ptr = Attribute_Ptr;
	y = TKP_Charsizes[Attr_Ptr->Charsize].Cell_Height;
	if (Attr_Ptr->CurY < y) {
		if ((Attr_Ptr->Flags & MARGIN_2) == 0) {
			if ((TKP_Options & MRGONE) != 0)
				Attr_Ptr->Flags |= PAGE_FULL;
		} else if ((TKP_Options & MRGTWO) != 0)
			Attr_Ptr->Flags |= PAGE_FULL;
	}
	if ((Attr_Ptr->Flags & (PAGE_FULL | BYPASS)) == 0) {
		if (Attr_Ptr->CurY < y) {
			Attr_Ptr->CurY = HOME_Y;
			Attr_Ptr->Flags ^= MARGIN_2;
		} else
			Attr_Ptr->CurY -= y;
		TKP_Map_XY (Attr_Ptr);
	}
}

TKP_Advance_Left (Attribute_Ptr)
struct Attributes *Attribute_Ptr;
{
	auto   struct Attributes *Attr_Ptr;
	auto   unsigned short x;

	Attr_Ptr = Attribute_Ptr;
	if ((Attr_Ptr->Flags & (PAGE_FULL | BYPASS)) == 0) {
		x = TKP_Charsizes[Attr_Ptr->Charsize].Cell_Width;
		if (Attr_Ptr->CurX < x)
			Attr_Ptr->CurX += ((Attr_Ptr->Flags & MARGIN_2) == 0) ? RASTER_X : RASTER_X >> 1;
		Attr_Ptr->CurX -= x;
		TKP_Map_XY (Attr_Ptr);
	}
}

TKP_Advance_Right (Attribute_Ptr)
struct Attributes *Attribute_Ptr;
{
	auto   struct Attributes *Attr_Ptr;
	auto   unsigned short x, Margin;

	Attr_Ptr = Attribute_Ptr;
	if ((Attr_Ptr->Flags & (PAGE_FULL | BYPASS)) == 0) {
		x = TKP_Charsizes[Attr_Ptr->Charsize].Cell_Width;
		Margin = ((Attr_Ptr->Flags & MARGIN_2) == 0) ? RASTER_X : RASTER_X >> 1;
		if (Attr_Ptr->CurX >= Margin - x) {
			TKP_Advance_To_Margin (Attr_Ptr);	/* CR */
			TKP_Advance_Down (Attr_Ptr);		/* LF */
		} else {
			Attr_Ptr->CurX += x;
			TKP_Map_XY (Attr_Ptr);
		}
	}
}

TKP_Advance_To_Margin (Attribute_Ptr)
struct Attributes *Attribute_Ptr;
{
	auto   struct Attributes *Attr_Ptr;

	Attr_Ptr = Attribute_Ptr;
	if ((Attr_Ptr->Flags & (PAGE_FULL | BYPASS)) == 0) {
		Attr_Ptr->CurX = 0;
		TKP_Map_XY (Attr_Ptr);
	}
}

TKP_Map_XY (Attribute_Ptr)
struct Attributes *Attribute_Ptr;
{
	auto   struct Attributes *Attr_Ptr;
	auto   unsigned short x, y;

	Attr_Ptr = Attribute_Ptr;
	x = Attr_Ptr->CurX;
	y = Attr_Ptr->CurY;
	Attr_Ptr->Cur_Mapped_X = XN_Div_D_T_M (x, Attr_Ptr->Matrix[0].Numerator, Attr_Ptr->Matrix[0].Denominator) +
				 XN_Div_D_T_M (y, Attr_Ptr->Matrix[2].Numerator, Attr_Ptr->Matrix[2].Denominator) +
				 Attr_Ptr->Offset[0];
	Attr_Ptr->Cur_Mapped_Y = XN_Div_D_T_M (x, Attr_Ptr->Matrix[1].Numerator, Attr_Ptr->Matrix[1].Denominator) +
				 XN_Div_D_T_M (y, Attr_Ptr->Matrix[3].Numerator, Attr_Ptr->Matrix[3].Denominator) +
				 Attr_Ptr->Offset[1];
}

TKP_Set_Font_Params (X_Scale, Y_Scale)
long X_Scale, Y_Scale;
{
	auto   struct Font_Definition *Font_Ptr_1, *Font_Ptr_2;
	auto   unsigned long Size;
	auto   unsigned int Index;
	extern long Unconvert();

	for (Index = 0; Index < 4; Index++) {
		TKP_Width[Index] = (unsigned short) XN_Div_D_R_M (TKP_Charsizes[Index].Char_Width,
								  X_Scale, (35*RASTER_X/10)<<10);
		TKP_Height[Index] = (unsigned short) XN_Div_D_R_M (TKP_Charsizes[Index].Char_Height,
								   Y_Scale, (105*RASTER_Y/40)<<10);
		TKP_SP_Width[Index] = XN_Div_D_R_M (TKP_Charsizes[Index].Cell_Width, X_Scale, RASTER_X);
		TKP_LF_Height[Index] = XN_Div_D_R_M (TKP_Charsizes[Index].Cell_Height, Y_Scale, RASTER_Y);
		TKP_Fonts[0][Index].Font->Flags = TEMP_FONT;
		if (TKP_Fonts[1][Index].Font != TKP_Fonts[0][Index].Font)
			TKP_Fonts[1][Index].Font->Flags = TEMP_FONT;
	}
}

/*
 *	Routine TKP_Gen_Charset builds the raster definitions for the
 *	hardware characters for a specific raster size. The raster is
 *	also rotated according to the image orientation.
 *
 *	It is not completely clear where, exactly, the origin of the
 *	Tektronix character cell is. Close examination of the terminal
 *	operation suggests the X origin is one pixel to the left of the
 *	character and the Y origin is at the top of the baseline pixel,
 *	so that's where I'm putting it.
 */

TKP_Gen_Charset (Tek_Font_Ptr, Width, Height, SP_Width, LF_Height, Orientation, Reflect)
struct Tek_Font_Def *Tek_Font_Ptr;
unsigned short Width, Height;
unsigned long SP_Width, LF_Height;
int Orientation, Reflect;
{
	auto   struct Font_Definition *Font_Ptr;
	auto   unsigned char *Scratch_Raster;
	auto   int Index;
	extern struct Ratio DVI_Fraction;
	extern char *Mem_Alloc();
	extern long Convert(), Unconvert();
/*
 *	Initialize:
 */
	Font_Ptr = Tek_Font_Ptr->Font;
	Scratch_Raster = (unsigned char *) Mem_Alloc (Width * Height);
/*
 *	Compute font design size:
 */
	Font_Ptr->Design_Size = ((Orientation & 0x01) == 0) ? Unconvert (SP_Width) : Unconvert (LF_Height);
	Font_Ptr->At_Size = Font_Ptr->Design_Size;
	Font_Ptr->Font_Design_Size = XN_Div_D_T_M (Font_Ptr->Design_Size,
						   XN_Div_D_T_M (473628672, DVI_Fraction.Numerator,
								 DVI_Fraction.Denominator),
						   1587500);
	if (Reflect != 0)
		Font_Ptr->Flags |= MIRROR_FONT;
/*
 *	For each character in the character set, generate the scaled,
 *	rotated character definition:
 */
	for (Index = 0; Index < 94; Index++)
		TKP_Gen_Character (Font_Ptr, Tek_Font_Ptr->Char_Set[Index], Index+33, Width, Height,
				   SP_Width, LF_Height, Orientation, Scratch_Raster);
/*
 *	Set remaining font parameters:
 */
	Font_Ptr->Font_Index = Font_Count++;
	Font_Ptr->Font_Number = ++Max_Font_Number;
	Set_Pixel_Bounds (Font_Ptr);
/*
 *	Clean up and return:
 */
	Mem_Free (Scratch_Raster);
}

TKP_Gen_Character (Font_Ptr, Raster, Char_Code, Width, Height, SP_Width, LF_Height, Orientation,
		   Scratch_Raster)
struct Font_Definition *Font_Ptr;
unsigned char *Raster, *Scratch_Raster;
int Char_Code, Orientation;
unsigned short Width, Height;
unsigned long SP_Width, LF_Height;
{
	auto   struct Char_Definition *Char_Ptr;
	auto   unsigned short New_Width, New_Height;
	auto   short New_X_Origin, New_Y_Origin;
	static unsigned char Base_Raster[84], Rotated_Raster[84];
	extern char *Mem_Alloc();
	extern long Convert();
/*
 *	Rotate the standard 7 by 12 raster definition into one with the
 *	specified width, height and orientation.
 */
	Unpack_Raster_Data_M (7, 12, Raster, Base_Raster);
	TKP_Rotate_Char (Base_Raster, Rotated_Raster, Orientation);
/*
 *	Overlay the 7 by 12 (or 12 by 7) raster onto the new raster:
 */
	if ((Orientation & 0x01) == 0) {	/* Up or down */
		New_Width = Width;
		New_Height = Height;
		Overlay_Raster_M (Rotated_Raster, 7, 12, Scratch_Raster, New_Width, New_Height);
		if (Orientation == 0) {		/* Up */
			New_X_Origin = -1;
			New_Y_Origin = (short) XN_Div_D_R_M (New_Height, 8, 12);
		} else {			/* Down */
			New_X_Origin = New_Width;
			New_Y_Origin = New_Height - (short) XN_Div_D_R_M (New_Height, 8, 12) - 1;
		}
	} else {				/* Left or right */
		New_Width = Height;
		New_Height = Width;
		Overlay_Raster_M (Rotated_Raster, 12, 7, Scratch_Raster, New_Width, New_Height);
		if (Orientation == 1) {		/* Left */
			New_X_Origin = (short) XN_Div_D_R_M (New_Width, 8, 12);
			New_Y_Origin = New_Height;
		} else {			/* Right */
			New_X_Origin = New_Width - (short) XN_Div_D_R_M (New_Width, 8, 12) - 1;
			New_Y_Origin = -1;
		}
	}
/*
 *	Reduce the raster definition to the smallest possible
 *	size; check for characters too small to print; reflect
 *	the characters if called for:
 */
	Char_Ptr = Font_Ptr->Font_Directory[Char_Code];
	Reduce_Raster_M (Scratch_Raster, New_Width, New_Height, New_X_Origin, New_Y_Origin,
		         &Char_Ptr->Pixel_Width, &Char_Ptr->Pixel_Height,
			 &Char_Ptr->X_Origin, &Char_Ptr->Y_Origin);
	if (Char_Ptr->Pixel_Width == 0 || Char_Ptr->Pixel_Height == 0) {
		Char_Ptr->Pixel_Width = Char_Ptr->Pixel_Height = 1;
		*Scratch_Raster = 1;
	}
	Char_Ptr->Pixel_Array = (unsigned char *)
			Mem_Alloc (((Char_Ptr->Pixel_Width + 7) >> 3) * Char_Ptr->Pixel_Height);
	Pack_Raster_Data_M (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height, Scratch_Raster,
			    Char_Ptr->Pixel_Array);
	if ((Font_Ptr->Flags & MIRROR_FONT) == 0)
		Char_Ptr->DVI_Width = Font_Ptr->At_Size;
	else {
		Char_Ptr->DVI_Width = -(long) Font_Ptr->At_Size;
		Char_Ptr->X_Origin = Char_Ptr->Pixel_Width - Char_Ptr->X_Origin;
		Reflect_Raster_M (Char_Ptr->Pixel_Width, Char_Ptr->Pixel_Height, Char_Ptr->Pixel_Array);
	}
	Char_Ptr->H_Escapement = Convert (Char_Ptr->DVI_Width);
	Char_Ptr->V_Escapement = 0;
	Char_Ptr->Character_Code = Char_Code;
}

TKP_Rotate_Char (Raster, Rotated_Raster, Orientation)
unsigned char *Raster, *Rotated_Raster;
unsigned int Orientation;
{
	auto   int x, y, Index;
/*
 *	Rotate the 7 by 12 raster:
 */
	for (y = 0; y < 12; y++)
	for (x = 0; x < 7; x++) {
		switch (Orientation) {

		case 0: Index = y * 7 + x; break;
		case 1: Index = (6 - x) * 12 + y; break;
		case 2: Index = (11 - y) * 7 + 6 - x; break;
		case 3: Index = x * 12 + 11 - y; break;
		}
		Rotated_Raster[Index] = Raster[y*7+x];
	}
}

/*
 *	Routine TKP_Parse_Options parses the input option list.
 *	Note that the input option values are destroyed during
 *	the parse.
 */

struct Option_Table {
	char *Option_Name;
	char *Value_List;
	unsigned short Mask;
	unsigned char Min_Len;
	unsigned char Negatable;
};

unsigned short TKP_Parse_Options (Optv, Optc)
char *Optv[];
int Optc;
{
	auto   int Index;
	auto   unsigned short Option_Mask;
	static struct Option_Table Select_List[] = {
		{ "ALT",     0, ALTCHR, 2, 0 },
		{ "ASCII",   0, ASCIICHR, 2, 0 },
		{ "PROGRAM", 0, 0x0000, 1, 0 },
		{ 0 }
	}, Margin_List[] = {
		{ "1",   0, MRGONE, 1, 0 },
		{ "2",   0, MRGTWO, 1, 0 },
		{ "OFF", 0, 0x0000, 2, 0 },
		{ "ONE", 0, MRGONE, 2, 0 },
		{ "TWO", 0, MRGTWO, 1, 0 },
		{ 0 }
	}, Option_List[] = {
		{ "ALTDEL", 0, ALTDEL, 1, 1 },
		{ "CRLF",   0, CRLF, 1, 1 },
		{ "EGM",    0, EGM, 1, 1 },
		{ "LFCR",   0, LFCR, 1, 1 },
		{ "MARGIN", Margin_List, MARGIN, 1, 0 },
		{ "SELECT", Select_List, SELECT, 1, 0 },
		{ 0 }
	};
	extern unsigned short TKP_Parse_Option_Entry();

	Option_Mask = ALTDEL | EGM;	/* These are the defaults */
	for (Index = 0; Index < Optc; Index++)
		Option_Mask = TKP_Parse_Option_Entry (Option_Mask, Optv[Index], Option_List);
	if ((Option_Mask & SELECT) == 0)
		Option_Mask |= SELECT | ASCIICHR;
	return (Option_Mask);
}

unsigned short TKP_Parse_Option_Entry (Option_Mask, Option_Value, Table)
unsigned short Option_Mask;
char *Option_Value;
struct Option_Table *Table;
{
	auto   struct Option_Table *Tab_Ptr;
	auto   char *Value_Ptr;
	auto   int Cond;
	auto   unsigned short New_Mask;
	auto   char c;
	msgcode DVIOUT_AMBOPTION, DVIOUT_UNKOPTION, DVIOUT_VALNOTALLOW;

	New_Mask = Option_Mask;
	for (Value_Ptr = Option_Value; (c = *Value_Ptr) != '\0' && c != '=' && c != ':'; Value_Ptr++)
		;
	*Value_Ptr = '\0';
	if (c == ':' || c == '=')
		Value_Ptr++;
	for (Tab_Ptr = Table; Tab_Ptr->Option_Name != 0; Tab_Ptr++)
	if ((Cond = Compare_Keyword_M (Option_Value, Tab_Ptr->Option_Name, Tab_Ptr->Min_Len, 4,
				       Tab_Ptr->Negatable)) != 0) {
		if (Cond == 2 || Cond == -2)
			Message (DVIOUT_AMBOPTION, Option_Value, 1);
		else {
			if (Cond > 0)
				New_Mask |= Tab_Ptr->Mask;
			else
				New_Mask &= ~Tab_Ptr->Mask;
			if (c == ':' || c == '=') {
				if (Tab_Ptr->Value_List == 0)
					Message (DVIOUT_VALNOTALLOW, Tab_Ptr->Option_Name, 1);
				else
					New_Mask = TKP_Parse_Option_Entry (New_Mask, Value_Ptr,
									   Tab_Ptr->Value_List);
			}
		}
		break;
	}
	if (Tab_Ptr->Option_Name == 0)
		Message (DVIOUT_UNKOPTION, Option_Value, 1);
	return (New_Mask);
}
