/* lspre.c -- display a long listing of a file, with preamble comment.
   Adapted from listfile.c.
   Copyright (C) 1991, 1993 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <config.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <time.h>

#include "defs.h"
#include "modetype.h"

/* Since major is a function on SVR4, we can't use `ifndef major'.  */
#ifdef MAJOR_IN_MKDEV
#include <sys/mkdev.h>
#define HAVE_MAJOR
#endif
#ifdef MAJOR_IN_SYSMACROS
#include <sys/sysmacros.h>
#define HAVE_MAJOR
#endif

#ifdef major			/* Might be defined in sys/types.h.  */
#define HAVE_MAJOR
#endif
#ifndef HAVE_MAJOR
#define major(dev)  (((dev) >> 8) & 0xff)
#define minor(dev)  ((dev) & 0xff)
#endif
#undef HAVE_MAJOR

void mode_string P_((unsigned short mode, char *str));

/* NAME is the name to print.
   RELNAME is the path to access it from the current directory.
   STATP is the results of stat or lstat on it.
   STREAM is the stdio stream to print on.  */

void
list_file_pre (name, relname, statp, stream)
     char *name;
     char *relname;
     struct stat *statp;
     FILE *stream;
{
  static int kilobytes = -1;	/* -1 = uninitialized, 0 = 512, 1 = 1024. */
  char modebuf[20];
  char timebuf[40];
  time_t current_time = time ((time_t *) 0);

  if (kilobytes == -1)
    kilobytes = getenv ("POSIXLY_CORRECT") == 0;

  mode_string (statp->st_mode, modebuf);
  modebuf[10] = '\0';

  strcpy (timebuf, ctime (&statp->st_mtime));
  if (current_time > statp->st_mtime + 6L * 30L * 24L * 60L * 60L /* Old. */
      || current_time < statp->st_mtime - 60L * 60L) /* In the future. */
    {
      /* The file is fairly old or in the future.
	 POSIX says the cutoff is 6 months old;
	 approximate this by 6*30 days.
	 Allow a 1 hour slop factor for what is considered "the future",
	 to allow for NFS server/client clock disagreement.
	 Show the year instead of the time of day.  */
      strcpy (timebuf + 11, timebuf + 19);
    }
  timebuf[16] = 0;

  if (file_status < FILE_HAVE_PRE)
    read_file_pre (rel_pathname, statp);

  /* The space between the mode and the number of links is the POSIX
     "optional alternate access method flag". */
  fprintf (stream, "%s %3u ", modebuf, statp->st_nlink);

  fprintf (stream, "%-32s", file_pre);

  if (S_ISCHR (statp->st_mode) || S_ISBLK (statp->st_mode))
#ifdef HAVE_ST_RDEV
    fprintf (stream, "%3u, %3u ", major (statp->st_rdev), minor (statp->st_rdev));
#else
    fprintf (stream, "         ");
#endif
  else
    fprintf (stream, "%8lu ", statp->st_size);

  fprintf (stream, "%s ", timebuf + 4);

  print_name_with_quoting (name, stream);

#ifdef S_ISLNK
  if (S_ISLNK (statp->st_mode))
    {
      char *linkname = get_link_name (name, relname);

      if (linkname)
	{
	  fputs (" -> ", stream);
	  print_name_with_quoting (linkname, stream);
	  free (linkname);
	}
    }
#endif
  putc ('\n', stream);
}
