/* CID-keyed Font */
#include "config.h"

#include <string.h>
#include <stdio.h>
#include "mem.h"
#include "error.h"
#include "pdfobj.h"
#include "numbers.h"
#include "pdflimits.h"
#include "type0.h"
#include "cidbase.h"
#include "cid.h"
static unsigned num_cids = 0, max_cids = 0;
static struct a_cid_font *cids = NULL;
static unsigned char verbose = 0;

void cid_set_verbose(void)
{
  if (verbose < 255) {
    verbose += 1;
  }
  cidbase_set_verbose();
}

/* Character Collection and CMap */
static const struct {
  char *registry;
  char *ordering;
  int supplement;
} cid_cc_names[4] = {
  {"Adobe", "GB1", 0}, /* not supportd */
  {"Adobe", "CNS1", 1}, /* not supportd */
  {"Adobe", "Japan1", 2}, /* supplement = 1 ? */
  {"Adobe", "Korea1", 1}, /* not supportd */
};

void cid_add_system_info(pdf_obj *dict, int cc)
{
  pdf_obj *systeminfo;
  char *reg, *ord;
  int sup;

  if (cc < 0 || cc > CC_KOREAN) {
    ERROR("invalid Character Collection ID");
  } else {
    systeminfo = pdf_new_dict ();

    reg = cid_cc_names[cc].registry;
    ord = cid_cc_names[cc].ordering;
    sup = cid_cc_names[cc].supplement;

    pdf_add_dict (systeminfo,
		  pdf_new_name ("Registry"),
		  pdf_new_string (reg, strlen(reg)));
    pdf_add_dict (systeminfo,
		  pdf_new_name ("Ordering"),
		  pdf_new_string (ord, strlen(ord)));
    pdf_add_dict (systeminfo,
		  pdf_new_name ("Supplement"),
		  pdf_new_number (sup));

    pdf_add_dict (dict, 
		  pdf_new_name ("CIDSystemInfo"),		 
		  systeminfo);
  }
  return;
}

int cid_find_parent(const char *map_name, char *encoding)
{
  int i, cid_id, parent;
  char *enc_name;
  if ((cid_id = cid_is_known_font(map_name)) >= 0) {
    for (i=0;i<cids[cid_id].num_parents;i++) {
      parent = (cids[cid_id].parent)[i];
      enc_name = type0_font_encoding(parent); /* back type0 again, not good. */
      if (enc_name && !strcmp(enc_name, encoding)) {
	return parent;
      }
    }
  }
  return -1;
}

pdf_obj *cid_call_descendant(int type0_id, int cid_id)
{
  int i;
  if (cid_id>=0 && cid_id<num_cids) {
    i = cids[cid_id].num_parents;
    if (i < CID_MAX_PARENT) {
      (cids[cid_id].parent)[i] = type0_id;
      cids[cid_id].num_parents += 1;
    } else {
      ERROR ("cid_call_descendant(): too many parents");
    }
    return pdf_link_obj(cids[cid_id].indirect);
  } else {
    ERROR ("cid_call_descendant(): invalid cid font id");
    return NULL;
  }
}

struct a_cid_font *new_cid_font(void)
{
  struct a_cid_font *cid_font;

  if (num_cids >= max_cids) {
    max_cids += MAX_FONTS;
    cids = RENEW (cids, max_cids, struct a_cid_font);
  }

  if (verbose>3) fprintf(stderr, "new cid font (id: %d)\n", num_cids);
  /* descriptor and fontfile should not be here ? */
  cid_font = &(cids[num_cids]);
  cids[num_cids].fontname = NULL;
  cids[num_cids].cid_name = NULL;
  cids[num_cids].direct = NULL;  /* fixme */
  cids[num_cids].indirect = NULL;
  cids[num_cids].descriptor = NULL;
  cids[num_cids].fontfile = NULL;  /* fixme */
  cids[num_cids].num_parents = 0;
  cids[num_cids].base_id = -1;
  cids[num_cids].cc = CC_JAPANESE; /* fixme */
  num_cids += 1;
  
  return cid_font;
}

/* returns PostScript name */
char *cid_fontname (int cid_id)
{
  if (cid_id >= 0 && cid_id < num_cids) {
    return cids[cid_id].fontname;
  } else {
    return NULL;
  }
}

/* fixme */
int cid_is_known_font (const char *map_name)
{
  int i;

  for (i=0; i<num_cids; i++) {
    if (cids[i].cid_name && !strcmp (cids[i].cid_name, map_name)) {
      break;
    }
  }
  if (i == num_cids) i = -1;
  return i;
}

int cid_get_id (const char *map_name)
{
  int result;

#ifdef MEM_DEBUG
  MEM_START
#endif

  if ((result = cid_is_known_font(map_name)) < 0) {
    result = num_cids;
    if (cidbase_get_id(map_name)) {
      if(verbose>3) fprintf(stderr, "a CID font\n");
    } else {
      if(verbose>3) fprintf(stderr, "not a CID font\n");
      result = -1;
    }
  }
  
#ifdef MEM_DEBUG
  MEM_END
#endif /* MEM_DEBUG */
  return result;
}

void cid_flush_all (void)
{
  int i;

  for (i=0; i<num_cids; i++) {
    if (verbose>1) fprintf(stderr, "(CID:%s)", cid_fontname(i));
    do_cidbase(i);
    cidbase_release(i);
    if (cids[i].descriptor) pdf_release_obj (cids[i].descriptor);
    if (cids[i].direct) pdf_release_obj (cids[i].direct); /* fixme */
    if (cids[i].indirect) pdf_release_obj (cids[i].indirect);
    if (cids[i].fontfile) pdf_release_obj (cids[i].fontfile); /* fixme */
    if (cids[i].cid_name) RELEASE (cids[i].cid_name);
    if (cids[i].fontname) RELEASE (cids[i].fontname);
  }
  if (cids) RELEASE(cids);
}
