﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/datasync/DataSync_EXPORTS.h>

#include <utility>

namespace Aws {
namespace Utils {
namespace Json {
class JsonValue;
class JsonView;
}  // namespace Json
}  // namespace Utils
namespace DataSync {
namespace Model {

/**
 * <p>Specifies configuration information for a DataSync-managed secret, such as an
 * authentication token, secret key, password, or Kerberos keytab that DataSync
 * uses to access a specific storage location, with a customer-managed KMS key.</p>
 *  <p>You can use either <code>CmkSecretConfig</code> or
 * <code>CustomSecretConfig</code> to provide credentials for a
 * <code>CreateLocation</code> request. Do not provide both parameters for the same
 * request.</p> <p><h3>See Also:</h3>   <a
 * href="http://docs.aws.amazon.com/goto/WebAPI/datasync-2018-11-09/CmkSecretConfig">AWS
 * API Reference</a></p>
 */
class CmkSecretConfig {
 public:
  AWS_DATASYNC_API CmkSecretConfig() = default;
  AWS_DATASYNC_API CmkSecretConfig(Aws::Utils::Json::JsonView jsonValue);
  AWS_DATASYNC_API CmkSecretConfig& operator=(Aws::Utils::Json::JsonView jsonValue);
  AWS_DATASYNC_API Aws::Utils::Json::JsonValue Jsonize() const;

  ///@{
  /**
   * <p>Specifies the ARN for the DataSync-managed Secrets Manager secret that that
   * is used to access a specific storage location. This property is generated by
   * DataSync and is read-only. DataSync encrypts this secret with the KMS key that
   * you specify for <code>KmsKeyArn</code>.</p>
   */
  inline const Aws::String& GetSecretArn() const { return m_secretArn; }
  inline bool SecretArnHasBeenSet() const { return m_secretArnHasBeenSet; }
  template <typename SecretArnT = Aws::String>
  void SetSecretArn(SecretArnT&& value) {
    m_secretArnHasBeenSet = true;
    m_secretArn = std::forward<SecretArnT>(value);
  }
  template <typename SecretArnT = Aws::String>
  CmkSecretConfig& WithSecretArn(SecretArnT&& value) {
    SetSecretArn(std::forward<SecretArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies the ARN for the customer-managed KMS key that DataSync uses to
   * encrypt the DataSync-managed secret stored for <code>SecretArn</code>. DataSync
   * provides this key to Secrets Manager.</p>
   */
  inline const Aws::String& GetKmsKeyArn() const { return m_kmsKeyArn; }
  inline bool KmsKeyArnHasBeenSet() const { return m_kmsKeyArnHasBeenSet; }
  template <typename KmsKeyArnT = Aws::String>
  void SetKmsKeyArn(KmsKeyArnT&& value) {
    m_kmsKeyArnHasBeenSet = true;
    m_kmsKeyArn = std::forward<KmsKeyArnT>(value);
  }
  template <typename KmsKeyArnT = Aws::String>
  CmkSecretConfig& WithKmsKeyArn(KmsKeyArnT&& value) {
    SetKmsKeyArn(std::forward<KmsKeyArnT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_secretArn;

  Aws::String m_kmsKeyArn;
  bool m_secretArnHasBeenSet = false;
  bool m_kmsKeyArnHasBeenSet = false;
};

}  // namespace Model
}  // namespace DataSync
}  // namespace Aws
