#!/bin/bash

source $(dirname $(realpath $0))/functions.sh

DIST="$(cygpath -u "$PROGRAMFILES")/qemu"
DIST_BIN="$DIST"
DIST_SHARE="$DIST/share"

SYSROOT="/usr/x86_64-w64-mingw32/sys-root/mingw"
SYSROOT_BIN="$SYSROOT/bin"
SYSROOT_SHARE="$SYSROOT/share/qemu"

REL_STD_TEST_DIR="$(basename $0)"
TMP_TEST_DIR="/tmp/$REL_STD_TEST_DIR"
HOME_TEST_DIR="$HOME/$REL_STD_TEST_DIR"

QEMU="qemu-system-x86_64"

function winPath {
	local CYGPATH="$1"
	cygpath -w -m "$CYGPATH"
}

function isHome {
	[ "$CURRENT_TEST_DIR" == "$HOME_TEST_DIR" ]
}

function isTmp {
	[ "$CURRENT_TEST_DIR" == "$TMP_TEST_DIR" ]
}

function relInHomeOnly {
	return 1
}

function qtestPrepare {
	mkdir -p "$CURRENT_TEST_DIR" && cd "$CURRENT_TEST_DIR" || exit 1
	[ -d "$DIR" ] && ( rm -r "$DIR" || exit 1 )
	mkdir -p "$DIR" || exit 1
}

function qtestEvaluate {
	infoLog
	infoLog "$DIR"
	infoLog "  WORKING_DIR:"
	infoLog "    '$(pwd)'"
	if [ -n "$LINKED_FILE" ]
	then
		infoLog "  LINKED_FILE:"
		infoLog "    '$LINKED_FILE'"
	fi
	infoLog "  PARAMETER_FILE:"
	infoLog "    '$PARAMETER_FILE'"
	infoLog "  TRANSLATED_FILE:"
	infoLog "    '$TRANSLATED_FILE'"

	local WRAPPED_DUMP="$DIR/wrapped.dmp"
	local SYSROOT_DUMP="$DIR/sysroot.dmp"
	local DIST_DUMP="$DIR/dist.dmp"

	export QEXEC_DUMP="$WRAPPED_DUMP"
	testCmd "" "$QEMU" "$PARAMETER_FILE"

	export QEXEC_DUMP="$SYSROOT_DUMP"
	testCmd "qexec" "$SYSROOT_BIN/$QEMU" "$TRANSLATED_FILE"

	export QEXEC_DUMP="$DIST_DUMP"
	testCmd "qexec" "$DIST_BIN/$QEMU" "$TRANSLATED_FILE"

	infoLog
	if [ -f "$WRAPPED_DUMP" ]
	then
		if diff "$WRAPPED_DUMP" "$SYSROOT_DUMP" &> /dev/null ||
			diff "$WRAPPED_DUMP" "$DIST_DUMP" &> /dev/null
		then
			rm -r "$DIR"
			infoLog "OK"
			infoLog
			infoLog "================================================================================"
			return
		else
			infoLog "NOK"
			infoLog
			>&2 diff "$WRAPPED_DUMP" "$SYSROOT_DUMP"
			>&2 diff "$WRAPPED_DUMP" "$SYSROOT_DUMP"
			infoLog
			>&2 diff "$WRAPPED_DUMP" "$DIST_DUMP"
			>&2 diff "$WRAPPED_DUMP" "$DIST_DUMP"
		fi
	else
		infoLog "Error: WRAPPED_DUMP='$WRAPPED_DUMP'"
	fi
	infoLog
	ALL_OK="false"
	infoLog "================================================================================"
}

function testCmd {
	local EXEC="$1" QEMU="$2" FILE="$3"
	$EXEC "$QEMU" \
		-M q35 \
		-accel whpx,kernel-irqchip=off \
		-m 1536 \
		-pidfile .qemupid.1653717533 \
		-audiodev id=audio0,driver=dsound,in.voices=0 \
		-device ich9-intel-hda \
		-device hda-micro,audiodev=audio0 \
		-vga qxl \
		-spice port=5905,addr=127.0.0.1,disable-ticketing=on \
		-device virtio-serial \
		-chardev spicevmc,id=spicechannel0,name=vdagent \
		-device virtserialport,chardev=spicechannel0,name=com.redhat.spice.0 \
		-drive "logfile=$FILE,media=disk"\
		-drive "filename=$FILE,media=disk"\
		-drive "file.filename=$FILE,media=disk"\
		-drive "path=$FILE,media=disk"\
		-drive "splash=$FILE,media=disk"\
		-drive "any=none,script=$FILE"\
		-drive "any=none,x509-file=$FILE,media=disk"\
		-cdrom "$FILE"
}

#########################################################
#
# qtest0X - relatively addressed link
# Parameter file relative: translate relative if possible
#

# Linked file in subdir is addressable relatively
function qtest00 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local REL_LINKED_FILE="$DIR/test a/abs test file"
	local LINKED_FILE="$CURRENT_TEST_DIR/$REL_LINKED_FILE"
	local PARAMETER_FILE="$DIR/rel test link"
	mkdir -p "$(dirname "$LINKED_FILE")"
	touch "$LINKED_FILE"
	ln -s "$LINKED_FILE" "$PARAMETER_FILE"

	local TRANSLATED_FILE="$REL_LINKED_FILE"
	relInHomeOnly && isTmp && TRANSLATED_FILE="$(winPath "$LINKED_FILE")"

	qtestEvaluate
}

# Linked file is addressable relatively
function qtest01 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local LINKED_FILE="/usr/share/doc/qemu-integration/LICENSE"
	local PARAMETER_FILE="$DIR/rel test link"
	ln -s "$LINKED_FILE" "$PARAMETER_FILE"

	local TRANSLATED_FILE="../..$LINKED_FILE"
	isHome && TRANSLATED_FILE="../$TRANSLATED_FILE"
	relInHomeOnly && TRANSLATED_FILE="$(winPath "$LINKED_FILE")"

	qtestEvaluate
}

# Linked file is NOT addressable relatively
function qtest02 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local LINKED_FILE="/cygdrive/c/Windows/win.ini"
	local PARAMETER_FILE="$DIR/rel test link"
	ln -s "$LINKED_FILE" "$PARAMETER_FILE"

	local TRANSLATED_FILE="$(winPath "$LINKED_FILE")"

	qtestEvaluate
}

# Linked file is in subdirectory of level1-superdirectory
function qtest03 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local REL_LINKED_FILE="../${REL_STD_TEST_DIR}2/test file"
	local LINKED_FILE="$CURRENT_TEST_DIR/$REL_LINKED_FILE"
	local PARAMETER_FILE="$DIR/rel test link"
	ln -s "$LINKED_FILE" "$PARAMETER_FILE"
	mkdir -p "$(dirname "$LINKED_FILE")"
	touch "$LINKED_FILE"

	local TRANSLATED_FILE="$REL_LINKED_FILE"
	relInHomeOnly && isTmp && TRANSLATED_FILE="$(winPath "$LINKED_FILE")"

	qtestEvaluate
}

# Linked file is in subdirectory of level2-superdirectory
function qtest04 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local REL_LINKED_FILE="../../${REL_STD_TEST_DIR}2/test file"
	local LINKED_FILE="$CURRENT_TEST_DIR/$REL_LINKED_FILE"
	local PARAMETER_FILE="$DIR/rel test link"
	ln -s "$LINKED_FILE" "$PARAMETER_FILE"
	mkdir -p "$(dirname "$LINKED_FILE")"
	touch "$LINKED_FILE"

	local TRANSLATED_FILE="$REL_LINKED_FILE"
	relInHomeOnly && TRANSLATED_FILE="$(winPath "$LINKED_FILE")"

	qtestEvaluate

	rm "$TRANSLATED_FILE"
	rmdir "$(dirname "$TRANSLATED_FILE")"
}

# Linked file is in subdirectory of outofbound-level-superdirectory (far super of / !)
# Link resolution is adjusted by Cygwin/Shell/?
function qtest05 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local REL_LINKED_FILE="../../../../../${REL_STD_TEST_DIR}2/test file"
	local LINKED_FILE="$CURRENT_TEST_DIR/$REL_LINKED_FILE"
	local PARAMETER_FILE="$DIR/rel test link"
	ln -s "$LINKED_FILE" "$PARAMETER_FILE"
	mkdir -p "$(dirname "$LINKED_FILE")"
	touch "$LINKED_FILE"

	local TRANSLATED_FILE="../../${REL_STD_TEST_DIR}2/test file"
	isHome && TRANSLATED_FILE="../$TRANSLATED_FILE"
	relInHomeOnly && TRANSLATED_FILE="$(winPath "$LINKED_FILE")"

	qtestEvaluate

	rm "$TRANSLATED_FILE"
	rmdir "$(dirname "$TRANSLATED_FILE")"
}

# Linked file is relative to link in superdirectory
# Linked file is NOT relative to working directory
function qtest06 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local LINKED_FILE="$DIR/test a/rel test file"
	local PARAMETER_FILE="$DIR/abs test link"
	(
		cd "$(dirname "$PARAMETER_FILE")"
		mkdir -p "$(dirname "$LINKED_FILE")"
		touch "$LINKED_FILE"
	)
	ln -s "$LINKED_FILE" "$PARAMETER_FILE"

	local TRANSLATED_FILE="$(dirname "$PARAMETER_FILE")/$LINKED_FILE"

	qtestEvaluate
}

#########################################################
#
# qtest1X - absolutely addressed link
# Parameter file absolute: translate absolute
#

# Linked file is in subdirectory
function qtest10 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local LINKED_FILE="$CURRENT_TEST_DIR/$DIR/test a/abs test file"
	local PARAMETER_FILE="$CURRENT_TEST_DIR/$DIR/abs test link"
	mkdir -p "$(dirname "$LINKED_FILE")"
	touch "$LINKED_FILE"
	ln -s "$LINKED_FILE" "$PARAMETER_FILE"

	local TRANSLATED_FILE="$(winPath "$LINKED_FILE")"

	qtestEvaluate
}

# Linked file is absolute in completely different tree
function qtest11 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local LINKED_FILE="/usr/share/doc/qemu-integration/LICENSE"
	local PARAMETER_FILE="$CURRENT_TEST_DIR/$DIR/abs test link"
	ln -s "$LINKED_FILE" "$PARAMETER_FILE"

	local TRANSLATED_FILE="$(winPath "$LINKED_FILE")"

	qtestEvaluate
}

# Linked file is absolute outside cygwin-scope
function qtest12 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local LINKED_FILE="/cygdrive/c/Windows/win.ini"
	local PARAMETER_FILE="$CURRENT_TEST_DIR/$DIR/abs test link"
	ln -s "$LINKED_FILE" "$PARAMETER_FILE"

	local TRANSLATED_FILE="$(winPath "$LINKED_FILE")"

	qtestEvaluate
}

# Linked file is relative to link in subdirectory of superdirectory
function qtest13 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local REL_LINKED_FILE="../${REL_STD_TEST_DIR}2/test file"
	local LINKED_FILE="$CURRENT_TEST_DIR/$REL_LINKED_FILE"
	local PARAMETER_FILE="$CURRENT_TEST_DIR/$DIR/abs test link"
	ln -s "$LINKED_FILE" "$PARAMETER_FILE"
	mkdir -p "$(dirname "$LINKED_FILE")"
	touch "$LINKED_FILE"

	local TRANSLATED_FILE="$(winPath "$LINKED_FILE")"

	qtestEvaluate
}

# Linked file is relative to link in superdirectory
# Linked file is NOT relative to working directory
function qtest14 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local LINKED_FILE="$DIR/test a/rel test file"
	local PARAMETER_FILE="$CURRENT_TEST_DIR/$DIR/abs test link"
	(
		cd "$(dirname "$PARAMETER_FILE")"
		mkdir -p "$(dirname "$LINKED_FILE")"
		touch "$LINKED_FILE"
	)
	ln -s "$LINKED_FILE" "$PARAMETER_FILE"

	local TRANSLATED_FILE="$(winPath "$(dirname "$PARAMETER_FILE")/$LINKED_FILE")"

	qtestEvaluate
}

#########################################################
#
# qtest2X - relatively addressed file
# Parameter file relative: translate relative
#

# File is in subdirectory
function qtest20 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local PARAMETER_FILE="$DIR/rel test file"
	touch "$PARAMETER_FILE"

	local TRANSLATED_FILE="$PARAMETER_FILE"

	qtestEvaluate
}

# File is addressed in subdirectory, but doesn't yet exist
function qtest21 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local PARAMETER_FILE="$DIR/rel test file"

	local TRANSLATED_FILE="$PARAMETER_FILE"

	qtestEvaluate
}

# File is in not normalized subdirectory
function qtest22 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local PARAMETER_FILE="../${REL_STD_TEST_DIR}/$DIR/rel test file"
	mkdir -p "$(dirname "$PARAMETER_FILE")"
	touch "$PARAMETER_FILE"

	local TRANSLATED_FILE="$DIR/rel test file"

	qtestEvaluate
}

# File is addressed not normalized subdirectory, but doesn't yet exist
function qtest23 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local PARAMETER_FILE="../${REL_STD_TEST_DIR}/$DIR/rel test file"
	mkdir -p "$(dirname "$PARAMETER_FILE")"

	local TRANSLATED_FILE="$DIR/rel test file"

	qtestEvaluate
}

# File is in subdirectory of super directory
function qtest24 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local PARAMETER_FILE="../${REL_STD_TEST_DIR}2/$DIR/rel test file"
	mkdir -p "$(dirname "$PARAMETER_FILE")"
	touch "$PARAMETER_FILE"

	local TRANSLATED_FILE="$PARAMETER_FILE"

	qtestEvaluate
}

# File is addressed subdirectory of super directory, but doesn't yet exist
function qtest25 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local PARAMETER_FILE="../${REL_STD_TEST_DIR}2/$DIR/rel test file"
	mkdir -p "$(dirname "$PARAMETER_FILE")"

	local TRANSLATED_FILE="$PARAMETER_FILE"

	qtestEvaluate
}

#########################################################
#
# qtest3X - absolutely addressed file
# Parameter file absolute: translate absolute
#

function qtest30 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local PARAMETER_FILE="$CURRENT_TEST_DIR/$DIR/abs test file"
	touch "$PARAMETER_FILE"

	local TRANSLATED_FILE="$(winPath "$PARAMETER_FILE")"

	qtestEvaluate
}

function qtest31 {
	local DIR="test ${FUNCNAME[0]}"
	qtestPrepare

	local PARAMETER_FILE="$CURRENT_TEST_DIR/$DIR/abs test file"

	local TRANSLATED_FILE="$(winPath "$PARAMETER_FILE")"

	qtestEvaluate
}

WORKING_DIRS="$TMP_TEST_DIR $HOME_TEST_DIR"
ALL_OK="true"
for CURRENT_TEST_DIR in $WORKING_DIRS
do
	qtest00
	qtest01
	qtest02
	qtest03
	qtest04
	qtest05
	qtest06
	qtest10
	qtest11
	qtest12
	qtest13
	qtest14
	qtest20
	qtest21
	qtest22
	qtest23
	qtest24
	qtest25
	qtest30
	qtest31
done
infoLog
[ "$ALL_OK" == "true" ] && infoLog "PASS - All tests passed." || infoLog "FAIL - One or more tests failed."
infoLog
