// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
// ----------------------------------------------------------------------
// BCondBase: the base class for all Field/Array-based boundary conditions.
//-----------------------------------------------------------------------------

#ifndef POOMA_BCONDS_BCONDBASE_H
#define POOMA_BCONDS_BCONDBASE_H

#include "BConds/BCondItem.h"

//-----------------------------------------------------------------------------
// BCondBase is an abstract base class for all boundary conditions. It extends
// BCondItem to allow specification of the destination/source domains and the
// subject.
//-----------------------------------------------------------------------------

template<class Domain, class Subject>
class BCondBase : public BCondItem {
public:

  //---------------------------------------------------------------------------
  // Convenient typedefs. Make this class, the domain type , and the subject
  // public.

  typedef BCondBase<Domain, Subject> This_t;  
  typedef Domain Domain_t;
  typedef Subject Subject_t;

  //---------------------------------------------------------------------------
  // Constructors. We do not support a default constructor since we always have
  // to have a subject. However, we can rely on the default constructors for
  // domains to make empty sets. We rely on shallow copy semantics for the 
  // subject.

  BCondBase(const Subject_t &s)
  : subject_m(s) { }

  template<class Arg>
  BCondBase(const Subject_t &s, const Arg &a)
  : subject_m(s, a) { }

  BCondBase(const Subject_t &s, const Domain_t &dest)
  : subject_m(s), destDomain_m(dest) { }

  template<class Arg>
  BCondBase(const Subject_t &s, const Arg &a, const Domain_t &dest)
  : subject_m(s, a), destDomain_m(dest) { }

  BCondBase(const Subject_t &s, const Domain_t &dest, const Domain_t &src)
  : subject_m(s), destDomain_m(dest), srcDomain_m(src) { }

  template<class Arg>
  BCondBase(const Subject_t &s, const Arg &a, const Domain_t &dest, 
    const Domain_t &src)
  : subject_m(s, a), destDomain_m(dest), srcDomain_m(src) { }

  //---------------------------------------------------------------------------
  // Copy constructor. We rely on shallow copy semantics for the subject.

  BCondBase(const This_t &model)
  : BCondItem(model), 
    destDomain_m(model.destDomain()),
    srcDomain_m(model.srcDomain()),
    subject_m(model.subject()) { }
  
  //---------------------------------------------------------------------------
  // Trivial destructor, but virtual since we will be subclassing.

  virtual ~BCondBase() { }
  
  //---------------------------------------------------------------------------
  // Modifiers/accessors for subject and domains.
  
  Subject &subject()
  {
    return subject_m;
  }
  
  Domain_t &destDomain() { return destDomain_m; }
  const Domain_t &destDomain() const { return destDomain_m; }

  Domain_t &srcDomain() { return srcDomain_m; }
  const Domain_t &srcDomain() const { return srcDomain_m; }

protected:
  
  //---------------------------------------------------------------------------
  // Virtual touches() functions used to determine if the input domain touches
  // either the destination domain (testDest == true) or the source domain
  // (testDest == false). Virtual because we do not know the type of these
  // domains in this class. Since they are virtual, we must enumerate all 
  // possible input domain types, which will almost always be the the base
  // domain for an array or field. It's not elegant and it violates the
  // Open-Closed Principle, but it is really the only solution given that
  // templated virtual functions don't exist. These functions are protected
  // since they implement internal logic.

  virtual bool touches(const Range<1> &d, bool testDest) const
  {
    return testDest ? ::touches(d, destDomain()) : ::touches(d, srcDomain());
  }
  virtual bool touches(const Range<2> &d, bool testDest) const
  {
    return testDest ? ::touches(d, destDomain()) : ::touches(d, srcDomain());
  }
  virtual bool touches(const Range<3> &d, bool testDest) const
  {
    return testDest ? ::touches(d, destDomain()) : ::touches(d, srcDomain());
  }
  virtual bool touches(const Range<4> &d, bool testDest) const
  {
    return testDest ? ::touches(d, destDomain()) : ::touches(d, srcDomain());
  }
  virtual bool touches(const Range<5> &d, bool testDest) const
  {
    return testDest ? ::touches(d, destDomain()) : ::touches(d, srcDomain());
  }
  virtual bool touches(const Range<6> &d, bool testDest) const
  {
    return testDest ? ::touches(d, destDomain()) : ::touches(d, srcDomain());
  }
  virtual bool touches(const Range<7> &d, bool testDest) const
   {
    return testDest ? ::touches(d, destDomain()) : ::touches(d, srcDomain());
  }
 
private:

  Domain_t srcDomain_m, destDomain_m;
  Subject_t subject_m;
    
};

#endif // POOMA_BCONDS_BCONDBASE_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: BCondBase.h,v $   $Author: swhaney $
// $Revision: 1.5 $   $Date: 2000/03/07 13:16:10 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
