// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
// ----------------------------------------------------------------------
// BCondItem: the ultimate base class for all boundary conditions.
//-----------------------------------------------------------------------------

#ifndef POOMA_BCONDS_BCONDITEM_H
#define POOMA_BCONDS_BCONDITEM_H

#include "Domain/Range.h"

#if defined(POOMA_DEBUG_BOUNDARY_CONDITIONS)
#include <iostream>
#endif

//-----------------------------------------------------------------------------
// BCondItem is an abstract base class for all boundary conditions. It
// provides an interface for the most basic boundary condition services:
// responding to pre-read and post-write events and setting/clearing the
// dirty flag. 
//
// This class is not templated because we want to store these puppies in
// BCondList containers, which are also not templated. Specifying the
// domain as a template parameter would be more convenient since it would
// eliminate the touches() virtual function calls. However, this is not
// practical since there is no easy way for Fields to supply this template
// parameter. Suppose, for example, that one has a 3D Array, which would
// mean the appropriate domain type would probably be Interval<3>, which could
// be deduced from the Field's Domain_t. However, if we take a 2D slice of that
// array, the new Array's Domain_t would be Interval<2>. This is a problem
// because we'd still want to use the same BCondList.
//-----------------------------------------------------------------------------

class BCondItem {
public:

  //---------------------------------------------------------------------------
  // Default and copy constructors. The default constructor relies on the
  // default constructors for the domains, which should create empty sets.

  BCondItem() : dirty_m(true) { }
  BCondItem(const BCondItem &model)
  : dirty_m(model.dirty()) { }
  
  //---------------------------------------------------------------------------
  // Trivial destructor, but virtual since we will be subclassing.

  virtual ~BCondItem() { }

  //---------------------------------------------------------------------------
  // Boundary condition application function. Subclasses must override to make
  // this boundary condition actually do something.
  
  virtual void applyBoundaryCondition() = 0; 

  //---------------------------------------------------------------------------
  // Check to see if we would need to apply the boundary condition if we were
  // reading from the specified domain. In this case, if the 
  // dirty flag is set and the and the domain being read touches the
  // destination domain, we must apply the boundary condition.

  bool needToApplyBoundaryCondition() const
  {
    return dirty();
  }
  
  template<class ReadDomain>
  bool needToApplyBoundaryCondition(const ReadDomain &readDomain) const
  {
    return dirty() && touches(readDomain, true);
  }

  //---------------------------------------------------------------------------
  // Notification functions. 

  void notifyPreRead()
  {
    // This function is called if somebody is getting ready to read somewhere 
    // that could include this B.C.'s destination domain. 
    
    if (needToApplyBoundaryCondition())
      {
        applyBoundaryCondition();
        clearDirty();
      }
  }

  template<class ReadDomain>
  void notifyPreRead(const ReadDomain &readDomain)
  {
    // This function is called if somebody is getting ready to read somewhere 
    // that could include this B.C.'s destination domain. 
    
    if (needToApplyBoundaryCondition(readDomain))
      {
        applyBoundaryCondition();
        clearDirty();
      }
  }
  
  void notifyPostWrite()
  {
    // This function is called if somebody has already written somewhere 
    // that could include this B.C.'s source domain. In this version, we simply
    // set the dirty flag no matter what.
    
    setDirty();
  }
  
  template<class WriteDomain>
  void notifyPostWrite(const WriteDomain &writeDomain)
  {
    // This function is called if somebody has already written somewhere 
    // that could include this B.C.'s source domain. In this case, we simply
    // set the dirty flag. Note: we do not check to see if someone writes
    // in the destination domain. It is up to them to cause an update
    // if this occurs.
    
    if (touches(writeDomain, false))
      {
        setDirty();
      }
  }

  //---------------------------------------------------------------------------
  // Accessors for the dirty flag.

  bool dirty() const { return dirty_m; }
    
  //---------------------------------------------------------------------------
  // Modifiers used to set/clear the dirty flag.
  
  void setDirty()
  {
#if defined(POOMA_DEBUG_BOUNDARY_CONDITIONS)
    std::cout << "Setting dirty flag for " << (void *) this 
              << std::endl;
#endif
    dirty_m = true;
  }  
  void clearDirty()
  {
#if defined(POOMA_DEBUG_BOUNDARY_CONDITIONS)
    std::cout << "Clearing dirty flag for " << (void *) this 
              << std::endl;
#endif
    dirty_m = false;
  }  

protected:

  //---------------------------------------------------------------------------
  // Virtual touches() functions used to determine if the input domain touches
  // either the destination domain (testDest == true) or the source domain
  // (testDest == false). Virtual because we do not know the type of these
  // domains in this class. Since they are virtual, we must enumerate all 
  // possible input domain types, which will almost always be the the base
  // domain for an array or field. It's not elegant and it violates the
  // Open-Closed Principle, but it is really the only solution given that
  // templated virtual functions don't exist. These functions are protected
  // since they implement internal logic.

  virtual bool touches(const Range<1> &d, bool testDest) const = 0;
  virtual bool touches(const Range<2> &d, bool testDest) const = 0;
  virtual bool touches(const Range<3> &d, bool testDest) const = 0;
  virtual bool touches(const Range<4> &d, bool testDest) const = 0;
  virtual bool touches(const Range<5> &d, bool testDest) const = 0;
  virtual bool touches(const Range<6> &d, bool testDest) const = 0;
  virtual bool touches(const Range<7> &d, bool testDest) const = 0;
    
private:

  bool dirty_m;
  
};

#endif // POOMA_BCONDS_BCONDITEM_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: BCondItem.h,v $   $Author: swhaney $
// $Revision: 1.9 $   $Date: 2000/04/27 22:31:30 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
