// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Brick-Engine non-inline template definitions.
//-----------------------------------------------------------------------------

#include "Engine/BrickEngine.h"

///////////////////////////////////////////////////////////////////////////////
//
// Brick-Engine Member Functions
//
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
//
// Engine<Dim,T,Brick> constructors:
//
//   Engine(const Domain_t &domain)
//   Engine(const Domain_t &domain, const T &model)
//   Engine(const Node<Domain_t> &node)
//   Engine(const Layout_t &layout)
//
// Constructs a Brick-Engine holding type T elements with the
// multidimensional domain given by either a Domain_t object, a
// Node object, or a Layout_t object. These constructors allocate
// memory for the elements, and initialize the memory with
// ElementProperties::construct. 
//
//-----------------------------------------------------------------------------

template <int Dim, class T>
Engine<Dim,T,Brick>::Engine(const Domain_t &dom)
  : Base_t(dom), data_m(dom.size())
{ }

template <int Dim, class T>
Engine<Dim,T,Brick>::Engine(const Node<Domain_t> &node)
  : Base_t(node.allocated()),
    data_m(node.allocated().size(), node.affinity(), 
           DataBlockPtr<T>::WithAffinity_t())
{ }

template <int Dim, class T>
Engine<Dim,T,Brick>::Engine(const Layout_t &layout)
  : Base_t(layout.domain()), data_m(layout.domain().size())
{ }

template <int Dim, class T>
Engine<Dim,T,Brick>::Engine(const Domain_t &dom, const T& model)
  : Base_t(dom), data_m(dom.size(), model)
{ }

//-----------------------------------------------------------------------------
//
// Engine<Dim,T,Brick>(T * foreignData, const Interval<Dim> &domain)
//
// Constructs a Brick-Engine holding type T elements with the
// multidimensional domain given by Interval<Dim>.  The actual data is
// provided in an external buffer pointed to by foreignData.
//
//-----------------------------------------------------------------------------

template <int Dim, class T>
Engine<Dim,T,Brick>::Engine(T * foreignData, const Domain_t &dom)
  : Base_t(dom), data_m(foreignData, dom.size())
{ }

//-----------------------------------------------------------------------------
//
// Engine<Dim,T,Brick>(const Engine<Dim,T,Brick> &)
//
// Copy constructor for Brick-Engine.
//
//-----------------------------------------------------------------------------

template <int Dim, class T>
Engine<Dim,T,Brick>::Engine(const This_t &modelEngine)
  : Base_t(modelEngine), data_m(modelEngine.data_m)
{ }

//-----------------------------------------------------------------------------
//
// Engine<Dim,T,Brick> & operator=(const Engine<Dim,T,Brick> &)
//
// Assignment operator for Brick-Engines.
//
//-----------------------------------------------------------------------------

template <int Dim, class T>
Engine<Dim,T,Brick> & Engine<Dim,T,Brick>::operator=(const This_t &modelEngine)
{
  // Can skip the rest if we're trying to assign to ourselves

  if (this == &modelEngine) return *this;

  // Copy the base and the data block
  
  Base_t::operator=(modelEngine);
  data_m = modelEngine.data_m;
  PAssert(data_m.isAtBeginning());
  return *this;
}

//-----------------------------------------------------------------------------
//
// ~Engine<Dim,T,Brick>()
//
// Trivial out-of-line destructor to shorten compile times.
//
//-----------------------------------------------------------------------------

template <int Dim, class T> Engine<Dim,T,Brick>::~Engine() { }

//-----------------------------------------------------------------------------
//
// Engine<Dim,T,Brick> & makeOwnCopy()
//
// Causes the Brick-Engine to obtain a private copy of the data
// that it refers to.
//
//-----------------------------------------------------------------------------

template <int Dim, class T>
Engine<Dim,T,Brick> &Engine<Dim,T,Brick>::makeOwnCopy()
{
  if (data_m.isValid() && data_m.count() > 1) 
    {
      PAssert(data_m.isAtBeginning());
      data_m.makeOwnCopy();
    }

  return *this;
}

///////////////////////////////////////////////////////////////////////////////
//
// BrickView Engine Member Functions
//
///////////////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
//
// ~Engine<Dim,T,BrickView<Dim2> >()
//
// Moved destructor out-of-line to shorten compile time. 
//
//-----------------------------------------------------------------------------

template <int Dim, class T>
Engine<Dim,T,BrickView>::~Engine() { }

//-----------------------------------------------------------------------------
//
// Engine(const Engine<Dim,T,BrickView<Dim2> > &);
//
// Copy constructor.
//
//-----------------------------------------------------------------------------

template <int Dim, class T>
Engine<Dim,T,BrickView>::
Engine(const This_t &modelEngine)
  : Base_t(modelEngine), data_m(modelEngine.data_m)
{ }

// What is this for again???

template <int Dim, class T>
Engine<Dim,T,BrickView>::
Engine(const This_t &modelEngine, const EngineConstructTag &)
  : Base_t(modelEngine), data_m(modelEngine.data_m)
{ }

//-----------------------------------------------------------------------------
//
// Engine<Dim,T,BrickView> & operator=(const Engine<Dim,T,BrickView> &)
//
// Assignment operator.
//
//-----------------------------------------------------------------------------

template <int Dim, class T>
Engine<Dim,T,BrickView> &
Engine<Dim,T,BrickView>::operator=(const This_t &modelEngine)
{
  if (this == &modelEngine) return *this;
  Base_t::operator=(modelEngine);
  data_m = modelEngine.data_m;
  return *this;
}

//-----------------------------------------------------------------------------
//
// Engine(const Engine<Dim,T,BrickView<Dim2> > &);
//
// Construct from compressible brick.
//
//-----------------------------------------------------------------------------

// Change this when CompressibleBrick is modified to use BaseDomain!

template <int Dim, class T>
Engine<Dim,T,BrickView>::
Engine(const Engine<Dim,T,CompressibleBrick> &model)
  : Base_t(model, false)
{
  data_m = DataBlockPtr<T>(model.dataBlock(),baseOffset());
}

//-----------------------------------------------------------------------------
//
// Engine(const Engine<Dim,T,BrickView<Dim2> > &);
//
// Construct from compressible brick view.
//
//-----------------------------------------------------------------------------

template <int Dim, class T>
Engine<Dim,T,BrickView>::
Engine(const Engine<Dim,T,CompressibleBrickView> &model)
  : Base_t(model, false)
{
  data_m = DataBlockPtr<T>(model.dataBlock(),baseOffset());
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: BrickEngine.cpp,v $   $Author: sa_smith $
// $Revision: 1.74 $   $Date: 2000/07/11 23:06:40 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
