// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// ump_test7
//-----------------------------------------------------------------------------

#include "Pooma/Pooma.h"
#include "Pooma/UMPArrays.h"
#include "Layout/GuardLayers.h"
#include "Layout/Node.h"
#include "Utilities/Tester.h"

#include <iterator>
#include <vector>
using std::vector;

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);
  
  // Useful typedefs, constants, declarations, &c.
  
  typedef Brick                      PTag_t;
  typedef UniformTag                 LTag_t;
  typedef MultiPatch<LTag_t,PTag_t>  MPTag_t;
  typedef Engine<3,int,MPTag_t>      UMPEngine_t;
  typedef Array<3,int,MPTag_t>       UMPArray_t;
  
  typedef Engine<3,int,PTag_t>       PatchEngine_t;
  typedef Array<3,int,PTag_t>        PatchArray_t;
  typedef Array<3,int,Brick>         BrickArray_t;
  
  typedef UniformGridLayout<3>       Layout_t;  // Change if LTag_t is changed.
  typedef Node<Interval<3> >         Node_t;
  
  using std::endl;
  
  // Run parameters...
  
  const int size            = 6;       // Array will be size x size
  const int nblocks         = 3;       // Will have nblocks x nblocks patches
  const int internal_guards = 2;
  const int external_guards = 1;
  const int badval          = -77777;
  
  // Create the total domain.
  
  Interval<1> D(size);
  Interval<3> domain(D, D, D);
  
  // Create the block sizes.
  
  Loc<3> blocks(nblocks,nblocks,nblocks);
  
  // OK, let's try some guard cells.
  
  GuardLayers<3> igcs(internal_guards), egcs(external_guards);

  // Create the partitioners.
  
  UniformGridPartition<3> partition(blocks,igcs,egcs);
  
  // Create the layout.
  
  UniformGridLayout<3> layout(domain, partition, ReplicatedTag());
  
  // Make a UMP engine
    
  UMPEngine_t a(layout);
  UMPArray_t aa(a);

  aa = badval;
    
  tester.out() << aa << endl << endl;

  Layout_t::iterator niter;
  int i, j, k;
  
  // Print out the patches.
 
  niter = layout.beginGlobal();
  while (niter != layout.endGlobal())
  {
    tester.out() << *niter; 
    tester.out() << endl << endl;
    PatchArray_t pa(a.globalPatch(*niter));

    // The guards are uninitialized, so this will cause UMRs.

    tester.out() << pa << endl << endl;
    ++niter;
  }
  
  // Zero the guards and print the patches again.
  
  a.setGuards(0);
  
  niter = layout.beginGlobal();
  while (niter != layout.endGlobal())
  {
    tester.out() << *niter; 
    tester.out() << endl << endl;
    PatchArray_t pa(a.globalPatch(*niter));
    tester.out() << pa << endl << endl;
    ++niter;
  }

  // Check that engine indexing is working.

  for (i = 0; i < size; ++i)
    for (j = 0; j < size; ++j)
      for (k = 0; k < size; ++k)
         a(i,j,k) = i + j + k;
      
  tester.out() << aa << endl << endl;
    
  for (i = 0; i < size; ++i)
    for (j = 0; j < size; ++j)
      for (k = 0; k < size; ++k)
        {
          tester.check(aa(i,j,k) == i + j + k);
          tester.check(aa.read(i,j,k) == i + j + k);
        }

  // Set the guards to badval and check the patches again.
  
  a.setGuards(badval);
  
  niter = layout.beginGlobal();
  while (niter != layout.endGlobal())
  {
    tester.out() << *niter; 
    tester.out() << endl << endl;
    PatchArray_t pa(a.globalPatch(*niter));
    tester.out() << pa << endl << endl;
    ++niter;
  }

  // Finally, check guard cell accumulation.
  
  aa = 0;
  
  a.setGuards(1);
  
  tester.out() << aa << endl << endl;
  
  niter = layout.beginGlobal();
  while (niter != layout.endGlobal())
  {
    tester.out() << *niter; 
    tester.out() << endl << endl;
    PatchArray_t pa(a.globalPatch(*niter));
    tester.out() << pa << endl << endl;
    ++niter;
  }
  
  // Now accumulate from the guards and see what we get.  
  
  a.accumulateFromGuards();
  
  niter = layout.beginGlobal();
  while (niter != layout.endGlobal())
  {
    tester.out() << *niter; 
    tester.out() << endl << endl;
    PatchArray_t pa(a.globalPatch(*niter));
    tester.out() << pa << endl << endl;
    ++niter;
  }

  // The total result should simply be the number of guard cells
  // overlapping any particular position.

  tester.out() << aa << endl << endl;

  int retval = tester.results("ump_test7");
  Pooma::finalize();
  return retval;
}
	     
// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ump_test7.cpp,v $   $Author: luchini $
// $Revision: 1.11 $   $Date: 2000/05/31 21:49:12 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
