// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Structs:
//   KernelTag1<Expr>
//   KernelTag<LHS,RHS>
//   CompressibleKernel<bool,bool>
// Tags:
//   InlineKernelTag
//   CompressibleViewKernelTag
//   CompressibleKernelTag
//-----------------------------------------------------------------------------

#ifndef POOMA_EVALUATOR_KERNELTAGS_H
#define POOMA_EVALUATOR_KERNELTAGS_H

//////////////////////////////////////////////////////////////////////

//-----------------------------------------------------------------------------
// Overview: 
// Kernel Tags are used for picking the appropriate Kernel given the
// engines in an expression.
//
// The external interface for KernelTags is the traits
// KernelTag1<Expr>::Tag_t
// KernelTag<LHS,RHS>::Tag_t
// which yields an evaluator tag, given the Array type or the types for the
// left hand side and right hand side.  To add new Kernels or new
// engines, specialize the KernelForEngine struct to give the Kernel
// tag for each engine, and specialize the NewKernel and KernelCombine
// struct to determine how to chose a new Kernel given two Kernels.
//
// Currently there are only three Kernels:
// InlineKernelTag            - simple loop
// CompressibleViewKernelTag  - takes a brickview of lhs then loops
// CompressibleKernelTag      - checks if both sides are compressed to
//                                  do compressed assign otherwise calls CVE
//
// The results for expressions with Bricks (B) and CompressibleBricks (C)
// are:
//
// B = B+B;   InlineKernelTag
// B = C+B;   InlineKernelTag
// B = C+C;   InlineKernelTag
// C = B+B;   CompressibleViewKernelTag
// C = C+B;   CompressibleViewKernelTag
// C = C+C;   CompressibleKernelTag
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Typedefs:
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Evaluator/CompressibleEngines.h"
#include "PETE/PETE.h"

//-----------------------------------------------------------------------------
// Forward Declarations:
//-----------------------------------------------------------------------------

template<class Expr> struct CreateLeaf;

//-----------------------------------------------------------------------------
//
// Full Description:
//
// Kernel Tags are used for picking the appropriate Kernel given the
// engines in an expression.  Each Kernel tag represents a set of engines
// that it is capable of dealing with.
//
// This file provides the following
// interface:
//
// KernelTag<LHSArray,RHSArray>::Kernel_t
//      - this is the main function that produces an Kernel tag for the
//        expression
//
// The implementation of this interface will probably change when other
// kernels are added.  Currently we only have the basic inline kernel plus two
// that deal with compression, so we pick the kernel tag by querying Compressible
// about the left and right hand sides.
//
// The following Kernel tags are currently defined:
//
// InlineKernelTag  - use the inline Kernel (simple loops, no patches)
// CompressibleViewKernelTag   - for a compressible lhs... takes a view
// CompressibleKernelTag       - deal with a compressible lhs and rhs
//
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
// Some Kernel tags (more tags can be defined elsewhere)
//-----------------------------------------------------------------------------

struct ErrorKernelTag 
{ 
  ErrorKernelTag(){}
  ~ErrorKernelTag(){}
};

struct InlineKernelTag 
{ 
  InlineKernelTag(){}
  ~InlineKernelTag(){}
};

struct CompressibleKernelTag 
{ 
  CompressibleKernelTag(){}
  ~CompressibleKernelTag(){}
};

struct CompressibleViewKernelTag 
{ 
  CompressibleViewKernelTag(){}
  ~CompressibleViewKernelTag(){}
};

//-----------------------------------------------------------------------------
// CompressibleKernel<bool,bool>
//
// Pick the appropriate kernel based on the compressibility of the left and
// right hand sides.
//-----------------------------------------------------------------------------

template<bool lhsComp,bool rhsComp>
struct CompressibleKernel
{
  CompressibleKernel(){}
  ~CompressibleKernel(){}
};

template<>
struct CompressibleKernel<false,false>
{
  CompressibleKernel(){}
  ~CompressibleKernel(){}
  typedef InlineKernelTag Kernel_t;
};

template<>
struct CompressibleKernel<false,true>
{
  CompressibleKernel(){}
  ~CompressibleKernel(){}
  typedef InlineKernelTag Kernel_t;
};

template<>
struct CompressibleKernel<true,false>
{
  CompressibleKernel(){}
  ~CompressibleKernel(){}
  typedef CompressibleViewKernelTag Kernel_t;
};

template<>
struct CompressibleKernel<true,true>
{
  CompressibleKernel(){}
  ~CompressibleKernel(){}
  typedef CompressibleKernelTag Kernel_t;
};


//-----------------------------------------------------------------------------
// KernelTag<LHS,RHS>
//
// Finally, this struct computes the Kernel tag for the whole expression
// given the types of the arrays on the left and right hand sides.
//-----------------------------------------------------------------------------

template<class Expr>
struct KernelTag1
{
  KernelTag1(){}
 ~KernelTag1(){}
  typedef typename Expr::Engine_t ExprEngine_t;
  typedef typename EngineFunctor<ExprEngine_t, Compressible>::Type_t Expr_t;
  enum { exprComp = Expr_t::val };
  typedef typename CompressibleKernel<exprComp,exprComp>::Kernel_t Kernel_t;
};


//-----------------------------------------------------------------------------
// KernelTag<LHS,RHS>
//
// Finally, this struct computes the Kernel tag for the whole expression
// given the types of the arrays on the left and right hand sides.
//-----------------------------------------------------------------------------

template<class LHS,class RHS>
struct KernelTag
{
  KernelTag(){}
 ~KernelTag(){}
  typedef typename LHS::Engine_t LHSEngine_t;
  typedef typename RHS::Engine_t RHSEngine_t;
  typedef typename EngineFunctor<LHSEngine_t,Compressible>::Type_t LHST_t;
  typedef typename EngineFunctor<RHSEngine_t,Compressible>::Type_t RHST_t;
  enum { lhsComp = LHST_t::val };
  enum { rhsComp = RHST_t::val };
  typedef typename CompressibleKernel<lhsComp,rhsComp>::Kernel_t Kernel_t;
};


#endif     // POOMA_EVALUATOR_KERNELTAGS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: KernelTags.h,v $   $Author: swhaney $
// $Revision: 1.12 $   $Date: 2000/03/28 23:31:50 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
