// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Simple Reductions on Remote Multipatch arrays.
//-----------------------------------------------------------------------------

// include files

#include "Pooma/Arrays.h"
#include "Pooma/Indices.h"
#include "Utilities/Tester.h"

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc,argv);

#if POOMA_CHEETAH

  Loc<1> blocks2(2), blocks5(5);
  UniformGridPartition<1> partition2(blocks2), partition5(blocks5);   
  UniformGridLayout<1> layout2(Interval<1>(10), partition2),
    layout5(Interval<1>(10), partition5);
  Array<1, int, MultiPatch<UniformTag, Remote<Brick> > > a(layout2),
    b(layout5);
  Array<1, int> c(10);

  Pooma::blockAndEvaluate();

  for (int i = 0; i < 10; i++)
    {
      a(i) = i + 1;
      b(i) = 2 * i;
      c(i) = i % 5;
    }

  int ret;
  bool bret;

  // Test various sorts of reductions with a single array.

  Reduction<MainEvaluatorTag>().evaluate(ret, OpAddAssign(), a);
  tester.check("sum", ret, 55);
  tester.out() << ret << std::endl;

  Reduction<MainEvaluatorTag>().evaluate(ret, OpMultiplyAssign(), 
    a(Interval<1>(9)));
  tester.check("prod", ret, 362880);
  tester.out() << ret << std::endl;

  Reduction<MainEvaluatorTag>().evaluate(ret, FnMinAssign(), a - 2);
  tester.check("min", ret, -1);
  tester.out() << ret << std::endl;

  Reduction<MainEvaluatorTag>().evaluate(bret, FnAndAssign(), a - 1);
  tester.check("all", bret, false);
  tester.out() << bret << std::endl;

  Reduction<MainEvaluatorTag>().evaluate(ret, OpBitwiseOrAssign(), a);
  tester.check("bitOr", ret, 15);
  tester.out() << ret << std::endl;

  // Test something with an expression engine (remote2 + remote5).

  Reduction<MainEvaluatorTag>().evaluate(ret, OpAddAssign(), a + b);
  tester.check("sum(a + b)", ret, 55 + 90);
  tester.out() << ret << std::endl;

  // Test something with an expression engine (remote5 + remote2).

  Reduction<MainEvaluatorTag>().evaluate(ret, OpAddAssign(), b + a);
  tester.check("sum(b + a)", ret, 90 + 55);
  tester.out() << ret << std::endl;

  // Test something with a brick (remote2 + remote5 + brick).

  Reduction<MainEvaluatorTag>().evaluate(ret, OpAddAssign(), a + b + c);
  tester.check("sum(a + b + c)", ret, 90 + 55 + 20);
  tester.out() << ret << std::endl;

  // Test something with a brick (brick + remote5 + remote2).

  Reduction<MainEvaluatorTag>().evaluate(ret, OpAddAssign(), c + b + a);
  tester.check("sum(c + b + a)", ret,  20 + 55 + 90);
  tester.out() << ret << std::endl;

  // Finish.

#endif // POOMA_CHEETAH

  int return_status = tester.results("ReductionTest4");

  Pooma::finalize();

  return return_status;
}


// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ReductionTest4.cpp,v $   $Author: rasmussn $
// $Revision: 1.2 $   $Date: 2000/05/03 18:58:59 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
