// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   ApplyFieldStencil     - Tag class for defining an engine capable of
//                           applying a Field-based stencil.
//   FieldStencil          - A wrapper class for a user-defined stencil.
//   Engine                - Specialization for ApplyFieldStencil
//   NewEngine             - Specializations for ApplyFieldStencil
//-----------------------------------------------------------------------------

#ifndef POOMA_FIELD_FIELDSTENCIL_H
#define POOMA_FIELD_FIELDSTENCIL_H

//-----------------------------------------------------------------------------
// Overview:
//
// This file contains the equipment required to write differential operators
// that take the form of stencil objects.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Interval.h"
#include "Engine/Engine.h"
#include "Geometry/DiscreteGeometry.h"
#include "Layout/INode.h"
#include "Layout/Node.h"
#include "PETE/ErrorType.h"

//-----------------------------------------------------------------------------
// Forward declarations
//-----------------------------------------------------------------------------

template <int Dim>
class DomainLayout;

template<class Functor> class FieldStencil;

//-----------------------------------------------------------------------------
// Full Description:
// 
// ApplyFieldStencil is just a tag class for the field-stencil-application
// engine.
//-----------------------------------------------------------------------------

template <class Functor, class Expression>
struct ApplyFieldStencil;


//-----------------------------------------------------------------------------
// Full Description:
// 
// Engine<Dim, T, ApplyFieldStencil<Functor, Expression> > is a specialization
// of Engine for ApplyFieldStencil<Functor>. It uses the supplied stencil 
// object to apply an arbitrary operation to the input field.
//-----------------------------------------------------------------------------

template<int Dim, class T, class Functor, class Expression>
class Engine<Dim, T, ApplyFieldStencil<Functor, Expression> >
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and constants

  typedef ApplyFieldStencil<Functor, Expression>   Tag_t;
  typedef Functor                                  Functor_t;
  typedef Expression                               Expression_t;
  typedef Engine<Dim, T, Tag_t>                    This_t;
  typedef This_t                                   Engine_t;
  typedef Interval<Dim>                            Domain_t;
  typedef T                                        Element_t;
  typedef ErrorType                                ElementRef_t;
  typedef typename Expression_t::Engine_t          ExprEngine_t;
  typedef DomainLayout<Dim>                        Layout_t;

  enum { dimensions = Dim };
  enum { hasDataObject = ExprEngine_t::hasDataObject };
  enum { dynamic = false };
  enum { zeroBased = ExprEngine_t::zeroBased };
  enum { multiPatch = ExprEngine_t::multiPatch };

  //---------------------------------------------------------------------------
  // Construct from a stencil, an input field, and a new total domain.
  // Note: the way domains work with FieldStencils is a little screwy.
  // When originally constructing a FieldStencil, the domain of the engine
  // must match the total domain of the stenciled field in order for the
  // indexing to work. This is why the zeroBased trait above is false.

  Engine(const Functor_t &functor, const Expression_t &f, 
	 const Interval<Dim> &domain)
    : domain_m(domain), field_m(f), functor_m(functor)
    {
      for (int d = 0; d < Dim; ++d)
        {
          firsts_m[d] = domain[d].first();
          offset_m[d] = 0;
        }
    }

  //---------------------------------------------------------------------------
  // Construct from another ApplyFieldStencil and an Interval. This is 
  // simpler than with other domains since we just need to bump the offset.

  Engine(const This_t &e, const Interval<Dim> &domain)
    : domain_m(Pooma::NoInit()), field_m(e.field()), functor_m(e.functor())
    {
      for (int d = 0; d < Dim; ++d)
        {
          domain_m[d] = Interval<1>(domain[d].length());
          offset_m[d] = e.offset_m[d] + domain[d].first();
          firsts_m[d] = 0;
        }   
    }    

  //---------------------------------------------------------------------------
  // Construct from a new expression and another ApplyFieldStencil.

  template<class OtherExp>
  Engine(const Expression &f,
	 const Engine<Dim, T, ApplyFieldStencil<Functor, OtherExp> > &e)
    : domain_m(e.domain()),
      field_m(f),
      functor_m(e.functor())
  {
    for (int d = 0; d < Dim; ++d)
    {
      offset_m[d] = e.offset(d);
      firsts_m[d] = e.first(d);
    }   
  }    

  //---------------------------------------------------------------------------
  // Construct from an ApplyFieldStencilNew with a different expression
  // and an INode.

  template<class OtherExp>
  Engine(const Engine<Dim, T, ApplyFieldStencil<Functor, OtherExp> > &e,
	 const INode<Dim> &inode)
    : domain_m(Pooma::NoInit()),
      field_m(e.field()(e.viewDomain(inode))),
      functor_m(e.functor())
  {
    for (int d = 0; d < Dim; ++d)
    {
      domain_m[d] = Interval<1>(inode.domain()[d].length());
      offset_m[d] = e.functor().lowerExtent(d);
      firsts_m[d] = 0;
    }   
  }    

  //---------------------------------------------------------------------------
  // Copy constructor.

  Engine(const This_t &model)
    : domain_m(model.domain()), field_m(model.field()), 
      functor_m(model.functor())
    {
      for (int d = 0; d < Dim; ++d)
        {
          offset_m[d] = model.offset_m[d];
          firsts_m[d] = model.firsts_m[d];
        }   
    }    

  //---------------------------------------------------------------------------
  // Element access via ints for speed.

  inline Element_t read(int i) const 
    {
      return functor_m(field(),
		       i + offset_m[0]
		       );
    }
  inline Element_t read(int i, int j) const 
    {
      return functor_m(field(),
		       i + offset_m[0],
		       j + offset_m[1]
		       );
    }
  inline Element_t read(int i, int j, int k) const 
    {
      return functor_m(field(),
		       i + offset_m[0],
		       j + offset_m[1],
		       k + offset_m[2]
		       );
    }

  inline Element_t read(const Loc<1> &loc) const 
    {
      return functor_m(field(),
		       loc[0].first() + offset_m[0]
		       );
    }
  inline Element_t read(const Loc<2> &loc) const 
    {
      return functor_m(field(),
		       loc[0].first() + offset_m[0],
		       loc[1].first() + offset_m[1]
		       );
    }
  inline Element_t read(const Loc<3> &loc) const 
    {
      return functor_m(field(),
		       loc[0].first() + offset_m[0],
		       loc[1].first() + offset_m[1],
		       loc[2].first() + offset_m[2]
		       );
    }

  inline Element_t operator()(int i) const 
    {
      return read(i);
    }
  inline Element_t operator()(int i, int j) const 
    {
      return read(i, j);
    }
  inline Element_t operator()(int i, int j, int k) const 
    {
      return read(i, j, k);
    }
  inline Element_t operator()(const Loc<Dim> &loc) const 
    {
      return read(loc);
    }

  //---------------------------------------------------------------------------
  // Return the domain.

  inline const Domain_t &domain() const { return domain_m; }

  //---------------------------------------------------------------------------
  // Return the first value for the specified direction.
  
  inline int first(int i) const
  {
    PAssert(i >= 0 && i < Dim);
    return firsts_m[i];
  }

  inline int offset(int i) const
  {
    PAssert(i >= 0 && i < Dim);
    return offset_m[i];
  }
  
  //---------------------------------------------------------------------------
  // Accessors.

  inline const Expression_t &field() const { return field_m; }
  inline const Functor_t &functor() const { return functor_m; }

  //---------------------------------------------------------------------------
  // Need to pass lock requests to the contained engine.

  template<class RequestType>
  inline
  typename DataObjectRequest<RequestType>::Type_t
  dataObjectRequest(const DataObjectRequest<RequestType> &req) const
    {
      return field().engine().dataObjectRequest(req);
    }

  //---------------------------------------------------------------------------
  // viewDomain() gives the region of the expression needed to compute a given
  // region of the stencil.
  //---------------------------------------------------------------------------

  inline
  Interval<Dim> viewDomain(const Interval<Dim> &domain) const
  {
    Interval<Dim> ret;
    int d;
    for (d = 0; d < Dim; ++d)
    {
      ret[d] =
	Interval<1>(
		    domain[d].first() + offset_m[d] - functor().lowerExtent(d),
		    domain[d].last() + offset_m[d] + functor().upperExtent(d)
		    );
    }

    return ret;
  }

  inline
  INode<Dim> viewDomain(const INode<Dim> &inode) const
  {
    return INode<Dim>(inode, viewDomain(inode.domain()));
  }

  inline
  Interval<Dim> intersectDomain() const
  {
    Interval<Dim> ret;
    int d;
    for (d = 0; d < Dim; ++d)
    {
      ret[d] =
	Interval<1>(
		    domain_m[d].first() + offset_m[d],
		    domain_m[d].last() + offset_m[d]
		    );
    }

    return ret;
  }

private:

  Interval<Dim> domain_m;
  Expression_t field_m;
  Functor_t functor_m;
  int offset_m[Dim];
  int firsts_m[Dim];
};

//-----------------------------------------------------------------------------
// Full Description:
// NewEngine<Engine,SubDomain>
//
// Specializations of NewEngine for subsetting a constant-function-engine with
// an arbitrary domain. 
//-----------------------------------------------------------------------------

template <int Dim, class T, class F, class E>
struct NewEngine<Engine<Dim, T, ApplyFieldStencil<F,E> >, Interval<Dim> >
{
  typedef Engine<Dim, T, ApplyFieldStencil<F,E> > Type_t;
};

template <int Dim, class T, class F, class E>
struct NewEngine<Engine<Dim, T, ApplyFieldStencil<F,E> >, INode<Dim> >
{
  typedef typename View1<E, INode<Dim> >::Type_t NewExpr_t;
  typedef ApplyFieldStencil<F, NewExpr_t> NewTag_t;
  typedef Engine<Dim, T, NewTag_t > Type_t;
};

//-----------------------------------------------------------------------------
// View types for stencil objects.  Stencils define operator() to return a
// stencil engine object.  If you wanted to store that object, you could write:
//
// A a;
// Laplace laplace;
// typename View1<FieldStencil<Laplace>,A>::Type_t b = laplace(a);
//-----------------------------------------------------------------------------

template<class Functor, class G, class T, class E>
struct View1<FieldStencil<Functor>, ConstField<G,T,E> >
{
  //---------------------------------------------------------------------------
  // Exported typedefs and constants

  typedef typename Functor::OutputCentering_t OutputCentering_t;
  typedef typename Functor::OutputElement_t OutputElement_t;
  typedef ConstField<G,T,E> InputField_t;  

  typedef G InputGeometry_t;
  typedef typename InputGeometry_t::Mesh_t InputMesh_t;
  typedef typename InputGeometry_t::Centering_t InputCentering_t;

  typedef DiscreteGeometry<OutputCentering_t, InputMesh_t> OutputGeometry_t;
  typedef ApplyFieldStencil<Functor, InputField_t> OutputEngineTag_t;
  typedef ConstField<OutputGeometry_t, OutputElement_t, OutputEngineTag_t>
    Type_t;
    
  enum { outputDim = InputGeometry_t::dimensions };
  typedef Engine<outputDim, OutputElement_t, OutputEngineTag_t>
    OutputEngine_t;

  //---------------------------------------------------------------------------
  // This function actually coughs up the output ConstField. The output
  // field has the same type as the input field, except that the input
  // centering is replaced by the output centering. The physical domain (PD)
  // of the output field comes from the PD of the output geometry. This 
  // implies that a stencil cannot shave off layers of the PD. The guard
  // layers of the resulting field are a little tricky to compute. There are
  // two effects: (1) the extent of the stencil and (2) the change in
  // centering.
  // The extent of the stencil is pretty simple to analyze: you lose a layer
  // of guard elements for every point the stencil extends. The change in
  // centering follows the rules: Vert -> Vert or Cell -> Cell - no change,
  // Vert -> Cell - add 1 upper layer, Cell -> Vert - subtract 1 upper layer.
  
  // Note: assumes that the output centering can be constructed with a default
  // constructor.

  static inline
  Type_t make(const FieldStencil<Functor> &s, const InputField_t &f) 
    {
      GuardLayers<outputDim> og(f.geometry().guardLayers());
      for (int d = 0; d < outputDim; d++)
        {
          og.lower(d) -= s.functor().lowerExtent(d);
          og.upper(d) -= s.functor().upperExtent(d);
          if (f.geometry().centering().centeringTag(d) == vert &&
              OutputCentering_t().centeringTag(d) == cell)
            og.upper(d)++;
          if (f.geometry().centering().centeringTag(d) == cell &&
              OutputCentering_t().centeringTag(d) == vert)
            og.upper(d)--;             
        }
      
      OutputGeometry_t g(f.geometry().mesh(), og);
      return Type_t(g, OutputEngine_t(s.functor(), f, g.totalDomain()), 
        f.bconds());
    }  
};

//-----------------------------------------------------------------------------
// Full Description:
//
// FieldStencil is used to wrap a user-defined field-based stencil class.
// The idea is to encapsulate the majority of the crazy type manipulations
// required to generate the output ConstField and the calculation of the
// new number of guard layers.
//
// To create a stencil, users must create a class similar to the one below,
// which computes a central difference divergence of a Vert-centered Field
// and maps it to a Cell-centered Field:
//
// template<class OutputCentering, class Geometry, class T>
// class Div { };
//  
// template<int Dim, class T1, class T2>
// class Div<Cell,
//   DiscreteGeometry<Vert, UniformRectilinearMesh<Dim, Cartesian<Dim>, T1> >, 
//   Vector<Dim, T2> >
// {
// public:
// 
//   typedef Cell OutputCentering_t;
//   typedef T2 OutputElement_t;
// 
//   int lowerExtent(int) const
//     {
//       return 1;
//     }
// 
//   int upperExtent(int) const
//     {
//       return 1;
//     }
//         
//   template<class F>
//   inline OutputElement_t
//   operator()(const F &f, int i1) const
//     {
//       return (f(i1 + 1)(0) - f(i1 - 1)(0)) / 
//         f.geometry().mesh().meshSpacing(0);
//     }
// 
//   template<class F>
//   inline OutputElement_t
//   operator()(const F &f, int i1, int i2) const
//     {
//       return (f(i1 + 1, i2)(0) - f(i1 - 1, i2)(0)) / 
//         f.geometry().mesh().meshSpacing()(0) +
//         (f(i1, i2 + 1)(1) - f(i1, i2 - 1)(1)) / 
//         f.geometry().mesh().meshSpacing()(1);
//     }
// };
//
// There are 2 required typedefs: OutputCentering_t and OutputElement_t. 
// These export the type of the output centering and the type resulting 
// from applying the stencil at a point. 
//
// Then, there are two accessors: lowerExtent(int dir) and 
// upperExtent(int dir). These return the extent of the stencil as a function 
// of direction. As another example, a forward difference would have a lower
// extent of 0 and an upper extent of 1. Finally, a series of inline apply()
// functions, which take a Field of some sort and a set indices, must be
// supplied. This is what actually computes the stencil.
//
// A ConstField that contains an ApplyFieldStencil-engine that operates on
// a Field f, is constructed by using operator()() for FieldStencil:
//
// View1<FieldStencil<Div<OutputCentering, Geometry, T> >, 
//  ConstField<Geometry, T, EngineTag> >::make(
//  Div<OutputCentering, Geometry, T>(), f);
// 
//-----------------------------------------------------------------------------

template<class Functor>
class FieldStencil
{
public:

  FieldStencil()
  { }

  FieldStencil(const Functor &functor)
    : functor_m(functor)
  { }

  template<class Init>
  FieldStencil(const Init &init)
    : functor_m(init)
  { }

  ~FieldStencil() { }

  template<class G, class T, class E>
  typename View1<FieldStencil<Functor>, ConstField<G,T,E> >::Type_t
  operator()(const ConstField<G,T,E> &expr) const
  {
    typedef View1<FieldStencil<Functor>, ConstField<G,T,E> > Ret_t;
    return Ret_t::make(*this, expr);
  }
  
  inline const Functor &functor() const { return functor_m; }

private:

  // Store the functor here.

  Functor functor_m;
};


//-----------------------------------------------------------------------------
// Specializations for selecting the appropriate evaluator for the Stencil
// engine.  We just get the appropriate types from the Expression's engine.
//-----------------------------------------------------------------------------

template<class Functor, class Expression>
struct EvaluatorEngineTraits<ApplyFieldStencil<Functor, Expression> >
{
  typedef typename CreateLeaf<Expression>::Leaf_t Expr_t;
  typedef typename
    ForEach<Expr_t, EvaluatorTypeTag, EvaluatorCombineTag>::Type_t
      Evaluator_t;
};


//-----------------------------------------------------------------------------
// FieldStencilIntersector is a special intersector that gets used when we come
// across a stencil object in an expression.
//-----------------------------------------------------------------------------

template<int Dim, class Intersect>
class FieldStencilIntersector
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and constants

  typedef typename Intersect::IntersectorData_t         IntersectorData_t;
  typedef FieldStencilIntersector<Dim, Intersect>       This_t;
  typedef typename IntersectorData_t::const_iterator    const_iterator;
  typedef RefCountedPtr<IntersectorData_t>              DataPtr_t;
  typedef Interval<Dim>                                 Domain_t;
  
  enum { dimensions = Intersect::dimensions };
  
  //---------------------------------------------------------------------------
  // Constructors

  FieldStencilIntersector(const This_t &model)
    : domain_m(model.domain_m), intersector_m(model.intersector_m)
  { }

  FieldStencilIntersector(const Domain_t &dom, const Intersect &intersect)
    : domain_m(dom), intersector_m(intersect)
  { }

  This_t &operator=(const This_t &model)
  {
    if (this != &model)
    {
      domain_m = model.domain_m;
      intersector_m = model.intersector_m;
    }
  }

  ~FieldStencilIntersector() { }

  inline DataPtr_t &data() { return intersector_m.data(); }
  inline const DataPtr_t &data() const { return intersector_m.data(); }

  //---------------------------------------------------------------------------
  // Accessors

  // STL iterator support.
  
  inline const_iterator begin() const { return data()->inodes_m.begin(); }
  inline const_iterator end() const { return data()->inodes_m.end(); }

  //---------------------------------------------------------------------------
  // Intersect routines

  // All domains.
  
  template<class Engine>
  inline void intersect(const Engine &engine) 
  {
    typedef typename NewEngine<Engine, Interval<Dim> >::Type_t NewEngine_t;

    NewEngine_t newEngine(engine, domain_m);

    intersector_m.intersect(newEngine);

    data()->shared(engine.layout().ID(), newEngine.layout().ID());
  }

  template<class Engine, int Dim2>
  inline bool intersect(const Engine &engine, const GuardLayers<Dim2> &) 
  {
    intersect(engine);
    return true;
  }

private:

  
  Interval<Dim> domain_m;
  Intersect     intersector_m;
};


//-----------------------------------------------------------------------------
// IntersectEngine specialization
//-----------------------------------------------------------------------------

template <int Dim, class T, class Functor, class Expression, class Intersect>
struct LeafFunctor<Engine<Dim, T, ApplyFieldStencil<Functor,Expression> >,
  EngineApply<IntersectorTag<Intersect> > >
{
  typedef int Type_t;

  static
  int apply(const Engine<Dim, T, ApplyFieldStencil<Functor,Expression> > 
	    &engine, const EngineApply<IntersectorTag<Intersect> > &tag)
  {
    // We offset the domain to get a domain in the viewed engine that
    // the stencil looks at.  The intersection is performed with a view
    // of the contained engine over this domain.  The resulting answer works
    // even though the stencil looks beyond this domain, because the viewed
    // field guarantees enough guard layers for the stencil to work.
    // (Presently this assumption isn't checked anywhere, so a lack of guard
    // cells results in an error in the multipatch inode view.)

    typedef FieldStencilIntersector<Dim, Intersect> NewIntersector_t;
    NewIntersector_t newIntersector(engine.intersectDomain(),
				    tag.intersector_m);
    EngineApply<IntersectorTag<NewIntersector_t> > newTag(newIntersector); 

    forEach(engine.field(), newTag, NullCombine());
    return 0;
  }
};

//---------------------------------------------------------------------------
// Specialization of  DataObjectRequest engineFunctor to pass the request to
// the contained engine.
//---------------------------------------------------------------------------

template<class RequestType> class DataObjectRequest;

template <int Dim, class T, class Functor, class Expression, class RequestType>
struct EngineFunctor<Engine<Dim, T, ApplyFieldStencil<Functor,Expression> >,
  DataObjectRequest<RequestType> >
{
  typedef typename DataObjectRequest<RequestType>::Type_t Type_t;

  static Type_t
  apply(const Engine<Dim, T, ApplyFieldStencil<Functor, Expression> > &engine,
	const DataObjectRequest<RequestType> &tag)
  {
    return engineFunctor(engine.field().engine(), tag);
  }
};

//-----------------------------------------------------------------------------
//
// The generic version of EngineView just accesses the contained engine and
// applies EngineView to it.
//
// The default version doesn't fiddle with the domain, since it is assumed
// that the typical view doesn't need to.  Specializations will be required
// for INode views etc...  Probably we should come up with a generic approach.
//
//-----------------------------------------------------------------------------

template <int Dim, class T, class Functor, class Expression, class Tag>
struct LeafFunctor<Engine<Dim, T, ApplyFieldStencil<Functor,Expression> >,
  EngineView<Tag> >
{
  typedef LeafFunctor<Expression, EngineView<Tag> > LeafFunctor_t;
  typedef typename LeafFunctor_t::Type_t NewViewed_t;
  typedef Engine<Dim, T, ApplyFieldStencil<Functor, NewViewed_t> > Type_t;

  static
  Type_t apply(const Engine<Dim, T,
	       ApplyFieldStencil<Functor, Expression> > &engine,
	       const EngineView<Tag> &tag)
  {
    return Type_t(LeafFunctor_t::apply(engine.field(), tag),
		  engine
		  );
  }
};

template <int Dim, class T, class Functor, class Expression, class Tag>
struct LeafFunctor<Engine<Dim, T, ApplyFieldStencil<Functor,Expression> >,
  EngineApply<Tag> >
{
  typedef LeafFunctor<Expression, EngineApply<Tag> > LeafFunctor_t;
  typedef typename LeafFunctor_t::Type_t Type_t;

  static
  Type_t apply(const Engine<Dim, T,
	       ApplyFieldStencil<Functor, Expression> > &engine,
	       const EngineApply<Tag> &tag)
  {
    return LeafFunctor_t::apply(engine.field(), tag);
  }
};

#endif // POOMA_FIELD_FIELDSTENCIL_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldStencil.h,v $   $Author: julianc $
// $Revision: 1.32 $   $Date: 2000/07/20 22:25:43 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
