// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// ----------------------------------------------------------------------------
// Scalar advection example, illustrating use of Field, Mesh,
// DiscreteGeometry, and the canned second-order divergence operator
// div<Cell>()
// ----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Clock.h"

#include <iostream>
#include <string>
#include <stdio.h>
using namespace std;

// Forward declarations:
// Ensight output:
void ensightCaseOut(char *fn, int increment, int lastTimeStep, double dt);
template<int Dim>
void ensightGeometryOut(char *fn, UniformRectilinearMesh<Dim> &mesh);
template<class T, class EngineTag>
void ensightVariableOut(char *fn, const Array<1, T, EngineTag> &a);
template<class T, class EngineTag>
void ensightVariableOut(char *fn, const Array<2, T, EngineTag> &a);
template<class T, class EngineTag>
void ensightVariableOut(char *fn, const Array<3, T, EngineTag> &a);

// Workaround for not-yet-correct canned stencil-based div():
template<class Geometry, class EngineTag>
void divv(Field<Geometry, Vector<1,double>, EngineTag> &v,
          Field<Geometry, double, EngineTag> &s);
template<class Geometry, class EngineTag>
void divv(Field<Geometry, Vector<2,double>, EngineTag> &v,
          Field<Geometry, double, EngineTag> &s);
template<class Geometry, class EngineTag>
void divv(Field<Geometry, Vector<3,double>, EngineTag> &v,
          Field<Geometry, double, EngineTag> &s);


// SAOptions - options parsing class.
// Member functions defined after main().

template <int D>
class SAOptions;

template <class OStr>
void usage(const std::string &name, OStr &out);

template <int D, class OStr>
void print(const SAOptions<D> &opts, OStr &out);

template <int D>
class SAOptions
{
public:

  double epsilon;           // Purge threshold
  bool purge;               // To purge or not to purge
  bool doTextOut;           // Text output of u, every outputIncrement steps
  bool doEnsightOut;        // Ensight output of u, every outputIncrement steps
  bool doSumOut;            // Output sum(u), every outputIncrement steps
  int lastTimeStep;         // Last time step
  int outputIncrement;      // Frequency of optional output
  int purgeIncrement;       // How often to purge
  int nPatches[D];          // Number of patches
  int nCells[D];            // Number of cells
  int nVerts[D];            // Number of verts (calculated from nCells)
  int pulseHalfWidthCells;  // Half-width (in cells) of (symmetrical) loaded
                            // pulse
  std::string progname;     // Name of program.
  
  // Default constructor, sets the default options.
  
  SAOptions();
  
  // Set options from argc, argv.
  
  SAOptions(int argc, char *argv[]);
  
  // Prints a summary of the options.
  
  template <class OStr>
  void print(OStr &out) { ::print(*this, out); }
  
  // Prints a usage message.
  
  template <class OStr>
  void usage(OStr &out) { ::usage(progname, out); }

private:

  // Helper functions used in parsing options with int and double arguments:
  
  static bool intArgument(int argc, char **argv, int pos, int &val);
  static bool doubleArgument(int argc, char **argv, int pos, double &val);
  
  // Report bad option value
  // (These are forwarded to globals as I don't want to put the
  // bodies inline here. Fix when CW is fixed.)
  
  template <class OStr>
  static void 
  badValue(OStr &out, const std::string &option) { ::badValue(out,option); }
  
  template <class OStr>
  static void 
  badOption(OStr &out, const char *str, const std::string &option) 
  { ::badOption(out,str,option); }
  
};

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv); // Initialize the library
  Inform pout(NULL, 0);         // Output, via process 0 only
  // PrintArray class, to format output as desired:
  PrintArray pa(3, 10, 3, 6, true, 1);

  const int Dim = 2; // Set the dimensionality
  
  // The SAOptions object sets the default option values
  // and parses argv for options that override the defaults.
  
  SAOptions<Dim> opts(argc, argv);
  
  opts.print(pout);

  // Create the physical domains:
  Interval<Dim> vertexDomain, cellDomain;
  int d;
  for (d = 0; d < Dim; d++) {
    vertexDomain[d] = Interval<1>(opts.nVerts[d]);
    cellDomain[d] = Interval<1>(opts.nCells[d]);
  }

  // Create the (uniform, logically rectilinear) mesh.
  Vector<Dim> origin(0.0), spacings(0.2);
  typedef UniformRectilinearMesh<Dim> Mesh_t;
  Mesh_t mesh(vertexDomain, origin, spacings);

  // Create two geometry objects - one allowing 1 guard layer to account for
  // stencil width and another with no guard layers to support temporaries:
  typedef DiscreteGeometry<Cell, UniformRectilinearMesh<Dim> > Geometry_t;
  Geometry_t geom(mesh, GuardLayers<Dim>(1));
  Geometry_t geomNG(mesh);

  // Create the layouts
  Loc<Dim> patches(8);
  for (d = 0; d < Dim; d++) { patches[d] = Loc<1>(opts.nPatches[d]); }
  GridLayout<Dim> layoutc(cellDomain, patches, 
                          GuardLayers<Dim>(1), GuardLayers<Dim>(1));
  GridLayout<Dim> layoutcNG(cellDomain, patches);
  typedef MultiPatch<GridTag, CompressibleBrick> MP_t;

  // Create the Fields:

  // The flow Field u(x,t), a duplicate (stored at the previous
  // timestep for staggered leapfrog), and a useful temporary:
  Field<Geometry_t, double, MP_t> 
    u(geom, layoutc), uPrev(geomNG, layoutcNG), uTemp(geomNG, layoutcNG),
    uTemp2(geomNG, layoutcNG);
  // Needed for div() bug WORKAROUND below:
  Field<Geometry_t, Vector<Dim,double>, MP_t> uv(geom, layoutc);

  // Initialize Fields to zero everywhere, even global guard layers:
  u.all() = 0.0;

  // Set up periodic boundary conditions on all mesh faces:
  u.addBoundaryConditions(AllPeriodicFaceBC());

  // Load initial condition u(x,0), a symmetric pulse centered around nCells/4
  // and decaying to zero away from nCells/4 all directions, with a height of
  // 1.0, with a half-width equal to opts.pulseHalfWidthCells (defaults to
  // nCells/8):
  double pulseHalfWidth = spacings(0)*opts.pulseHalfWidthCells;
  Loc<Dim> pulseCenter;
  for (d = 0; d < Dim; d++) { pulseCenter[d] = Loc<1>(opts.nCells[0]/4); }
  Pooma::blockAndEvaluate(); // Needed pre-scalar-indexing read
  Vector<Dim> u0 = u.x(pulseCenter);
  u = 1.0 * exp(-dot(u.x() - u0, u.x() - u0) / (2.0 * pulseHalfWidth));

  // Be sure to apply the boundary conditions after scalar code.
  //  u.applyBoundaryConditions();

  const Vector<Dim> v(0.2);   // Propagation velocity
  const double dt = 0.1;      // Timestep

  // Set up Ensight Case and DiscreteGeometry files:
  if (opts.doEnsightOut) {
    ensightCaseOut("ScalarAdvection.case", opts.outputIncrement, 20, dt*20);
    ensightGeometryOut("ScalarAdvection.geo", mesh);
  }

  // Print out information at t = 0:
  if (opts.purge) {
    u = where((fabs(u) < opts.epsilon), 0.0); // Purge initial conditions
  }
  Pooma::blockAndEvaluate(); // Needed for compressedFraction() validity
  pout << "t = " << double(0.0) << " ; compressedFraction(u) = "
       << compressedFraction(u) << std::endl;
  if (opts.doSumOut) pout << "sum(u) = " << sum(u) << std::endl;
  char timeStepFileName[6];
  if (opts.doEnsightOut) {
    sprintf(timeStepFileName, "u.%04d", 0);
    ensightVariableOut(timeStepFileName, average<Vert>(u).arrayRead());
  }
  if (opts.doTextOut) { pa.print(pout, u); }


  // Prime the leapfrog by setting the field at the previous timestep using the
  // initial conditions:
  uPrev = u;

  // Do a preliminary timestep using forward Euler, using the canned POOMA
  // div() function:
  // BROKEN  u -= div<Cell>(v * dt * u);
  // WORKAROUND:
  // Use the Field<Vector> temporay for most straightforward r1 comparison:
  uv.all() = u.all() * v * dt;
  divv(uv, uTemp2);
  u -= uTemp2;
  // WORKAROUND.

  // Start timer to time main loop; do blockAndEvaluate first so results are
  // meaningful:
  Pooma::blockAndEvaluate();
  double startTime = Pooma::Clock::value();

  // Now use staggered leapfrog (second-order) for the remaining timesteps.
  // The spatial derivative is just the second-order finite difference in the
  // canned POOMA stencil-based divergence operator div():
  for (int timestep = 2; timestep <= opts.lastTimeStep; timestep++) {
    uTemp = u;
    
    // BROKEN    u = uPrev - 2.0 * div<Cell>(v * dt * u);
    // WORKAROUND:
    // Use the Field<Vector> temporay for most straightforward r1 comparison:
    uv.all() = u.all() * v * dt;
    divv(uv, uTemp2);
    u = uPrev - 2.0 * uTemp2;
    // WORKAROUND.
    //tjw workaround.
    if (opts.purge) {
      if ((timestep % opts.purgeIncrement) == 0) {
        //WCY        u = where((abs(u) < opts.epsilon), 0.0);
        u = where((fabs(u) < opts.epsilon), 0.0); // WORKAROUND for WCY
      }
    }
    if ((timestep % opts.outputIncrement) == 0) {
      Pooma::blockAndEvaluate(); // Needed for compressedFraction() validity
      pout << "t = " << timestep*dt << " ; compressedFraction(u) = "
                << compressedFraction(u) << std::endl;
      if (opts.doSumOut) pout << "sum(u) = " << sum(u) << std::endl;
      if (opts.doEnsightOut) {
        //      u.applyBoundaryConditions(); // Update prior to output.
        sprintf(timeStepFileName, "u.%04d", timestep);
        ensightVariableOut(timeStepFileName, average<Vert>(u).arrayRead());
      }
      if (opts.doTextOut) { pa.print(pout, u); }
    }
    uPrev = uTemp;
  }

  // Compute the wallclock time for the loop; first do blockAndEvaluate() to
  // make sure the calculation has actually been done first:
  Pooma::blockAndEvaluate();
  double wallTime = Pooma::Clock::value() - startTime;

  pout << "Done. Wallclock seconds = " << wallTime << std::endl;
  Pooma::finalize();
  return 0;
}


// WORKAROUND: divv() functions, to use intil div() works; these are *only*
// correct if both input and output Field's are Cell-centered on
// UniformRectilinear meshes.

template<class Geometry, class EngineTag>
void divv(Field<Geometry, Vector<1,double>, EngineTag> &v,
          Field<Geometry, double, EngineTag> &s)
{
  Interval<1> I = v.geometry().mesh().physicalCellDomain()[0];
  Geometry::Mesh_t::PointType_t dx2i = v.geometry().mesh().meshSpacing();
  dx2i = 0.5/dx2i;
  s(I) = (v.comp(0)(I + 1) - v.comp(0)(I - 1))*dx2i(0);
}

template<class Geometry, class EngineTag>
void divv(Field<Geometry, Vector<2,double>, EngineTag> &v,
          Field<Geometry, double, EngineTag> &s)
{
  Interval<1> I = v.geometry().mesh().physicalCellDomain()[0];
  Interval<1> J = v.geometry().mesh().physicalCellDomain()[1];
  Geometry::Mesh_t::PointType_t dx2i = v.geometry().mesh().meshSpacing();
  dx2i = 0.5/dx2i;
  s(I,J) = 
    (v.comp(0)(I + 1, J) - v.comp(0)(I - 1, J))*dx2i(0) +
    (v.comp(1)(I, J + 1) - v.comp(1)(I, J - 1))*dx2i(1);
}

template<class Geometry, class EngineTag>
void divv(Field<Geometry, Vector<3,double>, EngineTag> &v,
          Field<Geometry, double, EngineTag> &s)
{
  Interval<1> I = v.geometry().mesh().physicalCellDomain()[0];
  Interval<1> J = v.geometry().mesh().physicalCellDomain()[1];
  Interval<1> K = v.geometry().mesh().physicalCellDomain()[2];
  Geometry::Mesh_t::PointType_t dx2i = v.geometry().mesh().meshSpacing();
  dx2i = 0.5/dx2i;
  s(I,J,K) = 
    (v.comp(0)(I + 1, J, K) - v.comp(0)(I - 1, J, K))*dx2i(0) +
    (v.comp(1)(I, J + 1, K) - v.comp(1)(I, J - 1, K))*dx2i(1) +
    (v.comp(2)(I, J, K + 1) - v.comp(2)(I, J, K - 1))*dx2i(2);
}


// Non-inline function definitions for SAOptions.
  
template <int D>
SAOptions<D>::
SAOptions()
  : epsilon(1.0e-4),
    purge(false),
    doTextOut(false),
    doEnsightOut(false),
    doSumOut(false),
    lastTimeStep(100),
    outputIncrement(10),
    purgeIncrement(1)
{
  for (int d = 0; d < D; ++d)
    {
      nPatches[d] = 10;
      nCells[d]   = 100;
      nVerts[d]   = nCells[d] + 1;
    }
  pulseHalfWidthCells = nCells[0]/8.0;
}
  
template <int D>
SAOptions<D>::
SAOptions(int argc, char *argv[])
{
    // Set the defaults (default copy constructor OK)
    
    *this = SAOptions();
    progname = argv[0];
    
    // Parse the argument list...
    
    int i = 1;
    while (i < argc)
      {
        using std::string;
    
        bool hasarg = false;
        
        string arg(argv[i]);
        
        if (arg == "-help")
          {
            usage(std::cerr);
            exit(0);
          }
        else if (arg == "-purge")
          {
            purge = true;
            
            // Check for optional argument:
            
	    hasarg = intArgument(argc, argv, i+1, purgeIncrement);
	    if (hasarg) ++i;	    
          }
        else if (arg == "-text")
          {
            doTextOut = true;
          }
        else if (arg == "-ensight")
          {
            doEnsightOut = true;
          }
        else if (arg == "-sum")
          {
            doSumOut = true;
          }
        else if (arg == "-steps")
          {
            if (i+1 == argc) badOption(std::cerr, 
                                       "No value specified for: ", arg);
            
	    hasarg = intArgument(argc, argv, i+1, lastTimeStep);
	    
	    if (!hasarg) badOption(std::cerr, 
                                   "No value specified for: ", arg);
	    
	    ++i;
          }
        else if (arg == "-out")
          {
            if (i+1 == argc) badOption(std::cerr, 
                                       "No value specified for: ", arg);
            
      	    hasarg = intArgument(argc, argv, i+1, outputIncrement);
	    
	    if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
	    
	    ++i;
          }
        else if (arg == "-cells")
          {
            // This can be followed by either 1 int or D ints.
            
            bool hasarg = intArgument(argc, argv, i+1, nCells[0]);
            
            if (hasarg) 
              { 
                ++i;
                if (D > 1)
                  {
                    bool moreArgs = intArgument(argc, argv, i+1, nCells[1]);
                    if (moreArgs)
                      {
                        for (int d = 1; d < D; ++d)
                          {
                            hasarg = intArgument(argc, argv, i+1, nCells[d]);
                            if (!hasarg) 
                              badOption(std::cerr, 
                                        "Not enough arguments for: ", arg);
                            ++i;
                          }
                      }
                    else
                      {
                        for (int d = 1; d < D; ++d)
                          {
                            nCells[d] = nCells[0];
                          }
                      }
                  }
              }
            else
              {
                badOption(std::cerr, "No argument specified for: ", arg);
              } 
            pulseHalfWidthCells = nCells[0]/8.0;
          }
        else if (arg == "-patches")
          {
            // This can be followed by either 1 int or D ints.
            
            bool hasarg = intArgument(argc, argv, i+1, nPatches[0]);
            
            if (hasarg) 
              { 
                ++i;
                if (D > 1)
                  {
                    bool moreArgs = intArgument(argc, argv, i+1, nPatches[1]);
                    if (moreArgs)
                      {
                        for (int d = 1; d < D; ++d)
                          {
                            hasarg = intArgument(argc, argv, i+1, nPatches[d]);
                            if (!hasarg) 
                              badOption(std::cerr, 
                                        "Not enough arguments for: ", arg);
                            ++i;
                          }
                      }
                    else
                      {
                        for (int d = 1; d < D; ++d)
                          {
                            nPatches[d] = nPatches[0];
                          }
                      }
                  }
              }
            else
              {
                badOption(std::cerr, "No argument specified for: ", arg);
              }            
          }
        else if (arg == "-epsilon")
          {
            if (i+1 == argc) badOption(std::cerr, 
                                       "No value specified for: ", arg);
            
	    hasarg = doubleArgument(argc, argv, i+1, epsilon);
	    
	    if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
	    
	    ++i;
          }
        else if (arg == "-pulseHalfWidthCells")
          {
            if (i+1 == argc) badOption(std::cerr, 
                                       "No value specified for: ", arg);
            
	    hasarg = intArgument(argc, argv, i+1, pulseHalfWidthCells);
	    
	    if (!hasarg) badOption(std::cerr, "No value specified for: ", arg);
	    
	    ++i;
          }
        else
          {
            std::cerr << "No such flag: " << arg << endl;
            usage(std::cerr);
            exit(0);
          }
                    
        ++i; // next arg
      }
      
    // Do some sanity checks:
    
    if (lastTimeStep < 1)   
      badValue(std::cerr, "-steps");
    if (outputIncrement < 1 || outputIncrement > lastTimeStep) 
      badValue(std::cerr, "-out");
    if (purgeIncrement < 1 || purgeIncrement > lastTimeStep) 
      badValue(std::cerr, "-purge");

    // Finally, initialize nVerts.
    
    for (int d = 0; d < D; ++d) nVerts[d] = nCells[d] + 1;
                
}

template <int D>
bool SAOptions<D>::
intArgument(int argc, char **argv, int pos, int &val)
{
    // Make sure there is an argument available

    if (pos >= argc)
      return false;

    // Make sure the 'pos' argument is a number.  If it starts with a number
    // or -number, it is OK.

    char firstchar = argv[pos][0];
    if (firstchar < '0' || firstchar > '9')
      {
        // first char is not a number.  Is the second, with the first a '-/+'?

        if ((firstchar != '-' && firstchar != '+') || argv[pos][1] == 0 ||
	    (argv[pos][1] < '0' || argv[pos][1] > '9'))
	  return false;
      }

    // Get the value and return it in the last argument

    val = atoi(argv[pos]);
    return true;
}

template <int D>
bool SAOptions<D>::
doubleArgument(int argc, char **argv, int pos, double &val)
{
    // Make sure there is an argument available

    if (pos >= argc)
      return false;

    // Make sure the 'pos' argument is a number.  If it starts with a number
    // or -number, it is OK.

    char firstchar = argv[pos][0];
    if (firstchar < '0' || firstchar > '9')
      {
        // first char is not a number.  Is the second, with the first a '-/+'?

        if ((firstchar != '-' && firstchar != '+') || argv[pos][1] == 0 ||
	    (argv[pos][1] < '0' || argv[pos][1] > '9'))
	  return false;
      }

    // Get the value and return it in the last argument

    val = atof(argv[pos]);
    return true;
}

//
// Helper functions: print, usage, badOption, badValue
// 
// To avoid having to put these functions in the class body, I've written 
// them as global template functions. The corresponding member functions
// simply call these.
//

template <int D, class OStr>
void print(const SAOptions<D> &opts, OStr &pout)
{
    using std::endl;
    
    int d;
    
    pout << "Program name: " << opts.progname << endl;
    pout << "Option values: " << endl;
    pout << "=====================================================" << endl;
    
    pout << "text                = " 
         << (opts.doTextOut ? "true ; " : "false ; ") << endl;
    pout << "sum                 = " 
         << (opts.doSumOut ? "true ; " : "false ; ") << endl;
    pout << "ensight             = " 
         << (opts.doEnsightOut ? "true ; " : "false ; ") << endl;
    pout << "purge               = " 
         << (opts.purge ? "true ; " : "false ; ") << endl;

    pout << "steps               = " << opts.lastTimeStep << endl;
    pout << "outSteps            = " << opts.outputIncrement << endl;
    pout << "purgeSteps          = " << opts.purgeIncrement << endl;
    
    pout << "nCells              = " << opts.nCells[0];
    for (d = 1; d < D; ++d) 
      pout << ", " << opts.nCells[d];
    pout << endl;
    
    pout << "nVerts              = " << opts.nVerts[0];
    for (d = 1; d < D; ++d) 
      pout << ", " << opts.nVerts[d];
    pout << endl;

    pout << "nPatches            = " << opts.nPatches[0];
    for (d = 1; d < D; ++d) 
      pout << ", " << opts.nPatches[d];
    pout << endl;
    
    pout << "epsilon             = " << opts.epsilon << endl;
    pout << "pulseHalfWidthCells = " << opts.pulseHalfWidthCells << endl;
    pout << "=====================================================" << endl 
         << endl;
}
  
template <class OStr>
void usage(const std::string &name, OStr &out)
{
    out << "Usage: " << name << std::endl
        << " [-cells <nCellsX> [<nCellsY> <nCellsZ>]]" 
        << std::endl
        << " [-patches <nPatchesX> [<nPatchesY> <nPatchesZ>]]" 
        << std::endl
        << " [-steps <lastTimeStep>]"
        << " [-out <outputIncrement>]"
        << " [-text]" 
        << " [-ensight]"
        << std::endl
        << " [-purge [<purgeIncrement>]]"
        << " [-sum]" 
        << " [-block]" 
        << std::endl
        << " [-epsilon <epsilon>]" 
        << " [-pulseHalfWidthCells <pulseHalfWidthCells>]" 
        << std::endl;
}

template <class OStr>
void badOption(OStr &out, const char *str, const std::string &option) 
{ 
  out << "Bad option: " << str << option << std::endl;
  exit(1);
}

template <class OStr>
void badValue(OStr &out, const std::string &option) 
{ 
  out << "Bad input value for option: " << option << std::endl;
  exit(1);
}

// These functions produce binary output of Field data suitable for feeding
// into the Ensight plotting package:

void ensightCaseOut(char *fn, int increment, int lastTimeStep, double dt) 
{
  Inform ec(NULL, fn, Inform::out);
  ec << "FORMAT" << std::endl;
  ec << "type: ensight gold" << std::endl;
  ec << "GEOMETRY" << std::endl;
  ec << "model:    ScalarAdvection.geo" << std::endl;
  ec << "VARIABLE" << std::endl;
  // Note: EnSight won't do isosurfaces on cell-centered data, so specify
  // node-centered data, and use the POOMA average() function to write out the
  // cell-centerd scalar field data as node (vertex) centered data.
  ec << "scalar per node: u u.****" << std::endl;
  ec << "TIME" << std::endl;
  ec << "time set:              1" << std::endl;
  ec << "number of steps:       " << lastTimeStep + 1 << std::endl;
  ec << "filename start number: 0" << std::endl;
  ec << "filename increment:    " << increment << std::endl;
  ec << "time values: " << std::endl;
  for (int i = 0; i <= lastTimeStep; i++) {
    ec << i*dt << " ";
  }
  ec << std::endl;
}

// DiscreteGeometry file:
template<int Dim>
void ensightGeometryOut(char *fn, UniformRectilinearMesh<Dim> &mesh) 
{
  char header[80];
  char descript1[80];
  char descript2[80];
 // char nodeid[80];
 // char elemid[80];
  char coords[80];
  char part[80];
  char descript3[80];
  FILE *fp;
  int i;
  fp = fopen(fn, "w");
  strcpy(header,"C Binary");
  fwrite(header,sizeof(char),80,fp);
  strcpy(descript1,"POOMA Field Scalar Advection Example");
  fwrite(descript1,sizeof(char),80,fp);
  strcpy(descript2,"Ensight DiscreteGeometry file");
  fwrite(descript2,sizeof(char),80,fp);
  strcpy(part,"part");
  fwrite(part,sizeof(char),80,fp);
  i = 1;
  fwrite(&i,sizeof(int),1,fp);
  strcpy(descript3,"u(x[,y[,z]])");
  fwrite(descript3,sizeof(char),80,fp);
  strcpy(coords, "block uniform");
  fwrite(coords,sizeof(char),80,fp);
  

  // Number of vertices along each dimension:
  // Ensight always assumes 3D; write 1 for number of vertices in all
  // directions greater than the actual dimensionality:
  int id[3];
  int d;
  for (d = 0; d < Dim; d++) {
    id[d] = mesh.physicalDomain()[d].size();
  }
  for (d = Dim; d < 3; d++) { 
    id[d] = 1;
  }
  fwrite(&id, sizeof(int), 3, fp);

  // Origin:
  float x0d[3];
  for (d = 0; d < Dim; d++) {
    x0d[d] = mesh.origin()(d);
  }
  for (d = Dim; d < 3; d++) { 
    x0d[d] = 0.0;
  }
  fwrite(&x0d, sizeof(float), 3, fp);

  // Mesh spacings:
  float dxd[3];
  for (d = 0; d < Dim; d++) {
    dxd[d] = mesh.meshSpacing()(d);
  }
  for (d = Dim; d < 3; d++) { 
    dxd[d] = 0.0;
  }
  fwrite(&dxd, sizeof(float), 3, fp);
  fclose(fp);
}

// Variable file:
// Partial specializations for 1D, 2D, 3D Arrays:
// 1D:
template<class T, class EngineTag>
void ensightVariableOut(char *fn, const Array<1, T, EngineTag> &a) 
{
  char      descript[80];
  FILE     *fp;
  int       ii;

  fp = fopen(fn, "w");
  sprintf(descript,"Scalar Variable u at %s",fn);
  fwrite(descript,sizeof(char),80,fp);
  strcpy(descript,"part");
  fwrite(descript,sizeof(char),80,fp);
  ii = 1;
  fwrite(&ii,sizeof(int),1,fp);
  strcpy(descript,"block uniform");
  fwrite(descript,sizeof(char),80,fp);

  float *var;
  int nnodes = a.domain().size();
  var = new float[nnodes];
  for (int i = a.domain()[0].first(); i <= a.domain()[0].last(); i++) {
    var[i] = a(i);
  }
  
  fwrite (var, sizeof(float), nnodes, fp);
  delete [] var;
  fclose(fp);
}

// 2D:
template<class T, class EngineTag>
void ensightVariableOut(char *fn, const Array<2, T, EngineTag> &a) 
{
  char      descript[80];
  FILE     *fp;
  int       ii;

  int id[3];
  int nelems = 1;
  int d;
  for (d = 0; d < 2; d++) {
    id[d] = a.domain()[d].size();
    nelems*= id[d];
  }
  for (d = 2; d < 3; d++) { 
    id[d] = 1;
  }

  fp = fopen(fn, "w");
  sprintf(descript,"Scalar Variable u at %s",fn);
  fwrite(descript,sizeof(char),80,fp);
  strcpy(descript,"part");
  fwrite(descript,sizeof(char),80,fp);
  ii = 1;
  fwrite(&ii,sizeof(int),1,fp);
  strcpy(descript,"block uniform");
  fwrite(descript,sizeof(char),80,fp);

  float *var;
  var = new float[nelems];
  int count = 0;
  for (int j = a.domain()[1].first(); j <= a.domain()[1].last(); j++) {
    for (int i = a.domain()[0].first(); i <= a.domain()[0].last(); i++) {
      var[count++] = a(i,j);
    }
  }
  
  fwrite (var, sizeof(float), nelems, fp);
  delete [] var;
  fclose(fp);
}

// 3D:
template<class T, class EngineTag>
void ensightVariableOut(char *fn, const Array<3, T, EngineTag> &a) 
{
  char      descript[80];
  FILE     *fp;
  int       ii;

  int id[3];
  int nelems = 1;
  for (int d = 0; d < 3; d++) {
    id[d] = a.domain()[d].size();
    nelems*= id[d];
  }

  fp = fopen(fn, "w");
  sprintf(descript,"Scalar Variable u at %s",fn);
  fwrite(descript,sizeof(char),80,fp);
  strcpy(descript,"part");
  fwrite(descript,sizeof(char),80,fp);
  ii = 1;
  fwrite(&ii,sizeof(int),1,fp);
  strcpy(descript,"block uniform");
  fwrite(descript,sizeof(char),80,fp);

  float *var;
  var = new float[nelems];
  int count = 0;
  for (int k = a.domain()[2].first(); k <= a.domain()[2].last(); k++) {
    for (int j = a.domain()[1].first(); j <= a.domain()[1].last(); j++) {
      for (int i = a.domain()[0].first(); i <= a.domain()[0].last(); i++) {
        var[count++] = a(i,j,k);
      }
    }
  }
  
  fwrite (var, sizeof(float), nelems, fp);
  delete [] var;
  fclose(fp);
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: ScalarAdvection.cpp,v $   $Author: swhaney $
// $Revision: 1.23 $   $Date: 2000/07/20 15:39:27 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
