// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

// -----------------------------------------------------------------------------
// Field test 6: Componentwise boundary conditions with a multi-component field.
// -----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

#include <iostream>
using namespace std;

//-----------------------------------------------------------------------------
// Main program:
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

  // Create the mesh

  const int D = 2; // dimensionality
  const int N = 6; // number of verts in each direction
  int d;
  
  Interval<D> allVerts;
  for (d = 0; d < D; d++) 
    allVerts[d] = Interval<1>(N);
  
  Vector<D,double> origin;
  Vector<D,double> spacings;
  for (d = 0; d < D; d++) 
    {
      origin(d) = d;
      spacings(d) = d + 1;
    }
  UniformRectilinearMesh<2> mesh(allVerts, origin, spacings);
  
  // Create the geometry.
  
  typedef RectilinearCentering<D,VectorFaceRCTag<D> > Centering_t;
  DiscreteGeometry<Centering_t , UniformRectilinearMesh<D> > 
    geom(mesh, GuardLayers<D>(1));
  
  // Make the field.
  
  Field<DiscreteGeometry<Centering_t, UniformRectilinearMesh<D> >, Vector<D,double> > 
    f(geom);

  // Add componentwise boundary conditions.
  
  typedef ComponentBC<1,NegReflectFaceBC> NegReflectFace_t;
  typedef ComponentBC<1,PosReflectFaceBC> PosReflectFace_t;
  for (int face = 0; face < 2 * D; face++)
    {
      d = face / 2;
      for (int c = 0; c < D; c++)
        {
          if (c == d)
            f.addBoundaryCondition(NegReflectFace_t(c, face));
          else
            f.addBoundaryCondition(PosReflectFace_t(c, face));
        }
    }

  // Initialize some values.
  
  for (int i = 0; i <= f.physicalDomain()[0].last(); i++) {
    for (int j = 0; j <= f.physicalDomain()[1].last(); j++) {
      f(i, j) = Vector<D,double>(i + j);
    }
  }
      
  // Apply componentwise boundary conditions.
  
  f.applyBoundaryConditions();
  
  // Block since we're going to be looking at some values.
  
  Pooma::blockAndEvaluate();

  // The rule for figuring these out is:
  //   o if an x-face, x-component is -vert, y-component is cell
  //   o if a y-face, x-component is cell, y-component is -vert
  
  tester.check("f(-1,-1)", f(-1,-1), Vector<D>(-1, -1));
  tester.check("f(2,6)", f(2,6), Vector<D>(7,-6));
  tester.check("f(-1,3)", f(-1,3), Vector<D>(-4, 3));
  tester.check("f(4,6)", f(4,6), Vector<D>(9,-8));
  tester.check("f(5,-1)", f(5,-1), Vector<D>(5,-6));

  tester.out() << "f:" << std::endl;
  tester.out() << f << std::endl;
  tester.out() << "f.x():" << std::endl;
  tester.out() << f.x() << std::endl;
  
  int ret = tester.results("field_test6");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: field_test6.cpp,v $   $Author: swhaney $
// $Revision: 1.3 $   $Date: 2000/03/07 13:17:16 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
