// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Field test 8: Testing Grad.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

#include <iostream>

//-----------------------------------------------------------------------------
// Globals
//-----------------------------------------------------------------------------

const int NX = 5, NY = 5;
Interval<1> I(NX), J(NY);
Interval<2> td(Interval<1>(-1,NX-1), Interval<1>(-1,NY-1)), pd(NX-1, NY-1);
  

//-----------------------------------------------------------------------------
// Test function
//-----------------------------------------------------------------------------

template<class Geom, class T, class T2, class EngineTag>
void doTest(Pooma::Tester &tester, Field<Geom, T, EngineTag> &f,
  Field<Geom, T2, EngineTag> &g)
{
  for (int i = td[0].first(); i <= td[0].last(); i++)
    for (int j = td[1].first(); j <= td[1].last(); j++)
      f(i, j) = 2.0 * i + 3.0 * j;
      
  Pooma::blockAndEvaluate();
  
  g = -grad<Cell>(f);
  
  tester.check(all(g == Vector<2>(-2.0, -1.5)));
  
  double s = sum(dot(grad<Cell>(f), grad<Cell>(f)));
  
  tester.out() << s << std::endl;
}


//-----------------------------------------------------------------------------
// Main program
//-----------------------------------------------------------------------------

int main(int argc, char *argv[])
{
  Pooma::initialize(argc,argv);
  Pooma::Tester tester(argc, argv);

  // Create the mesh.
  
  Vector<2> origin;
  Vector<2> spacings;
  for (int d = 0; d < 2; d++) 
    {
      origin(d) = d;
      spacings(d) = d + 1;
    }
  RectilinearMesh<2> mesh(I, J, origin, spacings);
  
  // Create the geometry.
  
  DiscreteGeometry<Cell, RectilinearMesh<2> > geom;
  geom.initialize(mesh, GuardLayers<2>(1));
  
  // Make a Brick-Engine-based field.
  
  Field<DiscreteGeometry<Cell, RectilinearMesh<2> > > f(geom);
  Field<DiscreteGeometry<Cell, RectilinearMesh<2> >, Vector<2> > g(geom);

  doTest(tester, f, g);
  
  // Make a CompressibleBrick-Engine-based field.
  
  Field<DiscreteGeometry<Cell, RectilinearMesh<2> >, double, 
        CompressibleBrick> fc(geom);
  Field<DiscreteGeometry<Cell, RectilinearMesh<2> >, Vector<2>, 
        CompressibleBrick> gc(geom);

  doTest(tester, fc, gc);
  
  // Make a Multipatch-Engine-based field.

  Loc<2> blocks(2,2);
  UniformGridLayout<2> layout(pd, blocks, GuardLayers<2>(1), 
                              GuardLayers<2>(1), ReplicatedTag());
  typedef MultiPatch<UniformTag, Brick> MP_t;
  Field<DiscreteGeometry<Cell, RectilinearMesh<2> >, double, MP_t> 
    fm(geom, layout);
  Field<DiscreteGeometry<Cell, RectilinearMesh<2> >, Vector<2>, MP_t> 
    gm(geom, layout);

  doTest(tester, fm, gm);
  
  // Make a Nonuniform Multipatch-Engine-based field.

  GridLayout<2> layout2(pd, blocks, GuardLayers<2>(1), GuardLayers<2>(1), 
                        ReplicatedTag());
  typedef MultiPatch<GridTag, Brick> MP2_t;
  Field<DiscreteGeometry<Cell, RectilinearMesh<2> >, double, MP2_t> 
    fg(geom, layout2);
  Field<DiscreteGeometry<Cell, RectilinearMesh<2> >, Vector<2>, MP2_t> 
    gg(geom, layout2);

  doTest(tester, fg, gg);
    
  int ret = tester.results("field_test8");
  Pooma::finalize();
  return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: field_test8.cpp,v $   $Author: sa_smith $
// $Revision: 1.4 $   $Date: 2000/06/28 23:35:42 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
