// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// where2.cpp - simple test of 2 argument where on fields.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Pooma/Fields.h"
#include "Utilities/Tester.h"

#include <iostream>
#include <string>
#include <stdio.h>

int main(int argc, char *argv[])
{
   Pooma::initialize(argc,argv);        // Initialize the library
   Pooma::Tester tester(argc, argv);

   PrintArray pa(3, 10, 3, 6, true, 1); // For formatted output

   // Create the physical domains:
   Interval<1> vertexDomain(33), cellDomain(32);

   // Create the (uniform, logically rectilinear) mesh.
   typedef UniformRectilinearMesh<1> Mesh_t;
   Mesh_t mesh(vertexDomain);

   // Create geometry objects, one with guards (problem in 2-arg where(), and
   // one with no guards (no problem in 2-arg-where):
   typedef DiscreteGeometry<Cell, UniformRectilinearMesh<1> > Geometry_t;
   Geometry_t geom(mesh, GuardLayers<1>(1));
   Geometry_t geomNG(mesh);

   // Create the layout, one with guards (problem in 2-arg where(), and
   // one with no guards (no problem in 2-arg-where):
   Loc<1> patches(8);
   GridLayout<1> layout(cellDomain, patches,
                        GuardLayers<1>(1), GuardLayers<1>(1), ReplicatedTag());
   GridLayout<1> layoutNG(cellDomain, patches, ReplicatedTag());
   typedef MultiPatch<GridTag, CompressibleBrick> MP_t;

   // Create the Fields:
   Field<Geometry_t, double, MP_t> u2(geom, layout), u3(geom, layout);
   Field<Geometry_t, double, MP_t> u2NG(geomNG, layoutNG);

   // Initialize Fields to zero everywhere, even global guard layers 
   // (if exist):
   u2.all() = 0.0;
   u3.all() = 0.0;
   u2NG = 0.0;

   // Initialize decaying-pulse values in Fields:
   double pulseHalfWidth = 1.0;
   Loc<1> pulseCenter(8);
   Pooma::blockAndEvaluate(); // Needed pre-scalar-indexing read
   Vector<1> u0 = u3.x(pulseCenter);
   u3 = 1.0 * exp(-dot(u3.x() - u0, u3.x() - u0) / (2.0 * pulseHalfWidth));
   u2.all() = u3.all();
   u2NG = u3;
   Pooma::blockAndEvaluate();

   // Now test the two where() functions, by purging values below epsilon:
   double epsilon = 1.0e-4;

   tester.out() << "BEFORE PURGE, u3 = " << std::endl;
   pa.print(tester.out(), u3);
   tester.out() << std::endl << std::endl;
   u3 = where((fabs(u3) < epsilon), 0.0, u3);
   Pooma::blockAndEvaluate();
   tester.out() << "AFTER PURGE, compressedFraction(u3) = " 
                << compressedFraction(u3) << " ; sum(u3) = " << sum(u3) 
                << " ; u3 = " << std::endl;
   pa.print(tester.out(), u3);
   tester.out() << std::endl << std::endl;

   tester.out() << "BEFORE PURGE, u2 = " << std::endl;
   pa.print(tester.out(), u2);
   tester.out() << std::endl << std::endl;
   u2 = where((fabs(u2) < epsilon), 0.0);
   Pooma::blockAndEvaluate();
   tester.out() << "AFTER PURGE, compressedFraction(u2) = " 
                << compressedFraction(u2) << " ; sum(u2) = " << sum(u2) 
                << " ; u2 = " << std::endl;
   pa.print(tester.out(), u2);
   tester.out() << std::endl << std::endl;

   tester.out() << "BEFORE PURGE, u2NG = " << std::endl;
   pa.print(tester.out(), u2NG);
   tester.out() << std::endl << std::endl;
   u2NG = where((fabs(u2NG) < epsilon), 0.0);
   Pooma::blockAndEvaluate();
   tester.out() << "AFTER PURGE, compressedFraction(u2NG) = "
        << compressedFraction(u2NG)
        << " ; sum(u2NG) = " << sum(u2NG) << " ; u2NG = " << std::endl;
   pa.print(tester.out(), u2NG);
   tester.out() << std::endl << std::endl;

   tester.check("u2==u3",
		sum((u2 - u3) * (u2 - u3)) < epsilon * epsilon);
   tester.check("u2NG==u3",
		sum((u2NG - u3) * (u2NG - u3)) < epsilon * epsilon);

   int ret = tester.results("where2");
   Pooma::finalize();
   return ret;
}

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: where2.cpp,v $   $Author: julianc $
// $Revision: 1.3 $   $Date: 2000/06/07 16:25:48 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
