// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// IO functions to support Pooma objects related to Layout
//-----------------------------------------------------------------------------

#ifndef POOMA_IO_LAYOUT_IO_H
#define POOMA_IO_LAYOUT_IO_H

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <iostream>
#include "Layout/Node.h"
#include "Layout/UniformGridLayout.h"
#include "Layout/GridLayout.h"
#include "Layout/SparseTileLayout.h"
#include "Layout/DynamicLayout.h"
#include "Layout/DomainLayout.h"
#include "IO/CTTI.h"
#include "IO/Serializers.h"
#include "IO/DomainIO.h"
#include <string>

#include <stdio.h>

//-----------------------------------------------------------
// I/O Functions for Node<Dom,OrigDom>
//-----------------------------------------------------------
// CTTI function for Node<Dom,OrigDom>
//-----------------------------------------------------------

template <class Dom, class OrigDom>
std::string PoomaCTTI(const Node<Dom,OrigDom>&);
template <class Dom, class OrigDom>
std::string PoomaCTTI(const Node<Dom,OrigDom>&){
  std::string strng= "Node<"+ PoomaCTTI(Dom())+ ","
    + PoomaCTTI(OrigDom())+ " >";
  return strng;
}

//-----------------------------------------------------------
// Serializers for Node<Dom,OrigDom>
//-----------------------------------------------------------

template <class Stream, class Dom, class OrigDom>
int serialize(Stream& s, const Node<Dom,OrigDom>& node);
template <class Stream, class Dom, class OrigDom>
int serialize(Stream& s, const Node<Dom,OrigDom>& node){
  int nBytes=0;
  // Acquire the owned and allocated domains.
  const typename Node<Dom,OrigDom>::Domain_t& dom= node.domain();
  const typename Node<Dom,OrigDom>::AllocatedDomain_t&
    alloc= node.allocated();
  // Assuming these are supported by the I/O system, we serialize
  // them.
  nBytes+= serialize(s,dom);
  nBytes+= serialize(s,alloc);
  // Acquire the other attributes.
  typename Node<Dom,OrigDom>::ID_t local= node.localID();
  typename Node<Dom,OrigDom>::ID_t global= node.globalID();
  typename Node<Dom,OrigDom>::Context_t context= node.context();
  int affinity= node.affinity();
  // Serialize the remaining attributes.
  nBytes+= serialize(s,local);
  nBytes+= serialize(s,global);
  nBytes+= serialize(s,context);
  nBytes+= serialize(s,affinity);
  return nBytes;
}

template <class Stream, class Dom, class OrigDom>
int deserialize(Node<Dom,OrigDom>& node, Stream& s);
template <class Stream, class Dom, class OrigDom>
int deserialize(Node<Dom,OrigDom>& node, Stream& s){
  int nBytes=0;
  // Construct the owned and allocated domains.
  // Provide appropriate targets.
  typedef typename Node<Dom,OrigDom>::Domain_t Domain_t;
  typedef typename Node<Dom,OrigDom>::AllocatedDomain_t AllocDomain_t;
  Domain_t dom;
  AllocDomain_t alloc;
  // Assuming their types are supported by the I/O system,
  // we deserialize them.
  nBytes+= deserialize(dom,s);
  nBytes+= deserialize(alloc,s);
  // Read the other attributes.
  typename Node<Dom,OrigDom>::ID_t local;
  typename Node<Dom,OrigDom>::ID_t global;
  typename Node<Dom,OrigDom>::Context_t context;
  int affinity;
  // Deserialize the remaining attributes.
  nBytes+= deserialize(local,s);
  nBytes+= deserialize(global,s);
  nBytes+= deserialize(context,s);
  nBytes+= deserialize(affinity,s);
  // Initialize the node using the recovered data.
  node.initialize(dom, alloc, context, global, local);
  node.affinity()= affinity;
  return nBytes;
}

template <class Dom, class OrigDom>
int serialSizeof(const Node<Dom,OrigDom>& node);
template <class Dom, class OrigDom>
int serialSizeof(const Node<Dom,OrigDom>& node){
  typedef Node<Dom,OrigDom> This_t;
  typedef typename This_t::Domain_t Domain_t;
  typedef typename This_t::AllocatedDomain_t AllocDomain_t;
  int nBytes=0;
  const Domain_t& dom= node.domain();
  const AllocDomain_t& alloc= node.allocated();
  nBytes+= serialSizeof(dom);
  nBytes+= serialSizeof(alloc);
  nBytes+= 2*serialSizeof(typename This_t::ID_t());
  nBytes+= serialSizeof(typename This_t::Context_t());
  nBytes+= sizeof(int);
  return nBytes;
}


//-----------------------------------------------------------
// I/O Functions for DomainLayout<Dim>
//-----------------------------------------------------------
// CTTI function for DomainLayout<Dim>
//-----------------------------------------------------------

template <int Dim>
std::string PoomaCTTI(const DomainLayout<Dim>&);
template <int Dim>
std::string PoomaCTTI(const DomainLayout<Dim>&){
  char numbuf[10];
  sprintf(numbuf,"%d\0",Dim);
  std::string strng= "DomainLayout<"+
    std::string(numbuf)+">";
  return strng;
}

//-----------------------------------------------------------
// Serializers for DomainLayout<Dim>
//-----------------------------------------------------------

template <class Stream, int Dim>
int serialize(Stream& s, const DomainLayout<Dim>& layout);
template <class Stream, int Dim>
int serialize(Stream& s, const DomainLayout<Dim>& layout){
  int nBytes=0;

  // Acquire all the necessary state information.
  // Acquire the domain.
  typename DomainLayout<Dim>::Domain_t& dom= layout.domain();

  // Serialize the information.
  nBytes+= serialize(s,dom);
  return nBytes;
}

template <class Stream, int Dim>
int deserialize(DomainLayout<Dim>& layout, Stream& s);
template <class Stream, int Dim>
int deserialize(DomainLayout<Dim>& layout, Stream& s){
  int nBytes=0;

  // Acquire all the necessary state information.
  // Acquire the the domain.
  typename DomainLayout<Dim>::Domain_t dom;

  // Read the domain.
  nBytes+= deserialize(dom,s);

  // Instantiate the layout.
  layout.initialize(dom);

  return nBytes;
}

template <int Dim>
int serialSizeof(const DomainLayout<Dim>& layout);
template <int Dim>
int serialSizeof(const DomainLayout<Dim>& layout){
  int nBytes=0;

  // Acquire the domain.
  const typename DomainLayout<Dim>::Domain_t& dom= layout.domain();

  // Size the domain.
  nBytes+= serialSizeof(dom);
  return nBytes;
}

//-----------------------------------------------------------
// I/O Functions for UniformGridLayout<Dim>
//-----------------------------------------------------------
// CTTI function for UniformGridLayout<Dim>
//-----------------------------------------------------------

template <int Dim>
std::string PoomaCTTI(const UniformGridLayout<Dim>&);
template <int Dim>
std::string PoomaCTTI(const UniformGridLayout<Dim>&){
  char numbuf[10];
  sprintf(numbuf,"%d\0",Dim);
  std::string strng= "UniformGridLayout<"+
    std::string(numbuf)+">";
  return strng;
}

//-----------------------------------------------------------
// Serializers for UniformGridLayout<Dim>
//-----------------------------------------------------------

template <class Stream, int Dim>
int serialize(Stream& s, const UniformGridLayout<Dim>& layout);
template <class Stream, int Dim>
int serialize(Stream& s, const UniformGridLayout<Dim>& layout){
  int nBytes=0;

  // Acquire all the necessary state information.
  // Acquire the complete domain.
  typename UniformGridLayout<Dim>::Domain_t dom= layout.domain();
  // Check for guard layers and acquire.
  int ig=0, eg=0;
  typename UniformGridLayout<Dim>::GuardLayers_t guards0, iguards,
    eguards;
  iguards= layout.internalGuards();
  if(iguards!=guards0){
    ig=1;
  }
  eguards= layout.externalGuards();
  int first, last;
  if(eguards!=guards0){
    eg=1;
    // Subtract the external guards
    // to obtain the ab initio domain
    for(int i=0;i<Dim;i++){
      first= dom[i].first()+ eguards.lower(i);
      last= dom[i].last()- eguards.upper(i);
      dom[i]= Interval<1>(first,last);
    }
  }
  // Acquire the Loc object that describes the block structure.
  Loc<Dim> blocks= layout.blocks();
  // Acquire the list of nodes.
  typedef typename UniformGridLayout<Dim>::List_t List_t;
  List_t nodeList= layout.nodeListGlobal();
  int listSize= nodeList.size();

  // Serialize the information.
  // Serialize the domain.
  nBytes+= serialize(s,dom);
  //std::cout<<"domain serialized"<<std::endl;
  // Serialize the blocks description.
  nBytes+= serialize(s,blocks);
  //std::cout<<"blocks serialized"<<std::endl;
  // Serialize the guard layers if they exist.
  nBytes+= serialize(s,ig);
  //std::cout<<"ig serialized"<<std::endl;
  if(ig==1){
    for(int i=0;i<Dim;i++){
      nBytes+= serialize(s,iguards.lower(i));
      nBytes+= serialize(s,iguards.upper(i));
    }
  }
  nBytes+= serialize(s,eg);
  //std::cout<<"eg serialized"<<std::endl;
  if(eg==1){
    for(int i=0;i<Dim;i++){
      nBytes+= serialize(s,eguards.lower(i));
      nBytes+= serialize(s,eguards.upper(i));
    }
  }
  // Serialize the node list.
  nBytes+= serialize(s,listSize);
  //std::cout<<"listSize serialized"<<std::endl;
  typename List_t::iterator listIter= nodeList.begin();
  typedef typename UniformGridLayout<Dim>::Value_t Node_t;
  Node_t* node;
  for(int i=0;i<listSize;i++){
    node= *listIter;
    nBytes+= serialize(s,*node);
    listIter++;
  }

  return nBytes;
}

template <class Stream, int Dim>
int deserialize(UniformGridLayout<Dim>& layout, Stream& s);
template <class Stream, int Dim>
int deserialize(UniformGridLayout<Dim>& layout, Stream& s){
  int nBytes=0;

  // Read all the necessary state information.
  // Read the domain.
  typename UniformGridLayout<Dim>::Domain_t dom;
  nBytes+= deserialize(dom,s);
  //std::cout<<"domain deserialized"<<std::endl;
  // Read the blocks description.
  Loc<Dim> blocks;
  nBytes+= deserialize(blocks,s);
  //std::cout<<"blocks deserialized"<<std::endl;
  // Check for guard layers and read if they exist.
  int ig=0, eg=0;
  typename UniformGridLayout<Dim>::GuardLayers_t iguards, eguards;
  bool hasIG= false, hasEG= false;
  nBytes+= deserialize(ig,s);
  //std::cout<<"deserialized ig="<<ig<<std::endl;
  if(ig==1){
    hasIG= true;
    for(int i=0;i<Dim;i++){
      nBytes+= deserialize(iguards.lower(i),s);
      nBytes+= deserialize(iguards.upper(i),s);
    }
  }
  nBytes+= deserialize(eg,s);
  //std::cout<<"deserialized eg="<<eg<<std::endl;
  if(eg==1){
    hasEG= true;
    for(int i=0;i<Dim;i++){
      nBytes+= deserialize(eguards.lower(i),s);
      nBytes+= deserialize(eguards.upper(i),s);
    }
  }
  // Read the node list.
  typedef typename UniformGridLayout<Dim>::List_t List_t;
  List_t nodeList;
  int listSize;
  nBytes+= deserialize(listSize,s);
  //std::cout<<"deserialized listSize="<<listSize<<std::endl;
  typedef typename UniformGridLayout<Dim>::Value_t Node_t;
  Node_t* node;
  for(int i=0;i<listSize;i++){
    node= new Node_t;
    nBytes+= deserialize(*node,s);
    nodeList.push_back(node);
  }

  // Use the special I/O initializer for UniformGridLayout
  // to instantiate.
  layout.initialize(dom, nodeList, blocks, hasIG, hasEG,
		    iguards, eguards);

  // std::cout<<"layout initialized"<<std::endl;
  return nBytes;
}

template <int Dim>
int serialSizeof(const UniformGridLayout<Dim>& layout);
template <int Dim>
int serialSizeof(const UniformGridLayout<Dim>& layout){
  int nBytes=0;

  // Acquire necessary state objects.
  // Acquire the complete domain.
  const typename UniformGridLayout<Dim>::Domain_t& dom= layout.domain();
  // Acquire guard layers.
  typename UniformGridLayout<Dim>::GuardLayers_t guards0, iguards,
    eguards;
  iguards= layout.internalGuards();
  eguards= layout.externalGuards();
  // Acquire the Loc object that describes the block structure.
  Loc<Dim> blocks= layout.blocks();
  // Acquire the list of nodes.
  typedef typename UniformGridLayout<Dim>::List_t List_t;
  List_t nodeList= layout.nodeListGlobal();
  int listSize= nodeList.size();

  // Calculate the sizes.
  // Size the domain.
  nBytes+= serialSizeof(dom);
  // Size the blocks description.
  nBytes+= serialSizeof(blocks);
  // Size the guard layers info.
  nBytes+= sizeof(int);
  if(iguards!=guards0){
    nBytes+= 2*Dim*sizeof(int);
  }
  nBytes+= sizeof(int);
  if(eguards!=guards0){
    nBytes+= 2*Dim*sizeof(int);
  }
  // Size the node list.
  nBytes+= sizeof(int);
  typename List_t::iterator listIter= nodeList.begin();
  typename UniformGridLayout<Dim>::Value_t* node;
  for(int i=0;i<listSize;i++){
    node= *listIter;
    nBytes+= serialSizeof(*node);
  }

  return nBytes;
}

//-----------------------------------------------------------
// I/O Functions for GridLayout<Dim>
//-----------------------------------------------------------
// CTTI function for GridLayout<Dim>
//-----------------------------------------------------------

template <int Dim>
std::string PoomaCTTI(const GridLayout<Dim>&);
template <int Dim>
std::string PoomaCTTI(const GridLayout<Dim>&){
  char numbuf[10];
  sprintf(numbuf,"%d\0",Dim);
  std::string strng= "GridLayout<"+
    std::string(numbuf)+ ">";
  return strng;
}

//-----------------------------------------------------------
// Serializers for GridLayout<Dim>
//-----------------------------------------------------------

template <class Stream, int Dim>
int serialize(Stream& s, const GridLayout<Dim>& layout);
template <class Stream, int Dim>
int serialize(Stream& s, const GridLayout<Dim>& layout){
  int nBytes=0;

  // Acquire all the necessary state information.
  // Acquire the complete domain.
  typename GridLayout<Dim>::Domain_t dom= layout.domain();
  // Check for guard layers and acquire.
  int ig=0, eg=0;
  typename GridLayout<Dim>::GuardLayers_t guards0, iguards,
    eguards;
  iguards= layout.internalGuards();
  if(iguards!=guards0){
    ig=1;
  }
  eguards= layout.externalGuards();
  int first, last;
  if(eguards!=guards0){
    eg=1;
    // Subtract the external guards
    // to obtain the ab initio domain
    for(int i=0;i<Dim;i++){
      first= dom[i].first()+ eguards.lower(i);
      last= dom[i].last()- eguards.upper(i);
      dom[i]= Interval<1>(first,last);
    }
  }
  // Acquire the Loc object that describes the block structure.
  Loc<Dim> blocks= layout.blocks();
  // Acquire the list of nodes.
  typedef typename GridLayout<Dim>::List_t List_t;
  List_t nodeList= layout.nodeListGlobal();
  int listSize= nodeList.size();


  // Serialize the information.
  // Serialize the domain.
  nBytes+= serialize(s,dom);
  //std::cout<<"domain serialized"<<std::endl;
  // Serialize the blocks description.
  nBytes+= serialize(s,blocks);
  //std::cout<<"blocks serialized"<<std::endl;
  // Serialize the guard layers if they exist.
  nBytes+= serialize(s,ig);
  //std::cout<<"ig serialized"<<std::endl;
  if(ig==1){
    for(int i=0;i<Dim;i++){
      nBytes+= serialize(s,iguards.lower(i));
      nBytes+= serialize(s,iguards.upper(i));
    }
  }
  nBytes+= serialize(s,eg);
  //std::cout<<"eg serialized"<<std::endl;
  if(eg==1){
    for(int i=0;i<Dim;i++){
      nBytes+= serialize(s,eguards.lower(i));
      nBytes+= serialize(s,eguards.upper(i));
    }
  }
  // Serialize the node list.
  nBytes+= serialize(s,listSize);
  //std::cout<<"listSize serialized"<<std::endl;
  typename List_t::iterator listIter= nodeList.begin();
  typedef typename GridLayout<Dim>::Value_t Node_t;
  Node_t* node;
  for(int i=0;i<listSize;i++){
    node= *listIter;
    nBytes+= serialize(s,*node);
    listIter++;
  }

  return nBytes;
}

template <class Stream, int Dim>
int deserialize(GridLayout<Dim>& layout, Stream& s);
template <class Stream, int Dim>
int deserialize(GridLayout<Dim>& layout, Stream& s){
  int nBytes=0;

  // Read all the necessary state information.
  // Read the domain.
  typename GridLayout<Dim>::Domain_t dom;
  nBytes+= deserialize(dom,s);
  //std::cout<<"domain deserialized"<<std::endl;
  // Read the blocks description.
  Loc<Dim> blocks;
  nBytes+= deserialize(blocks,s);
  //std::cout<<"blocks deserialized"<<std::endl;
  // Check for guard layers and read if they exist.
  int ig=0, eg=0;
  typename GridLayout<Dim>::GuardLayers_t iguards, eguards;
  bool hasIG= false, hasEG= false;
  nBytes+= deserialize(ig,s);
  //std::cout<<"deserialized ig="<<ig<<std::endl;
  if(ig==1){
    hasIG= true;
    for(int i=0;i<Dim;i++){
      nBytes+= deserialize(iguards.lower(i),s);
      nBytes+= deserialize(iguards.upper(i),s);
    }
  }
  nBytes+= deserialize(eg,s);
  //std::cout<<"deserialized eg="<<eg<<std::endl;
  if(eg==1){
    hasEG= true;
    for(int i=0;i<Dim;i++){
      nBytes+= deserialize(eguards.lower(i),s);
      nBytes+= deserialize(eguards.upper(i),s);
    }
  }
  // Read the node list.
  typedef typename GridLayout<Dim>::List_t List_t;
  List_t nodeList;
  int listSize;
  nBytes+= deserialize(listSize,s);
  //std::cout<<"deserialized listSize="<<listSize<<std::endl;
  typedef typename GridLayout<Dim>::Value_t Node_t;
  Node_t* node;
  for(int i=0;i<listSize;i++){
    node= new Node_t;
    nBytes+= deserialize(*node,s);
    nodeList.push_back(node);
  }

  // Use the special I/O initializer for GridLayout
  // to instantiate.
  layout.initialize(dom, nodeList, blocks, hasIG, hasEG,
		    iguards, eguards);

  // std::cout<<"layout initialized"<<std::endl;
  return nBytes;
}

template <int Dim>
int serialSizeof(const GridLayout<Dim>& layout);
template <int Dim>
int serialSizeof(const GridLayout<Dim>& layout){
  int nBytes=0;

  // Acquire necessary state objects.
  // Acquire the complete domain.
  const typename GridLayout<Dim>::Domain_t& dom= layout.domain();
  // Acquire guard layers.
  typename GridLayout<Dim>::GuardLayers_t guards0, iguards,
    eguards;
  iguards= layout.internalGuards();
  eguards= layout.externalGuards();
  // Acquire the Loc object that describes the block structure.
  Loc<Dim> blocks= layout.blocks();
  // Acquire the list of nodes.
  typedef typename GridLayout<Dim>::List_t List_t;
  List_t nodeList= layout.nodeListGlobal();
  int listSize= nodeList.size();

  // Calculate the sizes.
  // Size the domain.
  nBytes+= serialSizeof(dom);
  // Size the blocks description.
  nBytes+= serialSizeof(blocks);
  // Size the guard layers info.
  nBytes+= sizeof(int);
  if(iguards!=guards0){
    nBytes+= 2*Dim*sizeof(int);
  }
  nBytes+= sizeof(int);
  if(eguards!=guards0){
    nBytes+= 2*Dim*sizeof(int);
  }
  // Size the node list.
  nBytes+= sizeof(int);
  typename List_t::iterator listIter= nodeList.begin();
  typename GridLayout<Dim>::Value_t* node;
  for(int i=0;i<listSize;i++){
    node= *listIter;
    nBytes+= serialSizeof(*node);
  }

  return nBytes;
}

//-----------------------------------------------------------
// I/O Functions for SparseTileLayout<Dim>
//-----------------------------------------------------------
// CTTI function for SparseTileLayout<Dim>
//-----------------------------------------------------------

template <int Dim>
std::string PoomaCTTI(const SparseTileLayout<Dim>&);
template <int Dim>
std::string PoomaCTTI(const SparseTileLayout<Dim>&){
 char numbuf[10];
  sprintf(numbuf,"%d,\0",Dim);
  std::string strng= "SparseTileLayout<"+
    std::string(numbuf)+ ">";
  return strng;
}

//-----------------------------------------------------------
// Serializers for SparseTileLayout<Dim>
//-----------------------------------------------------------

template <class Stream, int Dim>
int serialize(Stream& s, const SparseTileLayout<Dim>& layout);
template <class Stream, int Dim>
int serialize(Stream& s, const SparseTileLayout<Dim>& layout){
  int nBytes=0;

  // Acquire all the necessary state information.
  // Acquire the complete domain.
  typename SparseTileLayout<Dim>::Domain_t dom= layout.domain();
  // Check for guard layers and acquire.
  int ig=0, eg=0;
  typename SparseTileLayout<Dim>::GuardLayers_t guards0, iguards,
    eguards;
  iguards= layout.internalGuards();
  if(iguards!=guards0){
    ig=1;
  }
  eguards= layout.externalGuards();
  int first, last;
  if(eguards!=guards0){
    eg=1;
    // Subtract the external guards
    // to obtain the ab initio domain
    for(int i=0;i<Dim;i++){
      first= dom[i].first()+ eguards.lower(i);
      last= dom[i].last()- eguards.upper(i);
      dom[i]= Interval<1>(first,last);
    }
  }
  // Acquire the list of nodes.
  typedef typename SparseTileLayout<Dim>::List_t List_t;
  List_t nodeList= layout.nodeListGlobal();
  int listSize= nodeList.size();


  // Serialize the information.

  // Serialize the domain.
  nBytes+= serialize(s,dom);
  //std::cout<<"domain serialized"<<std::endl;

  // Serialize the guard layers if they exist.
  nBytes+= serialize(s,ig);
  //std::cout<<"ig serialized"<<std::endl;
  if(ig==1){
    for(int i=0;i<Dim;i++){
      nBytes+= serialize(s,iguards.lower(i));
      nBytes+= serialize(s,iguards.upper(i));
    }
  }
  nBytes+= serialize(s,eg);
  //std::cout<<"eg serialized"<<std::endl;
  if(eg==1){
    for(int i=0;i<Dim;i++){
      nBytes+= serialize(s,eguards.lower(i));
      nBytes+= serialize(s,eguards.upper(i));
    }
  }
  // Serialize the node list.
  nBytes+= serialize(s,listSize);
  //std::cout<<"listSize serialized"<<std::endl;
  typename List_t::iterator listIter= nodeList.begin();
  typedef typename SparseTileLayout<Dim>::Value_t Node_t;
  Node_t* node;
  for(int i=0;i<listSize;i++){
    node= *listIter;
    nBytes+= serialize(s,*node);
    listIter++;
  }

  return nBytes;
}

template <class Stream, int Dim>
int deserialize(SparseTileLayout<Dim>& layout, Stream& s);
template <class Stream, int Dim>
int deserialize(SparseTileLayout<Dim>& layout, Stream& s){
  int nBytes=0;

  // Read all the necessary state information.
  // Read the domain.
  typename SparseTileLayout<Dim>::Domain_t dom;
  nBytes+= deserialize(dom,s);
  //std::cout<<"domain deserialized"<<std::endl;

  // Check for guard layers and read if they exist.
  int ig=0, eg=0;
  typename SparseTileLayout<Dim>::GuardLayers_t iguards, eguards;
  bool hasIG= false, hasEG= false;
  nBytes+= deserialize(ig,s);
  //std::cout<<"deserialized ig="<<ig<<std::endl;
  if(ig==1){
    hasIG= true;
    for(int i=0;i<Dim;i++){
      nBytes+= deserialize(iguards.lower(i),s);
      nBytes+= deserialize(iguards.upper(i),s);
    }
  }
  nBytes+= deserialize(eg,s);
  //std::cout<<"deserialized eg="<<eg<<std::endl;
  if(eg==1){
    hasEG= true;
    for(int i=0;i<Dim;i++){
      nBytes+= deserialize(eguards.lower(i),s);
      nBytes+= deserialize(eguards.upper(i),s);
    }
  }
  // Read the node list.
  typedef typename SparseTileLayout<Dim>::List_t List_t;
  List_t nodeList;
  int listSize;
  nBytes+= deserialize(listSize,s);
  //std::cout<<"deserialized listSize="<<listSize<<std::endl;
  typedef typename SparseTileLayout<Dim>::Value_t Node_t;
  Node_t* node;
  for(int i=0;i<listSize;i++){
    node= new Node_t;
    nBytes+= deserialize(*node,s);
    nodeList.push_back(node);
  }


  // Use the appropriate SparseTileLayout constructor
  // to instantiate.
  layout= SparseTileLayout<Dim>(dom, iguards, eguards,
				nodeList);

  //std::cout<<"layout initialized"<<std::endl;
  return nBytes;
}

template <int Dim>
int serialSizeof(const SparseTileLayout<Dim>& layout);
template <int Dim>
int serialSizeof(const SparseTileLayout<Dim>& layout){
  int nBytes=0;

  // Acquire necessary state objects.
  // Acquire the complete domain.
  const typename SparseTileLayout<Dim>::Domain_t& dom= layout.domain();
  // Acquire guard layers.
  typename SparseTileLayout<Dim>::GuardLayers_t guards0, iguards,
    eguards;
  iguards= layout.internalGuards();
  eguards= layout.externalGuards();

  // Acquire the list of nodes.
  typedef typename SparseTileLayout<Dim>::List_t List_t;
  List_t nodeList= layout.nodeListGlobal();
  int listSize= nodeList.size();

  // Calculate the sizes.
  // Size the domain.
  nBytes+= serialSizeof(dom);
  // Size the guard layers info.
  nBytes+= sizeof(int);
  if(iguards!=guards0){
    nBytes+= 2*Dim*sizeof(int);
  }
  nBytes+= sizeof(int);
  if(eguards!=guards0){
    nBytes+= 2*Dim*sizeof(int);
  }
  // Size the node list.
  nBytes+= sizeof(int);
  typename List_t::iterator listIter= nodeList.begin();
  typename SparseTileLayout<Dim>::Value_t* node;
  for(int i=0;i<listSize;i++){
    node= *listIter;
    nBytes+= serialSizeof(*node);
  }

  return nBytes;
}


//-----------------------------------------------------------
// I/O Functions for DynamicLayout
//-----------------------------------------------------------
// CTTI function for DynamicLayout
//-----------------------------------------------------------

inline std::string PoomaCTTI(const DynamicLayout&);
inline std::string PoomaCTTI(const DynamicLayout&){
  return std::string("DynamicLayout");
}

//-----------------------------------------------------------
// Serializers for DynamicLayout
//-----------------------------------------------------------

template <class Stream>
int serialize(Stream& s, const DynamicLayout& layout);
template <class Stream>
int serialize(Stream& s, const DynamicLayout& layout){
  int nBytes=0;

  // Acquire all the necessary state information.
  // Acquire the complete domain.
  DynamicLayout::Domain_t dom= layout.domain();

  // Acquire the list of nodes.
  typedef DynamicLayout::List_t List_t;
  List_t nodeList= layout.nodeListGlobal();
  int listSize= nodeList.size();

  // Serialize the information.

  // Serialize the domain.
  nBytes+= serialize(s,dom);
  //std::cout<<"domain serialized"<<std::endl;

  // Serialize the node list.
  nBytes+= serialize(s,listSize);
  //std::cout<<"listSize serialized"<<std::endl;
  List_t::iterator listIter= nodeList.begin();
  typedef DynamicLayout::Value_t Node_t;
  Node_t* node;
  for(int i=0;i<listSize;i++){
    node= *listIter;
    nBytes+= serialize(s,*node);
    listIter++;
  }

  return nBytes;
}

template <class Stream>
int deserialize(DynamicLayout& layout, Stream& s);
template <class Stream>
int deserialize(DynamicLayout& layout, Stream& s){
  int nBytes=0;

  // Read all the necessary state information.
  // Read the domain.
  DynamicLayout::Domain_t dom;
  nBytes+= deserialize(dom,s);
  //std::cout<<"domain deserialized"<<std::endl;

  // Read the node list.
  typedef DynamicLayout::List_t List_t;
  List_t nodeList;
  int listSize;
  nBytes+= deserialize(listSize,s);
  //std::cout<<"deserialized listSize="<<listSize<<std::endl;
  typedef DynamicLayout::Value_t Node_t;
  Node_t* node;
  for(int i=0;i<listSize;i++){
    node= new Node_t;
    nBytes+= deserialize(*node,s);
    nodeList.push_back(node);
  }


  // Use the I/O initializer to instantiate.
  layout.initialize(dom, nodeList);

  //std::cout<<"layout initialized"<<std::endl;
  return nBytes;
}

int serialSizeof(const DynamicLayout& layout);

#endif // POOMA_IO_LAYOUT_IO_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: LayoutIO.h,v $   $Author: swhaney $
// $Revision: 1.13 $   $Date: 2000/07/20 20:47:37 $
// ----------------------------------------------------------------------
// ACL:rcsinfo





