//-----------------------------------------------------------------------------
// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------
#include <string>
#include <iostream>
#include <fstream>
#include "Pooma/Pooma.h"
#include "Utilities/Tester.h"
#include "Engine/BrickEngine.h"
#include "Engine/CompressibleBrick.h"
#include "IO/EngineIO.h"

//-----------------------------------------------------------------------------
// Test of Pooma Layout serializers
//-----------------------------------------------------------------------------
//-----------------------------------------------------------------------------
void itemCheck(Pooma::Tester& tstr, bool cond, const std::string& m);
void itemCheck(Pooma::Tester& tstr, bool cond, const std::string& m){
  tstr.check(cond);
  if(cond){
    tstr.out() << "PASSED "<<m<<std::endl;
  }
  else{
    tstr.out() << "FAILED "<<m<<std::endl;
  }
}

template <class T, class EngineTag>
bool equalEngines(Engine<1,T,EngineTag>& e1,
		  Engine<1,T,EngineTag>& e2);
template <class T, class EngineTag>
bool equalEngines(Engine<1,T,EngineTag>& e1,
		  Engine<1,T,EngineTag>& e2){
  typename Engine<1,T,EngineTag>::Domain_t dom1, dom2;
  dom1= e1.domain();
  dom2= e2.domain();
  if(!(dom1==dom2))
    return false;
  for(int i=dom1[0].first();
      i<dom1[0].last(); i++){
    if(e1(i)!=e2(i))
      return false;
  }
  return true;
}

template <class T, class EngineTag>
bool equalEngines(Engine<2,T,EngineTag>& e1,
		  Engine<2,T,EngineTag>& e2);
template <class T, class EngineTag>
bool equalEngines(Engine<2,T,EngineTag>& e1,
		  Engine<2,T,EngineTag>& e2){
  typename Engine<2,T,EngineTag>::Domain_t dom1, dom2;
  dom1= e1.domain();
  dom2= e2.domain();
  if(!(dom1==dom2))
    return false;
  for(int j=dom1[1].first();
      j<dom1[1].last(); j++){
    for(int i=dom1[0].first();
	i<dom1[0].last(); i++){
      if(e1(i,j)!=e2(i,j))
	return false;
    }
  }
  return true;
}

template <class T, class EngineTag>
bool equalEngines(Engine<3,T,EngineTag>& e1,
		  Engine<3,T,EngineTag>& e2);
template <class T, class EngineTag>
bool equalEngines(Engine<3,T,EngineTag>& e1,
		  Engine<3,T,EngineTag>& e2){
  typename Engine<3,T,EngineTag>::Domain_t dom1, dom2;
  dom1= e1.domain();
  dom2= e2.domain();
  if(!(dom1==dom2))
    return false;
  for(int k=dom1[2].first();
      k<dom1[2].last(); k++){
    for(int j=dom1[1].first();
	j<dom1[1].last(); j++){
      for(int i=dom1[0].first();
	  i<dom1[0].last(); i++){
	if(e1(i,j,k)!=e2(i,j,k))
	  return false;
      }
    }
  }
  return true;
}



int main(int argc, char *argv[])
{
  Pooma::initialize(argc, argv);

  Pooma::Tester tester(argc, argv);
  bool checkResult;

  tester.out() <<"Test Brick and CompressibleBrick Engine serializers:"<< std::endl;

  // Open a file for the test.
  std::fstream datafile("BrickEngineSerializerTest.dat",
                        std::ios::out | std::ios::binary);

  int ksize;
  // Create an Interval domain.
  Interval<2> dom;
  dom[0]= Interval<1>(0,7);
  dom[1]= Interval<1>(0,3);

  // Create a Brick engine of doubles with the given
  // domain
  Engine<2,double,Brick> brickEngine(dom);
  // Assign values
  int i,j;
  for(j=0;j<4;j++){
    for(i=0;i<8;i++){
      brickEngine(i,j)= ((double)j)+
	((double)i)*0.01;
    }
  }

  // Serialize
  ksize=serialize(datafile,brickEngine);
  itemCheck(tester,(ksize==serialSizeof(brickEngine)),
	    "Size computation on Brick engine");

  // Create a compressible brick engine using the same
  // domain.
  Engine<2,int,CompressibleBrick> cbrickEngine(dom);

  // Assign constant value
  cbrickEngine(0,0)= 5;

  // Serialize
  ksize=serialize(datafile,cbrickEngine);
  itemCheck(tester,(ksize==serialSizeof(cbrickEngine)),
	    "Size computation on CompressibleBrick engine");

  // Close the file.
  datafile.close();

  // Reopen as another file object.
  std::fstream dfile("BrickEngineSerializerTest.dat",
                     std::ios::in | std::ios::binary);

  // Create a Brick engine target
  Engine<2,double,Brick> brickEngine2;

  // Deserialize
  deserialize(brickEngine2, dfile);
  checkResult= equalEngines(brickEngine,brickEngine2);
  itemCheck(tester,checkResult,
	    "Check on stored Brick engine values");

  // Print out the results.
  tester.out()<<"Brick engine object retrieved"<<std::endl;
  std::string typeString= PoomaCTTI(brickEngine);
  tester.out()<<"Type identifier: "<<typeString<<std::endl;

  // Create a CompressibleBrick engine target
  Engine<2,int,CompressibleBrick> cbrickEngine2;

  // Deserialize
  deserialize(cbrickEngine2, dfile);
  checkResult= equalEngines(cbrickEngine,cbrickEngine2);
  itemCheck(tester,checkResult,
	    "Check on stored CompressibleBrick engine values");

  // Print out the results.
  tester.out()<<"CompressibleBrick engine object retrieved"<<std::endl;
  typeString= PoomaCTTI(cbrickEngine);
  tester.out()<<"Type identifier: "<<typeString<<std::endl;


  dfile.close();

  int retval = 
    tester.results("Test of Brick and CompressibleBrick Engine serializers");
  Pooma::finalize();
  return retval;
}











