// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license

//-----------------------------------------------------------------------------
// Classes: 
//   FieldEngineBase specialized for Expression-engines
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_FIELDENGINE_FIELDENGINEBASE_EXPRENGINE__H
#define POOMA_NEWFIELD_FIELDENGINE_FIELDENGINEBASE_EXPRENGINE__H

//-----------------------------------------------------------------------------
// Overview: 
// 
// FieldEngineBase and related classes. POOMA supports a flexifble form 
// of "centering" that allows a hierarchy of multiple centering points per 
// cell. The centering information, managemed by the FieldEngineBase
// class, is initialized using a flexible set of functors.
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
// Includes:
//-----------------------------------------------------------------------------

#include "Domain/Loc.h"
#include "Engine/ExpressionEngine.h"
#include "Layout/GuardLayers.h"
#include "Utilities/PAssert.h"


//-----------------------------------------------------------------------------
// Forward declarations:
//-----------------------------------------------------------------------------

template<int Dim, class T, class EngineTag> class FieldEngineBase;
template<class GeometryTag, class T, class EngineTag> class Field;


//-----------------------------------------------------------------------------
// Handy combiners for getting the far-left field in an expression.
//-----------------------------------------------------------------------------

struct FarLeftTag
{
  POOMA_PURIFY_CONSTRUCTORS(FarLeftTag)
};

template<class G1, class T1, class E1, class Op>
struct Combine1<Field<G1, T1, E1>, Op, FarLeftTag>
{
  typedef Field<G1, T1, E1> Type_t;
  inline static
  const Type_t &combine(const Field<G1, T1, E1> &a, const Op &,
			const FarLeftTag &) 
    { 
      return a; 
    }
};

template<class G1, class T1, class E1, class G2, class T2, class E2, 
  class Op>
struct Combine2<Field<G1, T1, E1>, Field<G2, T2, E2>, Op, FarLeftTag>
{
  typedef Field<G1, T1, E1> Type_t;
  inline static
  const Type_t &combine(const Field<G1, T1, E1> &a, 
                        const Field<G2, T2, E2> &, Op, FarLeftTag)
    {
      return a;
    }
};

template<class T, class G2, class T2, class E2, class Op>
struct Combine2<T, Field<G2, T2, E2>, Op, FarLeftTag>
{
  typedef Field<G2, T2, E2> Type_t;
  inline static
  const Type_t &combine(const T &, 
                        const Field<G2, T2, E2> &b, Op, FarLeftTag)
    {
      return b;
    }
};

template<class G1, class T1, class E1, class T, class Op>
struct Combine2<Field<G1, T1, E1>, T, Op, FarLeftTag>
{
  typedef Field<G1, T1, E1> Type_t;
  inline static
  const Type_t &combine(const Field<G1, T1, E1> &a, 
                        const T &, Op, FarLeftTag)
    {
      return a;
    }
};


//-----------------------------------------------------------------------------
// This version of LeafFunctor is used by Expression-Engines to 
// get at the far-left field in an expression. 
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag>
struct LeafFunctor<Field<GeometryTag, T, EngineTag>, FarLeftTag>
{
  typedef Field<GeometryTag, T, EngineTag> Type_t;
  inline static
  const Type_t &apply(const Field<GeometryTag, T, EngineTag> &f, 
    const FarLeftTag &) 
    {
      return f;
    }
};

template<class T>
struct LeafFunctor<Scalar<T>, FarLeftTag>
{
  typedef Scalar<T> Type_t;
  inline static
  const Type_t &apply(const Scalar<T> &s, const FarLeftTag &) 
    {
      return s;
    }
};


// ----------------------------------------------------------------------------
// FieldEngineBase<Dim, T, ExpressionTag<Expr> >
//
// This is a specialization of FieldEngineBase for expression-engines.
// ----------------------------------------------------------------------------

template<int Dim, class T, class Expr>
class FieldEngineBase<Dim, T, ExpressionTag<Expr> >
{
public:

  //---------------------------------------------------------------------------
  // Exported typedefs and enumerations.
    
  typedef ExpressionTag<Expr> EngineTag_t;
  typedef FieldEngineBase<Dim, T, EngineTag_t> This_t;
  typedef Engine<Dim, T, EngineTag_t> Engine_t;
  typedef typename Engine_t::Domain_t Domain_t;
  typedef typename Engine_t::Layout_t Layout_t;
  typedef typename Engine_t::Element_t Element_t;
  typedef typename Engine_t::ElementRef_t ElementRef_t;
  enum { dimensions = Dim };
  typedef GuardLayers<Dim> GuardLayers_t;
  typedef typename ForEach<Expr, FarLeftTag, FarLeftTag>::Type_t 
    ReferenceField_t;


  //---------------------------------------------------------------------------
  // Constructors.

  // Expression constructor.  
  
  FieldEngineBase(const Engine_t &e)
  : engine_m(e),
    referenceField_m(
      forEachRef(engine_m.expression(), FarLeftTag(), FarLeftTag()))
    { }  
    
  // Domain view constructor. 

  template<class Expr2>  
  FieldEngineBase(const FieldEngineBase<Dim, T, ExpressionTag<Expr2> > &model, 
    const Domain_t &d)
  : engine_m(model.engine(), d),
    referenceField_m(
      forEachRef(engine_m.expression(), FarLeftTag(), FarLeftTag()))
    { }  
    
  // Sub-field view constructor. This is when we want to construct a view of
  // one of the subFields in our top-level list.

  template<class Expr2>  
  FieldEngineBase(const FieldEngineBase<Dim, T, ExpressionTag<Expr2> > &model, 
    const int &iSubField)
  : engine_m(model.engine(), iSubField),
    referenceField_m(
      forEachRef(engine_m.expression(), FarLeftTag(), FarLeftTag()))
    { }  
      

  //---------------------------------------------------------------------------
  // Accessors and modifiers.
    
  inline int numSubFields() const
    {
      return referenceField().numSubFields();
    }

  inline const Loc<Dim> &offsets() const
    {
      return referenceField().offsets();
    }
    
  inline const Engine_t &engine() const
    {
      return engine_m;
    }

  const ReferenceField_t &referenceField() const
  {
    return referenceField_m;
  }
  

  //---------------------------------------------------------------------------
  // Domain accessor functions. 
        
  inline const Domain_t &physicalCellDomain() const
    {
      return referenceField_m.physicalCellDomain();
    }
        
  inline Domain_t totalCellDomain() const
    {
      return referenceField_m.totalCellDomain();
    }

  Domain_t physicalDomain() const
    {
      return referenceField_m.physicalDomain();
    }

  Domain_t totalDomain() const
    {
      return referenceField_m.totalDomain();
    }

  Domain_t physicalDomain(int iSubField) const
    {
      return referenceField_m.physicalDomain(iSubField);
    }

  Domain_t totalDomain(int iSubField) const
    {
      return referenceField_m.totalDomain(iSubField);
    }

      
private:

  Engine_t engine_m;
  const ReferenceField_t &referenceField_m;
};

#endif // POOMA_NEWFIELD_FIELDENGINE_FIELDENGINEBASE_EXPRENGINE__H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldEngineBase.ExprEngine.h,v $   $Author: sa_smith $
// $Revision: 1.5 $   $Date: 2000/08/08 17:31:50 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
