// -*- C++ -*-
// ACL:license
// ----------------------------------------------------------------------
// This software and ancillary information (herein called "SOFTWARE")
// called POOMA (Parallel Object-Oriented Methods and Applications) is
// made available under the terms described here.  The SOFTWARE has been
// approved for release with associated LA-CC Number LA-CC-98-65.
// 
// Unless otherwise indicated, this SOFTWARE has been authored by an
// employee or employees of the University of California, operator of the
// Los Alamos National Laboratory under Contract No. W-7405-ENG-36 with
// the U.S. Department of Energy.  The U.S. Government has rights to use,
// reproduce, and distribute this SOFTWARE. The public may copy, distribute,
// prepare derivative works and publicly display this SOFTWARE without 
// charge, provided that this Notice and any statement of authorship are 
// reproduced on all copies.  Neither the Government nor the University 
// makes any warranty, express or implied, or assumes any liability or 
// responsibility for the use of this SOFTWARE.
// 
// If SOFTWARE is modified to produce derivative works, such modified
// SOFTWARE should be clearly marked, so as not to confuse it with the
// version available from LANL.
// 
// For more information about POOMA, send e-mail to pooma@acl.lanl.gov,
// or visit the POOMA web page at http://www.acl.lanl.gov/pooma/.
// ----------------------------------------------------------------------
// ACL:license
//-----------------------------------------------------------------------------
// Functions: 
//   sum             - sum all the elements in a subfield.
//   prod            - multiply all of the elements in a subfield.
//   max             - find the maximum value in a subfield.
//   min             - find the minimum value in a subfield.
//   all             - returns true if all of the subfield's elements are != 0.
//   any             - returns true if any of the subfield's elements are != 0.
//   bitOr           - does a bitwise or of all of the elements.
//   bitAnd          - does a bitwise and of all of the elements.
//-----------------------------------------------------------------------------

#ifndef POOMA_NEWFIELD_FIELDREDUCTIONS_H
#define POOMA_NEWFIELD_FIELDREDUCTIONS_H

//-----------------------------------------------------------------------------
// Includes
//-----------------------------------------------------------------------------

#include "Evaluator/Reduction.h"


//-----------------------------------------------------------------------------
// Forward definitions
//-----------------------------------------------------------------------------

template<class GeometryTag, class T, class EngineTag>
class Field;


//-----------------------------------------------------------------------------
// Specific global reduction functions for Fields.
//-----------------------------------------------------------------------------

// Sum up the elements of a Field.

template<class GeometryTag, class T, class EngineTag>
T sum(const Field<GeometryTag, T, EngineTag> &f)
{
  // These versions only work with fields with no sub-fields.

  PAssert(f.numSubFields() == 0);
  
  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, OpAddAssign(), f);
  return ret;
}

// Compute the product of the elements of a Field.

template<class GeometryTag, class T, class EngineTag>
T prod(const Field<GeometryTag, T, EngineTag> &f)
{
  // These versions only work with fields with no sub-fields.

  PAssert(f.numSubFields() == 0);

  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, OpMultiplyAssign(), f);
  return ret;
}

// Find the smallest element of a Field.

template<class GeometryTag, class T, class EngineTag>
T min(const Field<GeometryTag, T, EngineTag> &f)
{
  // These versions only work with fields with no sub-fields.

  PAssert(f.numSubFields() == 0);

  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, FnMinAssign(), f);
  return ret;
}

// Find the largest element of a Field.

template<class GeometryTag, class T, class EngineTag>
T max(const Field<GeometryTag, T, EngineTag> &f)
{
  // These versions only work with fields with no sub-fields.

  PAssert(f.numSubFields() == 0);

  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, FnMaxAssign(), f);
  return ret;
}

// Report if all of the elments of the Field are true.

template<class GeometryTag, class T, class EngineTag>
bool all(const Field<GeometryTag, T, EngineTag> &f)
{
  // These versions only work with fields with no sub-fields.

  PAssert(f.numSubFields() == 0);

  bool ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, FnAndAssign(), f);
  return ret;
}

// Report if some of the elments of the Field are true.

template<class GeometryTag, class T, class EngineTag>
bool any(const Field<GeometryTag, T, EngineTag> &f)
{
  // These versions only work with fields with no sub-fields.

  PAssert(f.numSubFields() == 0);

  bool ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, FnOrAssign(), f);
  return ret;
}

// Bitwise-or all of the elements together.

template<class GeometryTag, class T, class EngineTag>
T bitOr(const Field<GeometryTag, T, EngineTag> &f)
{
  // These versions only work with fields with no sub-fields.

  PAssert(f.numSubFields() == 0);

  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, OpBitwiseOrAssign(), f);
  return ret;
}

// Bitwise-and all of the elements together.

template<class GeometryTag, class T, class EngineTag>
T bitAnd(const Field<GeometryTag, T, EngineTag> &f)
{
  // These versions only work with fields with no sub-fields.

  PAssert(f.numSubFields() == 0);

  T ret;
  Reduction<MainEvaluatorTag>().evaluate(ret, OpBitwiseAndAssign(), f);
  return ret;
}

#endif // POOMA_NEWFIELD_FIELDREDUCTIONS_H

// ACL:rcsinfo
// ----------------------------------------------------------------------
// $RCSfile: FieldReductions.h,v $   $Author: swhaney $
// $Revision: 1.1 $   $Date: 2000/07/24 23:36:22 $
// ----------------------------------------------------------------------
// ACL:rcsinfo
